package org.unitils.selenium;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.support.PageFactory;
import org.unitils.core.Module;
import org.unitils.core.TestListener;
import org.unitils.selenium.annotation.BaseUrl;
import org.unitils.selenium.annotation.TestWebDriver;
import org.unitils.selenium.annotation.WebPage;
import org.unitils.util.AnnotationUtils;
import org.unitils.util.PropertyUtils;
import org.unitils.util.ReflectionUtils;

/**
 * 
 * @author Jeroen Horemans
 * @author Thomas De Rycke
 * @author Willemijn Wouters
 * 
 * @since 1.0.0
 * 
 */
public class WebDriverModule implements Module {

    protected static final String PACKAGENAME = "org.unitils.selenium";
    
    protected static final String BROWSER_NAME_KEY = PACKAGENAME + ".browser.name";

    protected static final String BASE_URL_KEY = PACKAGENAME + ".baseUrl";

    protected static final String PROXY_HOST_KEY = PACKAGENAME + ".browser.proxy";
    
    protected static final String DOWNLOADPATH = PACKAGENAME + ".downloadpath";

    protected static final String FILETYPE = PACKAGENAME + ".filetype";
    
    protected static final String LIST_AUTOMATICALLY_DOWNLOAD = "application/pdf, application/vnd.fdf, application/x-msdos-program, application/x-unknown-application-octet-stream, application/vnd.ms-powerpoint, application/excel, application/vnd.ms-publisher, application/x-unknown-message-rfc822, application/vnd.ms-excel, application/msword, application/x-mspublisher, application/x-tar, application/zip, application/x-gzip,application/x-stuffit,application/vnd.ms-works, application/powerpoint, application/rtf, application/postscript, application/x-gtar, video/quicktime, video/x-msvideo, video/mpeg, audio/x-wav, audio/x-midi, audio/x-aiff, application/octet-stream";
    
    private BrowserChoice browserChoice;

    private String baseUrl;

    private String proxyUrl;
    
    private String downloadPath;
    
    private String fileType;

    private static final Log LOGGER = LogFactory.getLog(WebDriverModule.class);

    @Override
    public void init(Properties configuration) {
        browserChoice = resolveBrowserChoice(configuration);
        baseUrl = resolveBaseUrl(configuration);
        proxyUrl = resolveProxyHost(configuration);
        LOGGER.debug("Driver Module loaded");
        downloadPath = PropertyUtils.getString(DOWNLOADPATH, "", configuration);
        fileType = PropertyUtils.getString(FILETYPE, LIST_AUTOMATICALLY_DOWNLOAD, configuration);
    }


    @Override
    public void afterInit() {
        // nothing for now.
    }
    /***
     * Get the base URL out of unitils.properties
     * 
     * @param configuration
     * @return {@link String}
     */
    private String resolveBaseUrl(Properties configuration) {
        String result = configuration.getProperty(BASE_URL_KEY);
        if (StringUtils.isEmpty(result)) {
            throw new IllegalArgumentException("plz fill in a value in the unitils.properties for " + BASE_URL_KEY);
        }
        return result;
    }

    /**
     * Gets the name of the browser out of unitils.properties
     * @param configuration
     * @return {@link BrowserChoice}
     */
    private BrowserChoice resolveBrowserChoice(Properties configuration) {
        String browserName = configuration.getProperty(BROWSER_NAME_KEY);
        if (StringUtils.isEmpty(browserName)) {
            LOGGER.info(BROWSER_NAME_KEY + " not set. Will choose browser FIREFOX");
            return BrowserChoice.FIREFOX;
        }
        return BrowserChoice.valueOf(browserName);

    }

    /**
     * Get the proxy host out of unitils.properties
     * @param configuration
     * @return {@link String}
     */
    private String resolveProxyHost(Properties configuration) {
        String result = configuration.getProperty(PROXY_HOST_KEY);
        if (StringUtils.isEmpty(result)) {
            LOGGER.info(PROXY_HOST_KEY + " not set. No proxy used");
            return "";
        }
        LOGGER.info("proxy: [" + result + "] set. Proxy used");
        return result;        
    }

    /**
     * Initialises the webdriver. The method searches if there are fields with the {@link TestWebDriver} 
     * and uses the browser choice (in unitils.properties) to create the driver.
     * @param testObject
     */
    public void initWebDriver(Object testObject) {
        Set<Field> fields = AnnotationUtils.getFieldsAnnotatedWith(testObject.getClass(), TestWebDriver.class);
        
        if (fields.size() > 1) {
            //warn if there are more than one fields with the @TestWebDriver
            StringBuilder builder = new StringBuilder();
            builder.append("There are more than one webdrivers.\n");
            for (Field field : fields) {
                builder.append("class: ");
                builder.append(field.getDeclaringClass().getName());
                builder.append(", field: ");
                builder.append(field.getName());
                builder.append("\n");
            }
            LOGGER.warn(builder.toString());
        }
        
        for (Field field : fields) {
            WebDriver driver;
            if (proxyUrl.isEmpty()) {
                driver = WebDriverFactory.createDriver(browserChoice, downloadPath, fileType);
            } else {
                driver = WebDriverFactory.createDriver(browserChoice, proxyUrl, downloadPath, fileType);
            }
            driver.manage().deleteAllCookies();
            ReflectionUtils.setFieldValue(testObject, field, driver);
        }
    }

    /**
     * All the webdrivers (all the fields with the {@link TestWebDriver} of the testObject will be killed.
     * 
     * @param testObject
     */
    protected void killWebDriver(Object testObject) {
        Set<Field> fields = AnnotationUtils.getFieldsAnnotatedWith(testObject.getClass(), TestWebDriver.class);
        for (Field field : fields) {
            WebDriver driver = ReflectionUtils.getFieldValue(testObject, field);
            LOGGER.debug("closing a driver that is on page : " + driver.getCurrentUrl());
            driver.quit();
            nastyDoubleCheck(driver);
            nastyDoubleCheck(driver);
        }
    }

    private void nastyDoubleCheck(WebDriver driver) {
        try {
            Thread.sleep(500);
            driver.getTitle();
            driver.close();
            driver.quit();
        } catch (WebDriverException e) {
            // continue
        } catch (InterruptedException e) {
            // continue
        }

    }

    /**
     * All the elements with the @BaseUrlString will be initialised with the base url value of the unitils.properties.
     * @param testObject
     */
    public void initBaseUrl(Object testObject) {
        Set<Field> fields = AnnotationUtils.getFieldsAnnotatedWith(testObject.getClass(), BaseUrl.class);
        for (Field field : fields) {
            ReflectionUtils.setFieldValue(testObject, field, baseUrl);
        }

    }
    /***
     * Searches all the fields with the {@link WebPage} annotation and sets the correct elements.
     * @param testObject
     */
    public void initElements(Object testObject)  {
        //find fields that has the @WebPage annotation
        Set<Field> fields = AnnotationUtils.getFieldsAnnotatedWith(testObject.getClass(), WebPage.class);
        //find the webdriver
        Set<Field> webdrivers = AnnotationUtils.getFieldsAnnotatedWith(testObject.getClass(), TestWebDriver.class);
        if (webdrivers.size() > 0) {
            //initialise the page and set the object in the correct field.
            WebDriver webdriver = ReflectionUtils.getFieldValue(testObject, webdrivers.iterator().next());
            for (Field field : fields) {
                if (webdriver != null) {
                    ReflectionUtils.setFieldValue(testObject, field, getElement(webdriver, field.getType()));
                }
            }
        
        } else {
            LOGGER.error("The TestWebDriver cannot be found.");

        }
        
    }

    /**
     *  This is the actual method that creates an object of the correct type and initialises all the elements.
     * 
     * @param webdriver
     * @param type
     * @return {@link Object}
     * */
    protected Object getElement(WebDriver webdriver, Class<?> type) {
        return PageFactory.initElements(webdriver, type);
    }

    @Override
    public TestListener getTestListener() {
        return new TestListener() {

            @Override
            public void beforeTestSetUp(Object testObject, Method testMethod) {
                super.beforeTestSetUp(testObject, testMethod);
                initWebDriver(testObject);
                initBaseUrl(testObject);
                initElements(testObject);
            }

            /**
             * @see org.unitils.core.TestListener#afterTestTearDown(java.lang.Object, java.lang.reflect.Method)
             */
            @Override
            public void afterTestTearDown(Object testObject, Method testMethod) {
                killWebDriver(testObject);
                super.afterTestTearDown(testObject, testMethod); 
            }
            
        };
    }

}
