package org.unitils.selenium;

import java.io.File;
import java.util.Properties;
import java.util.logging.Level;

import org.apache.commons.lang.StringUtils;
import org.openqa.selenium.Proxy;
import org.openqa.selenium.chrome.ChromeOptions;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.openqa.selenium.firefox.FirefoxProfile;
import org.openqa.selenium.logging.LogType;
import org.openqa.selenium.logging.LoggingPreferences;
import org.openqa.selenium.remote.CapabilityType;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.unitils.core.Unitils;
import org.unitils.util.PropertyUtils;

/**
 * Creates the DesiredCapabilities for the WebDriver.
 * @author Willemijn Wouters
 * @since 5/05/15
 */
public class CapabilitiesFactory {

    /**
     * Creates the DesiredCapabilities for firefox.
     * @param proxyUrl
     * @param profile
     * @return DesiredCapabilities
     */
    protected static DesiredCapabilities createCapabilitesFirefox(String proxyUrl, FirefoxProfile profile) {
        DesiredCapabilities capabilities = DesiredCapabilities.firefox();
        if (!proxyUrl.isEmpty()) {
            capabilities.setCapability(FirefoxDriver.PROFILE, profile);
            profile.setPreference("network.proxy.type", 1);
            Proxy proxy = new Proxy();
            proxy.setHttpProxy(proxyUrl).setFtpProxy(proxyUrl).setSslProxy(proxyUrl).setNoProxy("");
            capabilities.setCapability(CapabilityType.PROXY, proxy);
        }
        capabilities.setCapability(CapabilityType.LOGGING_PREFS, createLoggingPreferences());

        return capabilities;
    }

    /**
     * Creates the DesiredCapabilities for an IE WebDriver.
     * @param proxyUrl
     * @return DesiredCapabilities
     */
    protected static DesiredCapabilities createCapabilitiesIE(String proxyUrl) {
        DesiredCapabilities cap = DesiredCapabilities.internetExplorer();
        cap.setCapability(CapabilityType.LOGGING_PREFS, createLoggingPreferences());
        cap.setCapability("ignoreProtectedModeSettings", true);
        cap.setCapability("ignoreZoomSetting", true);


        if (StringUtils.isNotEmpty(proxyUrl)) {
            Proxy proxy = new Proxy();
            proxy.setHttpProxy(proxyUrl).setFtpProxy(proxyUrl).setSslProxy(proxyUrl).setNoProxy("");

            cap.setCapability(CapabilityType.PROXY, proxy);
        }

        return cap;
    }

    /**
     * Creates the DesiredCapabilities for the Chrome Webdriver.
     * @return DesiredCapabilities
     */
    protected static DesiredCapabilities createCapabilitiesChrome() {
        DesiredCapabilities cap = DesiredCapabilities.chrome();

        cap.setCapability(CapabilityType.LOGGING_PREFS, createLoggingPreferences());
        String filePathBinary = PropertyUtils.getString(WebDriverModule.CHROME_BINARY_KEY, "", getUnitilsProperties());

        // This part checks if there is another path for the chrome binary than the default path.
        // This makes it possible to work with liberkey chrome etc.
        if (!StringUtils.isEmpty(filePathBinary)) {
            ChromeOptions options = new ChromeOptions();
            File tempFileLocation = new File(filePathBinary);
            if (tempFileLocation.exists()) {
                options.setBinary(tempFileLocation.getAbsolutePath());
                cap.setCapability(ChromeOptions.CAPABILITY, options);
            }
        }
        return cap;
    }

    /**
     * Creates a {@link org.openqa.selenium.logging.LoggingPreferences} object with the values of the unitils.properties.
     *
     * @return {@link org.openqa.selenium.logging.LoggingPreferences}
     */
    protected static LoggingPreferences createLoggingPreferences() {
        LoggingPreferences logPrefs = new LoggingPreferences();
        Properties unitilsProperties = getUnitilsProperties();
        logPrefs.enable(LogType.PERFORMANCE, Level.parse(PropertyUtils.getString(WebDriverModule.LOGGINGPROP_PERFORMANCE, "ALL", unitilsProperties)));
        logPrefs.enable(LogType.BROWSER, Level.parse(PropertyUtils.getString(WebDriverModule.LOGGINGPROP_BROWSER, "ALL", unitilsProperties)));
        logPrefs.enable(LogType.CLIENT, Level.parse(PropertyUtils.getString(WebDriverModule.LOGGINGPROP_CLIENT, "ALL", unitilsProperties)));
        logPrefs.enable(LogType.DRIVER, Level.parse(PropertyUtils.getString(WebDriverModule.LOGGINGPROP_DRIVER, "ALL", unitilsProperties)));
        logPrefs.enable(LogType.PROFILER, Level.parse(PropertyUtils.getString(WebDriverModule.LOGGINGPROP_PROFILER, "ALL", unitilsProperties)));
        logPrefs.enable(LogType.SERVER, Level.parse(PropertyUtils.getString(WebDriverModule.LOGGINGPROP_SERVER, "ALL", unitilsProperties)));

        return logPrefs;
    }

    private static Properties getUnitilsProperties() {
        return Unitils.getInstance().getConfiguration();
    }


}
