package org.unitils.selenium;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.firefox.FirefoxBinary;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.openqa.selenium.firefox.FirefoxProfile;
import org.openqa.selenium.ie.InternetExplorerDriver;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.unitils.core.Unitils;
import org.unitils.core.UnitilsException;
import org.unitils.util.PropertyUtils;
import org.unitils.util.ReflectionUtils;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Properties;

import static org.unitils.selenium.CapabilitiesFactory.*;
import static org.unitils.selenium.WebDriverModule.LOGGINGPROP_IE;


/**
 * This class creates all the {@link WebDriver}s.
 *
 * @author Jeroen Horemans
 * @author Thomas De Rycke
 * @author Willemijn Wouters
 * @since 1.0.0
 */
public class WebDriverFactory {

    private static final Log LOGGER = LogFactory.getLog(WebDriverFactory.class);

    private static final String BROWSERMESSAGE = "privacy.popups.showBrowserMessage";

    /**
     * The webdriver will be created in this method.
     *
     * @param browserChoice : the chosen browser (see: {@link WebDriverModule#BROWSER_NAME_KEY}).
     * @param downloadPath  : the location where all the files should be downloaded.
     * @param fileType      : all the types that can be downloaded automatically.
     * @return {@link WebDriver}
     */
    public static WebDriver createDriver(BrowserChoice browserChoice, String downloadPath, String fileType) {
        if (browserChoice == null || BrowserChoice.FIREFOX.equals(browserChoice)) {
            return createFireFoxDriver(downloadPath, fileType);
        } else if (BrowserChoice.CHROME.equals(browserChoice)) {
            return createChromeDriver();
        } else if (BrowserChoice.IE.equals(browserChoice)) {
            return new InternetExplorerDriver(createCapabilitiesIE(""));
        } else if(BrowserChoice.REMOTE.equals(browserChoice)) {
            return createRemoteWebdriver("", fileType, downloadPath);
        } else {
            return ReflectionUtils.createInstanceOfType(browserChoice.getDriverClass(), true);
        }

    }



    /**
     * This method creates a new {@link ChromeDriver}.
     *
     * @return {@link WebDriver}
     */
    protected static WebDriver createChromeDriver() {
        if (!StringUtils.isEmpty(PropertyUtils.getString(WebDriverModule.LOGGING_FILE_PROP, "", getUnitilsProperties()))) {
            System.setProperty("webdriver.chrome.logfile", PropertyUtils.getString(WebDriverModule.LOGGING_FILE_PROP, "", getUnitilsProperties()));
        }
        return new ChromeDriver(createCapabilitiesChrome());
    }

    /**
     * This method sets the log levels for IE.
     */
    public static void setLogPropertiesIE() {
        String logLevel = PropertyUtils.getString(LOGGINGPROP_IE, "FATAL", getUnitilsProperties());

        System.setProperty("webdriver.ie.driver.loglevel", logLevel);

        File fileFromProperty = getFileFromProperty(WebDriverModule.LOGGING_FILE_PROP);

        if (fileFromProperty != null) {
            System.setProperty("webdriver.ie.driver.logfile", fileFromProperty.getAbsolutePath());
        }
    }

    /**
     * This method creates a new {@link InternetExplorerDriver}.
     *
     * @param proxyUrl : the proxy url
     * @return {@link WebDriver}
     */
    protected static WebDriver createInternetExplorerDriver(String proxyUrl) {
        return new InternetExplorerDriver(createCapabilitiesIE(proxyUrl));
    }

    /**
     * checks which browser is chosen by the {@link BrowserChoice} and creates the {@link WebDriver}.
     *
     * @param browserChoice : the chosen browser (see: {@link WebDriverModule#BROWSER_NAME_KEY}).
     * @param proxyUrl      : the proxy url
     * @param downloadPath  : the location where all the files should be downloaded.
     * @param fileType      : all the types that can be downloaded automatically.
     * @return {@link WebDriver}
     */
    public static WebDriver createDriver(BrowserChoice browserChoice, String proxyUrl, String downloadPath, String fileType) {
        if (browserChoice == null || BrowserChoice.FIREFOX.equals(browserChoice)) {
            return createFireFoxDriver(proxyUrl, downloadPath, fileType);
        } else if (BrowserChoice.IE.equals(browserChoice)) {
            return createInternetExplorerDriver(proxyUrl);
        } else if(BrowserChoice.REMOTE.equals(browserChoice)) {
            return createRemoteWebdriver(proxyUrl, fileType, downloadPath);

        } else {
            return ReflectionUtils.createInstanceOfType(browserChoice.getDriverClass(), true);
        }
    }

    /**
     * Creates a {@link FirefoxDriver}.
     *
     * @param downloadPath : the location where all the files should be downloaded.
     * @param fileType     : all the types that can be downloaded automatically.
     * @return {@link WebDriver}
     */
    protected static WebDriver createFireFoxDriver(String downloadPath, String fileType) {
        return createFireFoxDriver("", downloadPath, fileType);
    }

    /**
     * This method creates a new {@link FirefoxDriver}.
     *
     * @param proxyUrl     : the proxy url
     * @param downloadPath : the location where all the files should be downloaded.
     * @param fileType     : all the types that can be downloaded automatically.
     * @return {@link WebDriver}
     */
    protected static WebDriver createFireFoxDriver(String proxyUrl, String downloadPath, String fileType) {

        FirefoxDriver firefoxDriver = null;
        FirefoxBinary binary = new FirefoxBinary();
        FirefoxProfile profile = createFirefoxProfile(fileType, downloadPath);
        try {
            firefoxDriver = new FirefoxDriver(binary, profile, createCapabilitesFirefox(proxyUrl, profile));
        } catch (WebDriverException e) {
            /*
             * This is a bug in Selenium. When an exception is thrown when selenium tries to create to create a driver, than their is
             * already a Firefox window open. This window needs to be closed.
             */

            binary.quit();
            // throw the exception again
            throw e;
        }

        // This sets the proxy optionally

        return firefoxDriver;
    }
    protected static FirefoxProfile createFirefoxProfile(String fileType, String downloadPath) {
        FirefoxProfile profile = new FirefoxProfile();

        // this should disable the proxy popup on some pc's
        profile.setPreference("network.proxy.type", 4);

        // This should fix popups concerning secure connections.
        profile.setAssumeUntrustedCertificateIssuer(true);
        profile.setAcceptUntrustedCertificates(true);


        profile.setPreference(BROWSERMESSAGE, false);

        profile.setPreference("dom.ipc.plugins.java.enabled", false);
        profile.setPreference("security.checkloaduri", false);

        profile.setPreference("browser.ssl_override_behavior", 0);
        profile.setPreference("security.ssl.allow_unrestricted_renego_everywhere__temporarily_available_pref;", true);
        profile.setPreference(BROWSERMESSAGE, false);
        profile.setPreference("profile.secure_ssl", true);
        profile.setPreference(BROWSERMESSAGE, false);
        profile.setPreference("security.csp.enable", false);
        profile.setPreference("security.OCSP.enabled", 0);
        profile.setPreference("browser.download.manager.quitBehavior", 1);

        profile.setPreference("browser.download.manager.alertOnEXEOpen", false);
        profile.setPreference("browser.download.manager.quitBehavior", 1);

        profile.setPreference("browser.download.manager.alertOnEXEOpen", false);
        profile.setPreference("security.csp.enable", false);
        profile.setPreference("security.OCSP.enabled", 0);
        profile.setPreference("browser.download.manager.quitBehavior", 1);

        profile.setPreference(BROWSERMESSAGE, false);
        profile.setPreference("browser.ssl_override_behavior", 0);

        profile.setPreference("security.enable_tls", false);

        profile.setPreference("security.checkloaduri", false);

        if (!StringUtils.isEmpty(fileType) && !StringUtils.isEmpty(downloadPath)) {
            // download a file automatically
            profile.setPreference("browser.download.folderList", 2);
            profile.setPreference("browser.download.manager.showWhenStarting", false);
            profile.setPreference("browser.download.dir", downloadPath);
            profile.setPreference("browser.helperApps.neverAsk.saveToDisk", fileType);
            profile.setPreference("browser.helperApps.neverAsk.openFile", fileType);
            profile.setPreference("browser.helperApps.alwaysAsk.force", false);
            profile.setPreference("plugin.disable_full_page_plugin_for_types", fileType);
            profile.setPreference("pdfjs.disabled", true);
        }

        File fileFromProperty = getFileFromProperty(WebDriverModule.LOGGING_FILE_PROP);

        if (fileFromProperty != null) {
            String absPath = fileFromProperty.getAbsolutePath();
            profile.setPreference("webdriver.log.file", absPath);
            profile.setPreference("webdriver.log.browser.file", absPath);
            profile.setPreference("webdriver.log.driver.file", absPath);
            profile.setPreference("webdriver.log.profiler.file", absPath);

        }
        return profile;

    }

    protected static WebDriver createRemoteWebdriver(String proxyUrl, String fileType, String downloadPath) {
        DesiredCapabilities capabilities = null;

        CapabilitiesChoice choice = getChosenCapabilities();
        switch (choice) {

            case CHROME:
                capabilities = createCapabilitiesChrome();
                break;
            case FIREFOX:
                capabilities = createCapabilitesFirefox(proxyUrl, createFirefoxProfile(fileType, downloadPath));
                break;
            case IE:
                capabilities = createCapabilitiesIE(proxyUrl);
                break;
        }


        URL remoteUrl;
        try {
            String urlProp = PropertyUtils.getString(WebDriverModule.REMOTE_URL_KEY, "", getUnitilsProperties());
            if (StringUtils.isBlank(urlProp)) {
                return new RemoteWebDriver(capabilities);
            }
            remoteUrl = new URL(urlProp);
        } catch (MalformedURLException e) {
            throw new UnitilsException("The following URL is not valid.", e);
        }
        return new RemoteWebDriver(remoteUrl, capabilities);
    }

    protected static CapabilitiesChoice getChosenCapabilities() {
        String chosenCapability = PropertyUtils.getString(WebDriverModule.REMOTE_CAPABILITIES, "", getUnitilsProperties());
        if (StringUtils.isBlank(chosenCapability)) {
            return CapabilitiesChoice.FIREFOX;
        }
        return CapabilitiesChoice.valueOf(chosenCapability);
    }

    /**
     * Checks if the value of the property is an already existing file, otherwise it tries to create a new file.
     *
     * @param property : the name of the property in the untils.property.
     * @return {@link File}
     */
    protected static File getFileFromProperty(String property) {
        String str = PropertyUtils.getString(property, "", getUnitilsProperties());

        if (StringUtils.isEmpty(str)) {
            return null;
        }

        File file = new File(str);
        if (!file.exists()) {
            LOGGER.debug("The file for property " + property + " is created at location: " + file.getAbsolutePath());
            try {
                file.createNewFile();
            } catch (IOException e) {
                LOGGER.error("Oops, something went wrong with creating " + file.getAbsolutePath() + ".");
                return null;
            }

        }
        return file;
    }

    /**
     * get the unitils.properties.
     *
     * @return {@link Properties} : all the properties defined in the unitils.properties, unitils-local.properties.
     */
    protected static Properties getUnitilsProperties() {
        return Unitils.getInstance().getConfiguration();
    }
}
