package org.unitils.spring.batch;

import static org.unitils.util.ReflectionUtils.setFieldValue;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.unitils.core.Module;
import org.unitils.core.TestListener;
import org.unitils.core.Unitils;
import org.unitils.core.UnitilsException;
import org.unitils.spring.SpringModule;
import org.unitils.spring.batch.annotations.BatchTestEnvironment;
import org.unitils.spring.batch.annotations.BatchTestPlaceHolder;
import org.unitils.util.AnnotationUtils;

/**
 * 
 * The batch module will create the spring batch context.
 * 
 * 
 * @author Jef Verelst
 * @author Jeroen Horemans
 * @author Thomas De Rycke
 * @author Willemijn Wouters
 * 
 * @since 1.0.0
 * 
 */
public class BatchModule implements Module {

    
    private static final String PARAM_OPTION = "param=";
    private static final Log LOGGER = LogFactory.getLog(BatchModule.class);

    @Override
    public void init(Properties configuration) {
        LOGGER.info("Batch module  : loaded");
    }

    protected void initializeBatchContext(Object testObject, Method testMethod) {

        fillupTestedObject(testObject, testMethod);

        fillupAnnotatedObjects(testObject);

    }

    protected void fillupAnnotatedObjects(Object testObject) {

        Set<Field> fields = AnnotationUtils.getFieldsAnnotatedWith(testObject.getClass(), BatchTestEnvironment.class);
        for (Field field : fields) {
            BatchTest value = createBatchJob(field, field.getAnnotation(BatchTestEnvironment.class), testObject);
            setFieldValue(testObject, field, value);
        }

    }

    protected void fillupTestedObject(Object testObject, Method testMethod) {
        BatchTestEnvironment annotation = testMethod.getAnnotation(BatchTestEnvironment.class);
        if (annotation == null) {
            return;
        }
        Field field = AnnotationUtils.getFieldsAnnotatedWith(testObject.getClass(), BatchTestPlaceHolder.class).iterator().next();
        
        if (!field.getType().isAssignableFrom(BatchTest.class)) {
            throw new UnitilsException("When annotating the method with @BatchTestPlaceHolder a tested object from the type @BatchTest is expected");
        }
        BatchTest value = createBatchJob(field, annotation, testObject);
        setFieldValue(testObject, field, value);

    }

    protected BatchTest createBatchJob(Field field, BatchTestEnvironment annotation, Object testObject) {
        BatchTest result = null;
        ApplicationContext contextFile = figureOutContext(annotation.contextFile(), testObject);
        String job = annotation.job();
        int exitCode = annotation.exitCode();
        BatchOption option = annotation.option();
        if ((!StringUtils.isEmpty(annotation.jobParameters()) && annotation.jobParameters().contains("=")) || annotation.parameter()) {
            
            String jobParameters = annotation.jobParameters();
            String parameter = PARAM_OPTION + "=" + System.currentTimeMillis();
            
            if(StringUtils.isNotEmpty(jobParameters)) {
                parameter = parameter + " " + jobParameters;
            }
            parameter += " " + new Date().toString();
            if (option.equals(BatchOption.NONE)) {
                result = new BatchTestImpl(contextFile, job, parameter, exitCode);
            } else {
                result = new BatchTestImpl(contextFile, job, parameter, option, exitCode);
            }
        } else if (option.equals(BatchOption.NONE)) {
            result = new BatchTestImpl(contextFile, job, exitCode);
        } else {
            result = new BatchTestImpl(contextFile, job, null, option, exitCode);
        }
        return result;
    }

    protected ApplicationContext figureOutContext(String contextFile, Object testObject) {
        try {
            ApplicationContext result = null;
            if (contextFile == null || contextFile.isEmpty()) {
                SpringModule mod = Unitils.getInstance().getModulesRepository().getModulesOfType(SpringModule.class).get(0);
                result = mod.getApplicationContext(testObject);
            } else {
                result = new ClassPathXmlApplicationContext(contextFile);
            }
            return result;
        } catch (Exception e) {
            throw new UnitilsException("Plz fill in the contextFile in the BatchTestEnvironment annotation, or use the @SpringContext from the SpringModule of unitils. ", e);
        }
    }

    @Override
    public void afterInit() {
        // do nothing
    }

    @Override
    public TestListener getTestListener() {
        return new BatchTestListener();
    }

    /**
     * 
     * BatchTestListener.
     * 
     * @since 
     *
     */
    private class BatchTestListener extends TestListener {

        @Override
        public void beforeTestSetUp(Object testObject, Method testMethod) {
            super.beforeTestSetUp(testObject, testMethod);
            initializeBatchContext(testObject, testMethod);

        }
        
    }
}
