package org.unitils.spring.batch;

import static org.junit.Assert.assertEquals;

import java.util.HashSet;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.springframework.context.ApplicationContext;
import org.unitils.core.UnitilsException;

/**
 * 
 * Implements the {@link BatchTest} by a custom soap ui implementation. This expects a project location (soap ui project xml file) , an
 * suite name and the name of the testCase. All should be exact.
 * 
 * @author Jef Verelst
 * @author Jeroen Horemans
 * @author Thomas De Rycke
 * @author Willemijn Wouters
 * 
 * @since 1.0.0
 * 
 */
public class BatchTestImpl implements BatchTest {

    private ApplicationContext context;

    private String jobName;

    private String parameter;

    private BatchOption option;

    private final int exitCode;

    /**
     * @param contextFile
     * @param jobName
     * @param exitCode
     * @throws UnitilsException
     */
    public BatchTestImpl(ApplicationContext contextFile, String jobName, int exitCode) throws UnitilsException {
        this.jobName = jobName;
        this.context = contextFile;
        this.exitCode = exitCode;
    }

    /**
     * Constructor to be able to test with a parameter, allowing for testing one job several time without clearing database over and over
     * again. A job might have to be able to run consecutively.
     * 
     * @param context place of the job xml Spring application context
     * @param jobName the name the job is known for in this context
     * @param parameter the additional parameter (added to the job as param=<i>parameter</i>)
     * @param exitCode the exit code to check
     * @throws UnitilsException when something unexpected happens
     */
    public BatchTestImpl(ApplicationContext context, String jobName, String parameter, int exitCode) throws UnitilsException {
        this.jobName = jobName;
        this.context = context;
        this.parameter = parameter;
        this.exitCode = exitCode;
    }

    /**
     * Constructor to be able to test with a parameter and an option (-restart, -stop, etc...), allowing for testing the restart of a job
     * for instance.
     * 
     * @param context 
     * @param jobName the name the job is known for in this context
     * @param parameter the additional parameter (added to the job as param=<i>parameter</i>)
     * @param exitCode the exit code to check
     * @param option the Batch option to add to the execution
     * @throws UnitilsException when something unexpected happens
     */
    public BatchTestImpl(ApplicationContext context, String jobName, String parameter, BatchOption option, int exitCode) throws UnitilsException {
        this.jobName = jobName;
        this.context = context;
        this.parameter = parameter;
        this.exitCode = exitCode;
        this.option = option;
    }

    @Override
    public void launchJob() throws Exception {
        UnitilsCommandLineJobRunner runner = new UnitilsCommandLineJobRunner(context);
        // TODO check the parameter thing.
        String[] parameters = initParameters(parameter);
        Set<String> opts = initOpts();
        int result = runner.start(jobName, parameters, opts);

        assertEquals("The batch exit code did not match", this.exitCode, result);
    }

    private String[] initParameters(String parametersToSplit) {
        if(parametersToSplit != null) {
            return StringUtils.split(parametersToSplit, ' ');
        } else {
            return new String[0];
        }
    }

    /**
     * @return
     */


    private Set<String> initOpts() {
        Set<String> set = new HashSet<String>();
        if (option != null && !option.equals(BatchOption.NONE)) {
            set.add(option.value());
        }
        return set;
    }
}
