package org.v2u.toy;

import org.apache.commons.dbutils.*;
import org.apache.commons.dbutils.handlers.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigInteger;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.AbstractMap.SimpleEntry;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public class Duck {
    public static final String IGNORE = "*IGNORE";
    public static final String FIELDS = "*FIELDS";
    private static final Pattern FORMAT_PH_PATTERN = Pattern.compile("([#@]?)\\{(\\d+)}");
    private static final Map<String, Optional<Info>> annotationCache = new ConcurrentHashMap<>();
    private static final Logger log = LoggerFactory.getLogger(Duck.class);
    protected final List<String> queryParts = new ArrayList<>();
    protected final List<Object> bindValues = new ArrayList<>();
    protected final Map<String, Integer> marks = new HashMap<>();
    protected Connection txConn = null;
    protected Function<String, String> quoter = makeQuoter("`");
    protected boolean isSnake = true;
    protected boolean isDebug = false;
    protected QueryRunner runner;

    public static Duck init(DataSource ds) {
        return new Duck(ds);
    }

    protected Duck init(Connection conn) {
        Duck duck = copy();
        duck.runner = new QueryRunner();
        duck.txConn = conn;
        return duck;
    }

    protected Duck(DataSource ds) {
        runner = new QueryRunner(ds);
    }

    public String getSql() {
        return String.join(" ", queryParts);
    }

    public Object[] getParams() {
        return bindValues.toArray();
    }

    public Duck reset() {
        this.queryParts.clear();
        this.bindValues.clear();
        this.marks.clear();
        return this;
    }

    public Connection getTxConn() {
        if (txConn == null) {
            throw new DuckException("Not in a transaction context");
        }
        return txConn;
    }

    protected Connection selectConn() throws SQLException {
        if (txConn != null) {
            return txConn;
        }

        if (runner.getDataSource() != null) {
            return runner.getDataSource().getConnection();
        }

        throw new DuckException("Unable to obtain database connection");
    }


    private <T> T execute(ConnectionCallback<T> action) {
        Connection conn = null;
        try {
            debugPrint();
            conn = selectConn();
            return action.execWithConn(conn);
        } catch (SQLException cause) {
            throw new DuckException(cause);
        } finally {
            if (conn != null && conn != txConn) {
                try {
                    conn.close();
                } catch (SQLException ex) {
                    log.warn("Failed to close connection: " + ex.getMessage());
                }
            }
        }
    }

    public <T> T fetch(final ResultSetHandler<T> rst) {
        return execute(conn -> runner.query(conn, getSql(), rst, getParams()));
    }

    public <T> List<T> fetchBeans(Class<T> tClass) {
        RowProcessor rowProcessor = new BasicRowProcessor(new DuckBeanProcessor());
        ResultSetHandler<List<T>> handler = new BeanListHandler<>(tClass, rowProcessor);
        return fetch(handler);
    }

    public <T> T fetchBean(Class<T> tClass) {
        List<T> result = fetchBeans(tClass);
        if (result.size() > 1) {
            String errmsg = String.format(
              "Non-unique result: query returned %d rows when expecting exactly one row", result.size());
            throw new DuckException(errmsg);
        }
        return result.isEmpty() ? null : result.get(0);
    }

    public <K, V> Map<K, V> fetchBeanMap(Class<V> vClass, Function<V, K> keyExtractor) {
        return fetchBeans(vClass)
          .stream()
          .collect(Collectors.toMap(
            keyExtractor,
            v -> v,
            (a, b) -> {
                throw new DuckException("duplicate key in map: " + keyExtractor.apply(a));
            }
          ));
    }

    public <K, V> Map<K, List<V>> fetchBeanGroup(Class<V> vClass, Function<V, K> keyExtractor) {
        return fetchBeans(vClass).stream().collect(Collectors.groupingBy(keyExtractor));
    }

    public List<Map<String, Object>> fetchMaps() {
        MapListHandler handler = new MapListHandler();
        return fetch(handler);
    }

    public Map<String, Object> fetchMap() {
        MapHandler handler = new MapHandler();
        return fetch(handler);
    }

    public <T> T fetchScalar(Class<T> tClass) {
        ScalarHandler<T> handler = new ScalarHandler<>();
        return fetch(handler);
    }

    public Duck select(String table, String where, Object... params) {
        return addRaw("select")
          .mark(FIELDS, "*")
          .addRaw("from " + this.quote(table))
          .addRaw("where " + where, params);
    }

    public Duck select(Object bean) {
        Info info = getTableConfig(bean);
        Map<String, Object> cond = beanToMap(bean, false, false);
        SimpleEntry<List<String>, List<Object>> pair = pair(cond);

        String where = pair.getKey().stream()
          .map(k -> String.format("%s = ?", k))
          .collect(Collectors.joining(" and "));

        return select(info.name(), where, pair.getValue().toArray());
    }

    public int update() {
        return execute(conn -> runner.update(conn, getSql(), getParams()));
    }

    public int updateMap(String tableName, Map<String, Object> params, String where, Object... whereParams) {
        SimpleEntry<String, Object[]> paramsKv = pairUpdate(params);
        String sql = String.format("update %s set %s", quote(tableName), paramsKv.getKey());
        return addRaw(sql, paramsKv.getValue())
          .addRaw("where " + where, whereParams)
          .update();
    }

    public int updateBean(Object bean, String where, Object... params) {
        Info info = getTableConfig(bean);
        return updateBean(info.name(), bean, where, params);
    }

    public int updateBean(String tableName, Object bean, String where, Object... whereParams) {
        Map<String, Object> params = beanToMap(bean, true, false);
        return updateMap(tableName, params, where, whereParams);
    }

    public int delete() {
        return update();
    }

    public int delete(String tableName, String where, Object... whereParams) {
        return addRaw("delete from " + quote(tableName))
          .addRaw("where " + where, whereParams)
          .delete();
    }

    public <T> T insert(Class<T> tClass) {
        return execute(conn -> runner.insert(conn, getSql(), new ScalarHandler<>(), getParams()));
    }

    public Long insert() {
        BigInteger pk = insert(BigInteger.class);
        return pk == null ? null : pk.longValue();
    }

    public Duck onDuplicateIgnore() {
        if (!marks.containsKey(IGNORE)) {
            throw new DuckException("no ignore mark!");
        }

        return mark(IGNORE, "ignore");
    }

    public Duck addInsertMap(String tableName, Map<String, Object> params) {
        SimpleEntry<String, List<Object>> kv = pairInsert(params);
        String sql = String.format("into %s (%s) values (?)", quote(tableName), kv.getKey());
        return addRaw("insert").mark(IGNORE, "").addRaw(sql, kv.getValue());
    }

    public Duck addInsertBean(Object bean) {
        Info info = getTableConfig(bean);
        return addInsertBean(info.name(), bean);
    }

    public Duck addInsertBean(String tableName, Object bean) {
        Map<String, Object> params = beanToMap(bean, false, true);
        return addInsertMap(tableName, params);
    }

    public List<Long> insertBeans(List<?> beans, boolean ignoreDuplicate) {
        Info info = getTableConfig(beans.get(0));
        return insertMaps(info.name(), processBatchInsertBeans(beans), ignoreDuplicate);
    }

    public <T> List<T> insertBeans(List<?> beans, boolean ignoreDuplicate, Class<T> tClass) {
        Info info = getTableConfig(beans.get(0));
        return insertMaps(info.name(), processBatchInsertBeans(beans), ignoreDuplicate, tClass);
    }

    public <T> List<T> insertBeans(String tableName, List<?> beans, boolean ignoreDuplicate, Class<T> tClass) {
        return insertMaps(tableName, processBatchInsertBeans(beans), ignoreDuplicate, tClass);
    }

    public List<Long> insertBeans(String tableName, List<?> beans, boolean ignoreDuplicate) {
        return insertMaps(tableName, processBatchInsertBeans(beans), ignoreDuplicate);
    }

    public List<Long> insertMaps(String tableName, List<Map<String, Object>> paramsList, boolean ignoreDuplicate) {
        List<BigInteger> ids = insertMaps(tableName, paramsList, ignoreDuplicate, BigInteger.class);
        List<Long> results = new ArrayList<>();
        for (BigInteger id : ids) {
            results.add(id == null ? null : id.longValue());
        }
        return results;
    }

    public <T> List<T> insertMaps(
      String tableName,
      List<Map<String, Object>> maps,
      boolean ignoreDuplicate,
      Class<T> tClass
    ) {
        if (maps == null || maps.isEmpty()) {
            throw new DuckException("Batch params empty");
        }

        // 获取第一个Map的所有键并固定顺序
        Map<String, Object> firstMap = maps.get(0);
        List<String> keys = new ArrayList<>(firstMap.keySet());

        // 构建SQL语句
        String ignore = ignoreDuplicate ? "ignore" : "";
        String fieldStr = keys.stream().map(this::quote).collect(Collectors.joining(", "));
        String placeholders = String.join(", ", Collections.nCopies(keys.size(), "?"));
        String sql = String.format(
          "insert %s into %s (%s) values (%s)", ignore, quote(tableName), fieldStr, placeholders);

        // 构建批量参数数组
        Object[][] batchArgs = new Object[maps.size()][];
        for (int i = 0; i < maps.size(); i++) {
            Map<String, Object> params = maps.get(i);
            Object[] values = new Object[keys.size()];
            for (int j = 0; j < keys.size(); j++) {
                values[j] = params.get(keys.get(j));
            }
            batchArgs[i] = values;
        }

        return execute(conn -> runner.insertBatch(conn, sql, new ColumnListHandler<>(), batchArgs));
    }

    public <R> R transaction(Function<Duck, R> action) {
        if (txConn != null) {
            throw new DuckException("Nested transactions are not allowed");
        }
        Connection conn = null;
        Boolean isAutoCommit = null;
        try {
            conn = runner.getDataSource().getConnection();
            isAutoCommit = conn.getAutoCommit();
            conn.setAutoCommit(false);
            Duck tx = init(conn);
            R result = action.apply(tx);
            conn.commit();
            return result;
        } catch (Throwable e) {
            try {
                if (conn != null) {
                    conn.rollback();
                }
            } catch (SQLException ex) {
                throw new DuckException("Failed to rollback transaction", ex);
            }
            throw new DuckException("Transaction failed", e);
        } finally {
            try {
                if (conn != null) {
                    if (isAutoCommit != null) {
                        conn.setAutoCommit(isAutoCommit);
                    }
                    conn.close();
                }
            } catch (SQLException e) {
                log.warn("Failed to close connection: " + e.getMessage());
            }
        }
    }

    private List<Map<String, Object>> processBatchInsertBeans(List<?> beans) {
        return beans.stream()
          .map(it -> beanToMap(it, false, true))
          .collect(Collectors.toList());
    }

    private Info getTableConfig(Object bean) {
        Info info = bean.getClass().getAnnotation(Info.class);
        if (info == null) {
            throw new DuckException("table config required");
        }
        return info;
    }

    protected Duck copy() {
        Duck query = new Duck(runner.getDataSource());
        query.txConn = txConn;
        query.queryParts.addAll(queryParts);
        query.bindValues.addAll(bindValues);
        query.marks.putAll(marks);
        query.quoter = quoter;
        query.isSnake = isSnake;
        query.isDebug = isDebug;

        return query;
    }

    public Duck setQuoter(Function<String, String> escaper) {
        Duck query = copy();
        query.quoter = escaper;
        return query;
    }

    public Function<String, String> makeQuoter(String quote) {
        return identifier -> {
            if (identifier == null) {
                throw new DuckException("Identifier cannot be null");
            }

            if (identifier.trim().isEmpty()) {
                throw new DuckException("Identifier cannot be empty");
            }

            return Arrays.stream(identifier.split("\\."))
              .map(part -> quote
                + part.replace(quote, quote + quote)
                + quote)
              .collect(Collectors.joining("."));
        };
    }

    public String quote(String ident) {
        return quoter.apply(ident);
    }

    public Duck add(String template, Object... args) {
        Matcher m = FORMAT_PH_PATTERN.matcher(template);
        StringBuffer sb = new StringBuffer();
        List<Object> sqlParams = new ArrayList<>();
        while (m.find()) {
            int idx = Integer.parseInt(m.group(2));
            if (idx < 0 || idx >= args.length) {
                throw new IllegalArgumentException("Index " + idx + " out of range");
            }
            String ident = m.group(1);
            String argStr = args[idx] == null ? "null" : args[idx].toString();
            switch (ident) {
                case "": //raw replace
                    m.appendReplacement(sb, Matcher.quoteReplacement(argStr));
                    break;
                case "@": //quote replace
                    m.appendReplacement(sb, Matcher.quoteReplacement(quote(argStr)));
                    break;
                case "#": //sql param
                    sqlParams.add(args[idx]);
                    m.appendReplacement(sb, "?");
                    break;
            }
        }
        m.appendTail(sb);
        return addRaw(sb.toString(), sqlParams.toArray());
    }

    public Duck addRaw(String sql, Object... params) {
        Duck query = copy();
        query.appendParams(sql, params);
        return query;
    }

    public Duck isSnake(boolean yes) {
        Duck duck = copy();
        duck.isSnake = yes;
        return duck;
    }

    public Duck debug(boolean debug) {
        Duck duck = copy();
        duck.isDebug = debug;
        return duck;
    }

    public Duck mark(String name, String sql) {
        Duck query = copy();
        if (query.marks.containsKey(name)) {
            query.queryParts.set(query.marks.get(name), sql);
        } else {
            query.queryParts.add(sql);
            query.marks.put(name, query.queryParts.size() - 1);
        }
        return query;
    }

    protected void appendParams(String sql, Object[] args) {
        String[] parts = (sql + " ").split("\\?");

        if (parts.length != args.length + 1) {
            String errmsg = String.format("Placeholders length (%d) doesn't match parameters length (%d)",
              parts.length - 1, args.length
            );
            throw new DuckException(errmsg);
        }

        for (int i = 0; i < parts.length; i++) {
            if (args.length <= i) {
                queryParts.add(parts[i]);
                return;
            }

            Object arg = args[i];

            if (arg == null) {
                queryParts.add(parts[i] + '?');
                bindValues.add(null);
            } else if (arg instanceof Collection) {
                appendArray(parts[i], ((Collection<?>) arg).toArray());
            } else if (arg.getClass().isArray()) {
                appendArray(parts[i], (Object[]) arg);
            } else {
                queryParts.add(parts[i] + '?');
                bindValues.add(arg);
            }
        }
    }

    protected void appendArray(String sql, Object[] arrayArg) {
        StringBuilder marks = new StringBuilder();
        for (int i = 0; i < arrayArg.length; i++) {
            if (i > 0) marks.append(',');
            marks.append('?');
        }
        queryParts.add(sql + marks);
        bindValues.addAll(Arrays.asList(arrayArg));
    }


    protected void debugPrint() {
        if (isDebug) {
            log.info("[Duck] SQL:\n" + this.toString());
        }
    }

    @Override
    public String toString() {
        return "-> SQL: " + getSql() + "\n-> Params: " + Arrays.toString(getParams());
    }

    public static String toSnake(String str) {
        if (str == null || str.isEmpty()) return str;
        String res = str.replaceAll("([A-Z]+)", "_$1").toLowerCase();
        if (res.startsWith("_") && !str.startsWith("_")) {
            res = res.substring(1);
        }
        return res;
    }

    public Map<String, Object> beanToMap(Object bean, boolean isUpdate, boolean isInsert) {
        try {
            Map<String, Object> result = new HashMap<>();
            Class<?> clazz = bean.getClass();
            PropertyDescriptor[] props = Introspector.getBeanInfo(clazz, Object.class).getPropertyDescriptors();
            for (PropertyDescriptor pd : props) {
                Method getter = pd.getReadMethod();  // getXxx()
                if (getter == null) {
                    continue;
                }

                boolean ignoreNull = true;
                String fieldName = pd.getName();
                String columnName = isSnake ? toSnake(fieldName) : fieldName;
                Optional<Info> optInfo = getCachedInfo(clazz, fieldName);
                if (optInfo.isPresent()) {
                    Info info = optInfo.get();
                    if (!info.exists()) continue;
                    if (isUpdate && !info.update()) continue;
                    if (isInsert && !info.insert()) continue;

                    ignoreNull = info.ignoreNull();

                    String annoColumnName = info.name();
                    if (annoColumnName != null && !annoColumnName.trim().isEmpty()) {
                        columnName = annoColumnName;
                    }
                }

                Object value = getter.invoke(bean);
                if (value != null || !ignoreNull) {
                    result.put(columnName, value);
                }
            }
            return result;
        } catch (Exception ex) {
            throw new DuckException(ex);
        }
    }

    public SimpleEntry<String, List<Object>> pairInsert(Map<String, Object> params) {
        SimpleEntry<List<String>, List<Object>> kv = pair(params);
        String sql = String.join(", ", kv.getKey());
        return new SimpleEntry<>(sql, kv.getValue());
    }

    public SimpleEntry<String, Object[]> pairUpdate(Map<String, Object> params) {
        SimpleEntry<List<String>, List<Object>> kv = pair(params);
        String sql = kv.getKey().stream()
          .map(k -> String.format("%s = ?", k))
          .collect(Collectors.joining(", "));

        return new SimpleEntry<>(sql, kv.getValue().toArray());
    }

    public SimpleEntry<List<String>, List<Object>> pair(Map<String, Object> params) {
        List<String> keys = new ArrayList<>(params.size());
        List<Object> values = new ArrayList<>(params.size());

        int index = 0;
        for (Map.Entry<String, Object> kv : params.entrySet()) {
            String key = quote(kv.getKey());
            keys.add(index, key);
            values.add(index, kv.getValue());
            index++;
        }

        return new SimpleEntry<>(keys, values);
    }

    protected static Optional<Info> getCachedInfo(Class<?> type, String fieldName) {
        String key = type.getName() + "#" + fieldName;
        return annotationCache.computeIfAbsent(key, k -> {
            try {
                Field field = type.getDeclaredField(fieldName);
                return Optional.ofNullable(field.getAnnotation(Info.class));
            } catch (NoSuchFieldException e) {
                return Optional.empty();
            }
        });
    }

    @FunctionalInterface
    protected interface ConnectionCallback<T> {
        T execWithConn(Connection conn) throws SQLException;
    }

    @Retention(RetentionPolicy.RUNTIME)
    @Target({ElementType.FIELD, ElementType.TYPE})
    public @interface Info {
        String name() default "";

        boolean exists() default true;

        boolean update() default true;

        boolean insert() default true;

        boolean ignoreNull() default true;
    }

    public static class DuckException extends RuntimeException {
        public DuckException(String message) {
            super(message);
        }

        public DuckException(Throwable cause) {
            super(cause);
        }

        public DuckException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class DuckBeanProcessor extends BeanProcessor {
        private Class<?> type;

        @Override
        public <T> List<T> toBeanList(final ResultSet resultSet, final Class<? extends T> type) throws SQLException {
            this.type = type;
            return super.toBeanList(resultSet, type);
        }

        @Override
        protected int[] mapColumnsToProperties(
          final ResultSetMetaData rsmd,
          final PropertyDescriptor[] props
        ) throws SQLException {

            final int cols = rsmd.getColumnCount();
            final int[] columnToProperty = new int[cols + 1];
            Arrays.fill(columnToProperty, PROPERTY_NOT_FOUND);

            for (int col = 1; col <= cols; col++) {
                String columnName = rsmd.getColumnLabel(col);

                if (null == columnName || columnName.isEmpty()) {
                    columnName = rsmd.getColumnName(col);
                }

                final String generousColumnName = columnName
                  .replace("_", "")   // more idiomatic to Java
                  .replace(" ", "");  // can't have spaces in property names

                for (int i = 0; i < props.length; i++) {
                    String propName = props[i].getName();

                    Optional<Info> optInfo = getCachedInfo(type, propName);
                    if (optInfo.isPresent() && !optInfo.get().name().isEmpty()) {
                        propName = optInfo.get().name();
                    }

                    // see if either the column name, or the generous one matches
                    if (columnName.equalsIgnoreCase(propName) ||
                      generousColumnName.equalsIgnoreCase(propName)) {
                        columnToProperty[col] = i;
                        break;
                    }
                }
            }

            return columnToProperty;
        }
    }
}