package org.v2u.toy.jetty;

import jakarta.servlet.DispatcherType;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpFilter;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.Data;
import lombok.SneakyThrows;
import lombok.experimental.Accessors;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.handler.ErrorHandler;
import org.eclipse.jetty.server.handler.HandlerList;
import org.eclipse.jetty.servlet.DefaultServlet;
import org.eclipse.jetty.servlet.FilterHolder;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.util.resource.Resource;
import org.eclipse.jetty.util.thread.QueuedThreadPool;
import org.v2u.toy.json.GsonMapper;
import org.v2u.toy.json.JsonMapper;

import java.io.File;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.EnumSet;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

public class Doge {
    public static Server<Exchange> create() {
        JsonMapper jsonMapper = new GsonMapper();
        return new Server<>((req, res) -> {
            Exchange c = new Exchange(req, res);
            c.jsonMapper(jsonMapper);
            return c;
        });
    }

    @Data
    @Accessors(fluent = true)
    public static class Server<T extends Exchange> {
        private final ExchangeFactory<T> exchangeFactory;

        private org.eclipse.jetty.server.Server server;
        private int port = 8080;
        private int maxRequestBodySize = 2 * 1024 * 1024;
        private int acceptQueueSize = 512;
        private int idleTimeout = 30;
        private int requestHeaderSize = 16 * 1024;
        private int responseHeaderSize = 16 * 1024;
        private int outputBufferSize = 32 * 1024;
        private int minThreads = 8;
        private int maxThreads = Math.max(64, Runtime.getRuntime().availableProcessors() * 8);
        private int stopTimeout = 5;
        private String staticDir = null;
        private ErrorHandler errorHandler = new ErrorHandler();

        private final Deque<String> pathStack = new ArrayDeque<>(); // 路由栈
        private final ServletContextHandler mainContext = new ServletContextHandler();

        public Server(ExchangeFactory<T> exchangeFactory) {
            this.exchangeFactory = exchangeFactory;
        }

        protected T makeExchange(HttpServletRequest req, HttpServletResponse res) {
            T exchange = exchangeFactory.create(req, res);
            exchange.maxRequestBodySize(maxRequestBodySize);
            return exchange;
        }

        @SneakyThrows
        public Server<T> start() {
            QueuedThreadPool pool = new QueuedThreadPool();
            pool.setName("DOGE");
            pool.setMinThreads(minThreads);
            pool.setMaxThreads(maxThreads);
            pool.setIdleTimeout((int) Duration.ofSeconds(idleTimeout).toMillis());

            this.server = new org.eclipse.jetty.server.Server(pool);

            // 优雅停机
            this.server.setStopTimeout(Duration.ofSeconds(stopTimeout).toMillis());
            this.server.setStopAtShutdown(true);

            //http1.1 connector
            HttpConfiguration httpConfig = new HttpConfiguration();
            httpConfig.setRequestHeaderSize(requestHeaderSize);
            httpConfig.setResponseHeaderSize(responseHeaderSize);
            httpConfig.setOutputBufferSize(outputBufferSize);
            ServerConnector http = new ServerConnector(server, new HttpConnectionFactory(httpConfig));
            http.setPort(port);
            http.setIdleTimeout(idleTimeout);
            http.setAcceptQueueSize(acceptQueueSize);
            this.server.addConnector(http);

            //bind handlers
            setDefaultHandlers();
            this.server.start();

            return this;
        }

        @SneakyThrows
        protected void setDefaultHandlers() {
            HandlerList handlerList = new HandlerList();
            mainContext.setErrorHandler(errorHandler);
            if(staticDir != null) {
                mainContext.setBaseResource(Resource.newResource(new File(staticDir)));
                DefaultServlet defaultServlet = new DefaultServlet();
                ServletHolder holder = new ServletHolder("default", defaultServlet);
                holder.setInitParameter("dirAllowed", "true"); // false 禁用目录浏览
                mainContext.addServlet(holder, "/*"); // 映射所有请求到 DefaultServlet
            }
            handlerList.addHandler(mainContext);

            server.setHandler(handlerList);
        }

        public Server<T> route(String path, Consumer<T> fn) {
            String fullPath = buildFullPath(path);
            HttpServlet servlet = new HttpServlet() {
                @Override
                protected void service(HttpServletRequest req, HttpServletResponse res) throws IOException {
                    fn.accept(makeExchange(req, res));
                }
            };

            mainContext.addServlet(new ServletHolder(servlet), fullPath);
            return this;
        }

        public Server<T> filter(String path, BiConsumer<T, Runnable> fn) {
            String fullPath = buildFullPath(path);
            HttpFilter filter = new HttpFilter() {
                @Override
                protected void doFilter(HttpServletRequest req, HttpServletResponse res, FilterChain chain) {
                    Runnable next = () -> {
                        try {
                            chain.doFilter(req, res);
                        } catch (IOException | ServletException e) {
                            throw new RuntimeException(e);
                        }
                    };
                    fn.accept(makeExchange(req, res), next);
                }
            };

            mainContext.addFilter(new FilterHolder(filter), fullPath, EnumSet.of(DispatcherType.REQUEST));
            return this;
        }


        public Server<T> group(String path, Runnable groupRoutes) {
            pathStack.push(normalizePath(path));
            try {
                groupRoutes.run();
            } finally {
                pathStack.pop();
            }
            return this;
        }

        protected String buildFullPath(String path) {
            StringBuilder fullPath = new StringBuilder();
            for (String part : pathStack) {
                fullPath.insert(0, part);
            }
            fullPath.append(normalizePath(path));
            return fullPath.toString();
        }

        protected String normalizePath(String path) {
            return "/" + path.replaceFirst("^/+", "")
              .replaceFirst("/+$", "")
              .replaceAll("/{2,}", "/");
        }
    }
}