package org.v2u.toy.jetty;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.ConstraintViolation;
import jakarta.validation.Validation;
import jakarta.validation.Validator;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;
import org.v2u.toy.json.JsonMapper;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.*;
import java.util.function.Consumer;


@Accessors(fluent = true)
public class Exchange {
    public final HttpServletRequest req;
    public final HttpServletResponse res;
    protected final Map<String, Object> attributes = new HashMap<>();
    protected int statusCode = 200;
    protected int maxRequestBodySize = Integer.MAX_VALUE;
    protected byte[] cachedBody;
    protected Map<String, String> cachedQueryParams;
    protected static final int DEFAULT_BUFFER_SIZE = 8192;
    protected boolean headersCommitted = false;
    protected boolean closed = false;
    @Getter @Setter
    protected JsonMapper jsonMapper;
    @Setter
    private Validator validator = Validation.buildDefaultValidatorFactory().getValidator();

    public Exchange(HttpServletRequest req, HttpServletResponse res) {
        this.req = req;
        this.res = res;
    }

    public String path() {
        return req.getRequestURI();
    }

    public String method() {
        return req.getMethod().toUpperCase();
    }

    /**
     * 读取所有请求头（只读视图）
     */
    public Map<String, List<String>> headers() {
        Map<String, List<String>> map = new LinkedHashMap<>();
        Enumeration<String> names = req.getHeaderNames();
        while (names.hasMoreElements()) {
            String name = names.nextElement();
            map.put(name, Collections.list(req.getHeaders(name)));
        }
        return Collections.unmodifiableMap(map);
    }

    public Exchange status(int code) {
        this.statusCode = code;
        return this;
    }

    public Exchange contentType(String type) {
        res.setContentType(type);
        return this;
    }

    public Exchange header(String name, String value) {
        res.setHeader(name, value);
        return this;
    }

    public String header(String name) {
        return req.getHeader(name);
    }

    public Exchange cookie(String name, String value, int maxAge) {
        // 简易 Set-Cookie；如需 SameSite/Secure/HttpOnly 可扩展
        res.addHeader("Set-Cookie", String.format("%s=%s; Max-Age=%d; Path=/", name, value, maxAge));
        return this;
    }

    public String cookie(String name) {
        String cookieHeader = req.getHeader("Cookie");
        if (cookieHeader == null) return null;
        String[] parts = cookieHeader.split(";");
        for (String p : parts) {
            String[] kv = p.trim().split("=", 2);
            if (kv.length == 2 && kv[0].equals(name)) return kv[1];
        }
        return null;
    }

    public void redirect(String location) {
        status(302).header("Location", location).result(new byte[0]);
    }

    public void download(File file, String filename) {
        if (!file.exists() || !file.isFile()) {
            status(404).result("File not found");
            return;
        }
        contentType("application/octet-stream").header("Content-Disposition",
          "attachment; filename=\"" + filename + "\""
        );
        try {
            result(Files.readAllBytes(file.toPath()));
        } catch (IOException e) {
            throw new RuntimeException("File read failed", e);
        }
    }

    public void html(String htmlContent) {
        contentType("text/html; charset=UTF-8");
        result(htmlContent);
    }

    /**
     * 可插拔 JSON 输出（需要先设置 JsonAdapter）
     */
    public void json(Object obj) {
        if (jsonMapper == null) {
            throw new UnsupportedOperationException("JSON adapter not configured");
        }
        try {
            byte[] bytes = jsonMapper.encode(obj).getBytes(StandardCharsets.UTF_8);
            contentType("application/json");
            result(bytes);
        } catch (Exception e) {
            throw new RuntimeException("JSON serialization failed", e);
        }
    }

    public Exchange attr(String name, Object value) {
        req.setAttribute(name, value);
        return this;
    }

    @SuppressWarnings("unchecked")
    public <T> T attr(String name) {
        return (T) req.getAttribute(name);
    }

    public Map<String, String[]> paramListMap() {
        return req.getParameterMap();
    }

    public Map<String, String> paramMap() {
        Map<String, String> result = new HashMap<>();
        for (Map.Entry<String, String[]> kv : paramListMap().entrySet()) {
            result.put(kv.getKey(), kv.getValue()[0]);
        }
        return result;
    }

    public String[] params(String name) {
        return req.getParameterValues(name);
    }

    public String param(String name) {
        return req.getParameter(name);
    }

    public String body() {
        return new String(bytes(), StandardCharsets.UTF_8);
    }

    public byte[] bytes() {
        if (cachedBody != null) return cachedBody;

        try (InputStream is = req.getInputStream(); ByteArrayOutputStream bos = new ByteArrayOutputStream()) {
            byte[] buf = new byte[DEFAULT_BUFFER_SIZE];
            int total = 0, n;
            while ((n = is.read(buf)) != -1) {
                total += n;
                if (total > maxRequestBodySize) throw new IOException("Request body too large");
                bos.write(buf, 0, n);
            }
            cachedBody = bos.toByteArray();
            return cachedBody;
        } catch (IOException e) {
            throw new RuntimeException("Failed to read request body", e);
        }
    }

    public void maxRequestBodySize(int size) {
        this.maxRequestBodySize = size;
    }

    public void result(Object obj) {
        byte[] bytes = (obj == null ? "null" : obj.toString()).getBytes(StandardCharsets.UTF_8);
        result(bytes);
    }

    public void result(byte[] bytes) {
        try {
            if (!headersCommitted) {
                res.setStatus(statusCode);
                res.setContentLength(bytes.length);
                headersCommitted = true;
            }
            OutputStream os = res.getOutputStream();
            os.write(bytes);
            os.flush();
        } catch (IOException e) {
            throw new RuntimeException("Failed to send response", e);
        }
    }

    public void result(InputStream in) {
        try {
            if (!headersCommitted) {
                // 不预先设置 content-length，交给 Jetty 使用 chunked 传输
                res.setStatus(statusCode);
                headersCommitted = true;
            }
            byte[] buf = new byte[DEFAULT_BUFFER_SIZE];
            int n;
            OutputStream os = res.getOutputStream();
            while ((n = in.read(buf)) != -1) {
                os.write(buf, 0, n);
            }
            os.flush();
        } catch (IOException e) {
            throw new RuntimeException("Failed to stream response", e);
        }
    }

    public void sse(Consumer<SseEmitter> handler) {
        contentType("text/event-stream; charset=UTF-8")
          .header("Cache-Control", "no-cache")
          .header("Connection", "keep-alive");

        try (PrintWriter writer = new PrintWriter(
          new OutputStreamWriter(res.getOutputStream(), StandardCharsets.UTF_8), true)) {
            if (!headersCommitted) {
                res.setStatus(200);
                headersCommitted = true;
            }
            SseEmitter emitter = new SseEmitter(writer);
            handler.accept(emitter);
        } catch (IOException e) {
            throw new RuntimeException("SSE failed", e);
        }
    }

    public static String urlDecode(String s) {
        try {
            return URLDecoder.decode(s, StandardCharsets.UTF_8.name());
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    public <T> T checkedParam(Class<T> targetClass) {
        return checkedParam(targetClass, new Class[]{});
    }

    public <T> T checkedParam(Class<T> targetClass, Class<?>... groups) {
        Map<String, String> params = paramMap();
        String json = jsonMapper.encode(params);
        T bean = jsonMapper.decode(json, targetClass);
        return check(bean, targetClass, groups);
    }

    public <T> T checkedBody(Class<T> targetClass) {
        return checkedBody(targetClass, new Class[]{});
    }

    public <T> T checkedBody(Class<T> targetClass, Class<?>... groups) {
        T body = jsonMapper.decode(body(), targetClass);
        return check(body, targetClass, groups);
    }

    public <T> T check(T bean, Class<T> targetClass, Class<?>... groups) {
        Set<ConstraintViolation<T>> errors = validator.validate(bean, groups);

        List<String> errList = new ArrayList<>();
        if (!errors.isEmpty()) {
            for (ConstraintViolation<T> err : errors) {
                errList.add(err.getMessage());
            }

            throw new IllegalArgumentException(String.join("\n", errList));
        }

        try {
            Method vm = targetClass.getMethod("validate", List.class);
            vm.invoke(bean, List.of(groups));
        } catch (NoSuchMethodException e) {
            //pass
        } catch (InvocationTargetException | IllegalAccessException e) {
            if (e.getCause() != null) {
                throw new IllegalArgumentException(e.getCause().getMessage());
            } else {
                throw new IllegalArgumentException(e.getMessage());
            }
        }

        return bean;
    }


    public static class SseEmitter {
        private final PrintWriter writer;

        public SseEmitter(PrintWriter writer) {
            this.writer = writer;
        }

        public void send(String data) {
            if (writer.checkError()) throw new RuntimeException("Client disconnected");
            writer.println("data: " + data);
            writer.println();
            writer.flush();
        }

        public void send(String event, String data) {
            if (writer.checkError()) throw new RuntimeException("Client disconnected");
            writer.println("event: " + event);
            writer.println("data: " + data);
            writer.println();
            writer.flush();
        }
    }
}