/*
 * Copyright (C) Verifyica project authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.verifyica.api;

import static java.lang.String.format;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.function.Consumer;

/**
 * Utility class for executing shell scripts using various Unix shells.
 */
public class ShellScript {

    /**
     * Enum representing supported Unix shells.
     */
    public enum Shell {

        /**
         * bash shell
         */
        BASH("bash"),

        /**
         * sh shell
         */
        SH("sh"),

        /**
         * zsh shell
         */
        ZSH("zsh"),

        /**
         * ksh shell
         */
        KSH("ksh"),

        /**
         * dash shell
         */
        DASH("dash"),

        /**
         * fish shell
         */
        FISH("fish");

        private final String shell;

        /**
         * Constructor
         *
         * @param shell the name of the shell executable
         */
        Shell(String shell) {
            this.shell = shell;
        }

        /**
         * Method to get the shell executable name.
         *
         * @return the shell executable name
         */
        public String getShell() {
            return shell;
        }

        @Override
        public String toString() {
            return shell;
        }
    }

    /**
     * Result class to encapsulate the result of executing a shell script.
     */
    public static class Result {

        private final int exitCode;
        private final Throwable throwable;

        /**
         * Constructor
         *
         * @param exitCode the exit code of the script execution
         * @param throwable if an error occurred during execution, null otherwise
         */
        private Result(int exitCode, Throwable throwable) {
            this.exitCode = exitCode;
            this.throwable = throwable;
        }

        /**
         * Method to get the exit code of the script execution.
         *
         * @return the exit code of the script execution
         */
        public int getExitCode() {
            return exitCode;
        }

        /**
         * Method to return whether an error occurred during execution.
         *
         * @return true if an error occurred, otherwise false
         */
        public boolean hasThrowable() {
            return throwable != null;
        }

        /**
         * Method to get the throwable if an error occurred during execution.
         *
         * @return the throwable, or null if no error occurred
         */
        public Throwable getThrowable() {
            return throwable;
        }
    }

    /**
     * Constructor
     */
    private ShellScript() {
        // INTENTIONALLY BLANK
    }

    /**
     * Method to check if a shell is available in the system PATH.
     *
     * @param shell the shell
     * @return true if the shell is available, otherwise false
     */
    private static boolean isShellAvailable(String shell) {
        String pathEnvironmentVariable = System.getenv("PATH");
        if (pathEnvironmentVariable == null || pathEnvironmentVariable.trim().isEmpty()) {
            return false;
        }

        String[] paths = pathEnvironmentVariable.split(File.pathSeparator);
        for (String dir : paths) {
            File candidate = new File(dir, shell);
            if (candidate.isFile() && candidate.canExecute()) {
                return true;
            }
        }

        return false;
    }

    /**
     * Execute a shell script using the specified shell in the current working directory.
     *
     * @param shell shell to use (e.g., BASH)
     * @param script the shell script
     * @param consumer consumer to handle each line of output
     * @return result with exit code and optional error
     */
    public static Result execute(Shell shell, Path script, Consumer<String> consumer) {
        return execute(shell, null, script, consumer);
    }

    /**
     * Execute a shell script using the specified shell in a given working directory.
     *
     * @param shell shell to use (e.g., BASH)
     * @param workingDirectory the directory to run the script in, or null for default
     * @param script the shell script
     * @param consumer consumer to handle each line of output
     * @return result with exit code and optional error
     */
    public static Result execute(Shell shell, Path workingDirectory, Path script, Consumer<String> consumer) {
        if (shell == null) {
            throw new IllegalArgumentException("shell is null");
        }

        if (!isShellAvailable(shell.getShell())) {
            throw new IllegalArgumentException(format("shell [%s] not found in system PATH", shell.getShell()));
        }

        if (script == null) {
            throw new IllegalArgumentException("script is null");
        }

        Path scriptPath =
                script.isAbsolute() ? script : workingDirectory != null ? workingDirectory.resolve(script) : script;

        if (!Files.isRegularFile(scriptPath)) {
            throw new IllegalArgumentException(format("script [%s] is not a file", scriptPath));
        }

        if (!Files.isReadable(scriptPath)) {
            throw new IllegalArgumentException(format("script [%s] is not readable", scriptPath));
        }

        if (!Files.isExecutable(scriptPath)) {
            throw new IllegalArgumentException(format("script [%s] is not executable", scriptPath));
        }

        if (consumer == null) {
            throw new IllegalArgumentException("consumer is null");
        }

        ProcessBuilder builder =
                new ProcessBuilder(shell.getShell(), scriptPath.toAbsolutePath().toString());

        if (workingDirectory != null) {
            builder.directory(workingDirectory.toFile());
        } else {
            Path scriptDir = scriptPath.getParent();
            builder.directory(scriptDir != null ? scriptDir.toFile() : new File("."));
        }

        builder.redirectErrorStream(true);

        try {
            Process process = builder.start();

            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()))) {
                String line;
                while ((line = reader.readLine()) != null) {
                    consumer.accept(line);
                }
            }

            int exitCode = process.waitFor();
            return new Result(exitCode, null);
        } catch (IOException | InterruptedException e) {
            return new Result(-1, e);
        }
    }
}
