/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.base.Preconditions;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.PriorityQueue;
import java.util.SortedSet;
import java.util.concurrent.CopyOnWriteArrayList;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hdfs.server.namenode.CheckableNameNodeResource;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogOutputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLog;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogOp;
import org.apache.hadoop.hdfs.server.namenode.FileJournalManager;
import org.apache.hadoop.hdfs.server.namenode.JournalManager;
import org.apache.hadoop.hdfs.server.namenode.NameNodeResourcePolicy;
import org.apache.hadoop.hdfs.server.namenode.RedundantEditLogInputStream;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.hdfs.server.protocol.RemoteEditLog;
import org.apache.hadoop.hdfs.server.protocol.RemoteEditLogManifest;
import org.apache.hadoop.util.ExitUtil;

public class JournalSet
implements JournalManager {
    static final Log LOG = LogFactory.getLog(FSEditLog.class);
    public static final Comparator<EditLogInputStream> EDIT_LOG_INPUT_STREAM_COMPARATOR = new Comparator<EditLogInputStream>(){

        @Override
        public int compare(EditLogInputStream a, EditLogInputStream b) {
            return ComparisonChain.start().compare(a.getFirstTxId(), b.getFirstTxId()).compare(b.getLastTxId(), a.getLastTxId()).result();
        }
    };
    private List<JournalAndStream> journals = new CopyOnWriteArrayList<JournalAndStream>();
    final int minimumRedundantJournals;

    JournalSet(int minimumRedundantResources) {
        this.minimumRedundantJournals = minimumRedundantResources;
    }

    @Override
    public void format(NamespaceInfo nsInfo) throws IOException {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean hasSomeData() throws IOException {
        throw new UnsupportedOperationException();
    }

    @Override
    public EditLogOutputStream startLogSegment(final long txId) throws IOException {
        this.mapJournalsAndReportErrors(new JournalClosure(){

            @Override
            public void apply(JournalAndStream jas) throws IOException {
                jas.startLogSegment(txId);
            }
        }, "starting log segment " + txId);
        return new JournalSetOutputStream();
    }

    @Override
    public void finalizeLogSegment(final long firstTxId, final long lastTxId) throws IOException {
        this.mapJournalsAndReportErrors(new JournalClosure(){

            @Override
            public void apply(JournalAndStream jas) throws IOException {
                if (jas.isActive()) {
                    jas.closeStream();
                    jas.getManager().finalizeLogSegment(firstTxId, lastTxId);
                }
            }
        }, "finalize log segment " + firstTxId + ", " + lastTxId);
    }

    @Override
    public void close() throws IOException {
        this.mapJournalsAndReportErrors(new JournalClosure(){

            @Override
            public void apply(JournalAndStream jas) throws IOException {
                jas.close();
            }
        }, "close journal");
    }

    @Override
    public void selectInputStreams(Collection<EditLogInputStream> streams, long fromTxId, boolean inProgressOk) throws IOException {
        PriorityQueue<EditLogInputStream> allStreams = new PriorityQueue<EditLogInputStream>(64, EDIT_LOG_INPUT_STREAM_COMPARATOR);
        for (JournalAndStream jas : this.journals) {
            if (jas.isDisabled()) {
                LOG.info((Object)("Skipping jas " + jas + " since it's disabled"));
                continue;
            }
            try {
                jas.getManager().selectInputStreams(allStreams, fromTxId, inProgressOk);
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Unable to determine input streams from " + jas.getManager() + ". Skipping."), (Throwable)ioe);
            }
        }
        JournalSet.chainAndMakeRedundantStreams(streams, allStreams, fromTxId);
    }

    public static void chainAndMakeRedundantStreams(Collection<EditLogInputStream> outStreams, PriorityQueue<EditLogInputStream> allStreams, long fromTxId) {
        EditLogInputStream elis;
        LinkedList<EditLogInputStream> acc = new LinkedList<EditLogInputStream>();
        while ((elis = allStreams.poll()) != null) {
            if (acc.isEmpty()) {
                acc.add(elis);
                continue;
            }
            long accFirstTxId = ((EditLogInputStream)acc.get(0)).getFirstTxId();
            if (accFirstTxId == elis.getFirstTxId()) {
                acc.add(elis);
                continue;
            }
            if (accFirstTxId < elis.getFirstTxId()) {
                outStreams.add(new RedundantEditLogInputStream(acc, fromTxId));
                acc.clear();
                acc.add(elis);
                continue;
            }
            if (accFirstTxId <= elis.getFirstTxId()) continue;
            throw new RuntimeException("sorted set invariants violated!  Got stream with first txid " + elis.getFirstTxId() + ", but the last firstTxId was " + accFirstTxId);
        }
        if (!acc.isEmpty()) {
            outStreams.add(new RedundantEditLogInputStream(acc, fromTxId));
            acc.clear();
        }
    }

    public boolean isEmpty() {
        return !NameNodeResourcePolicy.areResourcesAvailable(this.journals, this.minimumRedundantJournals);
    }

    private void disableAndReportErrorOnJournals(List<JournalAndStream> badJournals) {
        if (badJournals == null || badJournals.isEmpty()) {
            return;
        }
        for (JournalAndStream j : badJournals) {
            LOG.error((Object)("Disabling journal " + j));
            j.abort();
            j.setDisabled(true);
        }
    }

    private void mapJournalsAndReportErrors(JournalClosure closure, String status) throws IOException {
        LinkedList badJAS = Lists.newLinkedList();
        for (JournalAndStream jas : this.journals) {
            try {
                closure.apply(jas);
            }
            catch (Throwable t) {
                if (jas.isRequired()) {
                    String msg = "Error: " + status + " failed for required journal (" + jas + ")";
                    LOG.fatal((Object)msg, t);
                    this.abortAllJournals();
                    ExitUtil.terminate((int)1, (String)msg);
                    continue;
                }
                LOG.error((Object)("Error: " + status + " failed for (journal " + jas + ")"), t);
                badJAS.add(jas);
            }
        }
        this.disableAndReportErrorOnJournals(badJAS);
        if (!NameNodeResourcePolicy.areResourcesAvailable(this.journals, this.minimumRedundantJournals)) {
            String message = status + " failed for too many journals";
            LOG.error((Object)("Error: " + message));
            throw new IOException(message);
        }
    }

    private void abortAllJournals() {
        for (JournalAndStream jas : this.journals) {
            if (!jas.isActive()) continue;
            jas.abort();
        }
    }

    @Override
    public void setOutputBufferCapacity(final int size) {
        try {
            this.mapJournalsAndReportErrors(new JournalClosure(){

                @Override
                public void apply(JournalAndStream jas) throws IOException {
                    jas.getManager().setOutputBufferCapacity(size);
                }
            }, "setOutputBufferCapacity");
        }
        catch (IOException e) {
            LOG.error((Object)"Error in setting outputbuffer capacity");
        }
    }

    List<JournalAndStream> getAllJournalStreams() {
        return this.journals;
    }

    List<JournalManager> getJournalManagers() {
        ArrayList<JournalManager> jList = new ArrayList<JournalManager>();
        for (JournalAndStream j : this.journals) {
            jList.add(j.getManager());
        }
        return jList;
    }

    void add(JournalManager j, boolean required) {
        JournalAndStream jas = new JournalAndStream(j, required);
        this.journals.add(jas);
    }

    void remove(JournalManager j) {
        JournalAndStream jasToRemove = null;
        for (JournalAndStream jas : this.journals) {
            if (!jas.getManager().equals(j)) continue;
            jasToRemove = jas;
            break;
        }
        if (jasToRemove != null) {
            jasToRemove.abort();
            this.journals.remove(jasToRemove);
        }
    }

    @Override
    public void purgeLogsOlderThan(final long minTxIdToKeep) throws IOException {
        this.mapJournalsAndReportErrors(new JournalClosure(){

            @Override
            public void apply(JournalAndStream jas) throws IOException {
                jas.getManager().purgeLogsOlderThan(minTxIdToKeep);
            }
        }, "purgeLogsOlderThan " + minTxIdToKeep);
    }

    @Override
    public void recoverUnfinalizedSegments() throws IOException {
        this.mapJournalsAndReportErrors(new JournalClosure(){

            @Override
            public void apply(JournalAndStream jas) throws IOException {
                jas.getManager().recoverUnfinalizedSegments();
            }
        }, "recoverUnfinalizedSegments");
    }

    public synchronized RemoteEditLogManifest getEditLogManifest(long fromTxId) {
        ArrayList allLogs = Lists.newArrayList();
        for (JournalAndStream j : this.journals) {
            if (!(j.getManager() instanceof FileJournalManager)) continue;
            FileJournalManager fjm = (FileJournalManager)j.getManager();
            try {
                allLogs.addAll(fjm.getRemoteEditLogs(fromTxId, false));
            }
            catch (Throwable t) {
                LOG.warn((Object)("Cannot list edit logs in " + fjm), t);
            }
        }
        ImmutableListMultimap logsByStartTxId = Multimaps.index((Iterable)allLogs, RemoteEditLog.GET_START_TXID);
        long curStartTxId = fromTxId;
        ArrayList logs = Lists.newArrayList();
        while (true) {
            ImmutableList logGroup;
            if ((logGroup = logsByStartTxId.get((Object)curStartTxId)).isEmpty()) {
                SortedSet<Object> startTxIds = Sets.newTreeSet((Iterable)logsByStartTxId.keySet());
                if ((startTxIds = startTxIds.tailSet(curStartTxId)).isEmpty()) break;
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("Found gap in logs at " + curStartTxId + ": " + "not returning previous logs in manifest."));
                }
                logs.clear();
                curStartTxId = (Long)startTxIds.first();
                continue;
            }
            RemoteEditLog bestLog = (RemoteEditLog)Collections.max(logGroup);
            logs.add(bestLog);
            curStartTxId = bestLog.getEndTxId() + 1L;
        }
        RemoteEditLogManifest ret = new RemoteEditLogManifest(logs);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Generated manifest for logs since " + fromTxId + ":" + ret));
        }
        return ret;
    }

    String getSyncTimes() {
        StringBuilder buf = new StringBuilder();
        for (JournalAndStream jas : this.journals) {
            if (!jas.isActive()) continue;
            buf.append(jas.getCurrentStream().getTotalSyncTime());
            buf.append(" ");
        }
        return buf.toString();
    }

    private class JournalSetOutputStream
    extends EditLogOutputStream {
        JournalSetOutputStream() throws IOException {
        }

        @Override
        public void write(final FSEditLogOp op) throws IOException {
            JournalSet.this.mapJournalsAndReportErrors(new JournalClosure(){

                @Override
                public void apply(JournalAndStream jas) throws IOException {
                    if (jas.isActive()) {
                        jas.getCurrentStream().write(op);
                    }
                }
            }, "write op");
        }

        @Override
        public void writeRaw(final byte[] data, final int offset, final int length) throws IOException {
            JournalSet.this.mapJournalsAndReportErrors(new JournalClosure(){

                @Override
                public void apply(JournalAndStream jas) throws IOException {
                    if (jas.isActive()) {
                        jas.getCurrentStream().writeRaw(data, offset, length);
                    }
                }
            }, "write bytes");
        }

        @Override
        public void create() throws IOException {
            JournalSet.this.mapJournalsAndReportErrors(new JournalClosure(){

                @Override
                public void apply(JournalAndStream jas) throws IOException {
                    if (jas.isActive()) {
                        jas.getCurrentStream().create();
                    }
                }
            }, "create");
        }

        @Override
        public void close() throws IOException {
            JournalSet.this.mapJournalsAndReportErrors(new JournalClosure(){

                @Override
                public void apply(JournalAndStream jas) throws IOException {
                    jas.closeStream();
                }
            }, "close");
        }

        @Override
        public void abort() throws IOException {
            JournalSet.this.mapJournalsAndReportErrors(new JournalClosure(){

                @Override
                public void apply(JournalAndStream jas) throws IOException {
                    jas.abort();
                }
            }, "abort");
        }

        @Override
        public void setReadyToFlush() throws IOException {
            JournalSet.this.mapJournalsAndReportErrors(new JournalClosure(){

                @Override
                public void apply(JournalAndStream jas) throws IOException {
                    if (jas.isActive()) {
                        jas.getCurrentStream().setReadyToFlush();
                    }
                }
            }, "setReadyToFlush");
        }

        @Override
        protected void flushAndSync(final boolean durable) throws IOException {
            JournalSet.this.mapJournalsAndReportErrors(new JournalClosure(){

                @Override
                public void apply(JournalAndStream jas) throws IOException {
                    if (jas.isActive()) {
                        jas.getCurrentStream().flushAndSync(durable);
                    }
                }
            }, "flushAndSync");
        }

        @Override
        public void flush() throws IOException {
            JournalSet.this.mapJournalsAndReportErrors(new JournalClosure(){

                @Override
                public void apply(JournalAndStream jas) throws IOException {
                    if (jas.isActive()) {
                        jas.getCurrentStream().flush();
                    }
                }
            }, "flush");
        }

        @Override
        public boolean shouldForceSync() {
            for (JournalAndStream js : JournalSet.this.journals) {
                if (!js.isActive() || !js.getCurrentStream().shouldForceSync()) continue;
                return true;
            }
            return false;
        }

        @Override
        protected long getNumSync() {
            for (JournalAndStream jas : JournalSet.this.journals) {
                if (!jas.isActive()) continue;
                return jas.getCurrentStream().getNumSync();
            }
            return 0L;
        }
    }

    private static interface JournalClosure {
        public void apply(JournalAndStream var1) throws IOException;
    }

    static class JournalAndStream
    implements CheckableNameNodeResource {
        private final JournalManager journal;
        private boolean disabled = false;
        private EditLogOutputStream stream;
        private boolean required = false;

        public JournalAndStream(JournalManager manager, boolean required) {
            this.journal = manager;
            this.required = required;
        }

        public void startLogSegment(long txId) throws IOException {
            Preconditions.checkState((this.stream == null ? 1 : 0) != 0);
            this.disabled = false;
            this.stream = this.journal.startLogSegment(txId);
        }

        public void closeStream() throws IOException {
            if (this.stream == null) {
                return;
            }
            this.stream.close();
            this.stream = null;
        }

        public void close() throws IOException {
            this.closeStream();
            this.journal.close();
        }

        public void abort() {
            if (this.stream == null) {
                return;
            }
            try {
                this.stream.abort();
            }
            catch (IOException ioe) {
                LOG.error((Object)("Unable to abort stream " + this.stream), (Throwable)ioe);
            }
            this.stream = null;
        }

        boolean isActive() {
            return this.stream != null;
        }

        EditLogOutputStream getCurrentStream() {
            return this.stream;
        }

        public String toString() {
            return "JournalAndStream(mgr=" + this.journal + ", " + "stream=" + this.stream + ")";
        }

        void setCurrentStreamForTests(EditLogOutputStream stream) {
            this.stream = stream;
        }

        JournalManager getManager() {
            return this.journal;
        }

        boolean isDisabled() {
            return this.disabled;
        }

        private void setDisabled(boolean disabled) {
            this.disabled = disabled;
        }

        @Override
        public boolean isResourceAvailable() {
            return !this.isDisabled();
        }

        @Override
        public boolean isRequired() {
            return this.required;
        }
    }
}

