/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.client;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ComparisonChain;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.io.FileInputStream;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.client.ClientMmap;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.io.IOUtils;

@InterfaceAudience.Private
public class ClientMmapManager
implements Closeable {
    public static final Log LOG = LogFactory.getLog(ClientMmapManager.class);
    private boolean closed = false;
    private final int cacheSize;
    private final long timeoutNs;
    private final int runsPerTimeout;
    private final Lock lock = new ReentrantLock();
    private final TreeMap<Key, Waitable<ClientMmap>> mmaps = new TreeMap();
    private final TreeMap<Long, ClientMmap> evictable = new TreeMap();
    private final ScheduledThreadPoolExecutor executor = new ScheduledThreadPoolExecutor(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("ClientMmapManager").build());
    private CacheCleaner cacheCleaner;

    public static ClientMmapManager fromConf(Configuration conf) {
        return new ClientMmapManager(conf.getInt("dfs.client.mmap.cache.size", 1024), conf.getLong("dfs.client.mmap.cache.timeout.ms", 900000L), conf.getInt("dfs.client.mmap.cache.thread.runs.per.timeout", 4));
    }

    public ClientMmapManager(int cacheSize, long timeoutMs, int runsPerTimeout) {
        this.cacheSize = cacheSize;
        this.timeoutNs = timeoutMs * 1000000L;
        this.runsPerTimeout = runsPerTimeout;
    }

    long getTimeoutMs() {
        return this.timeoutNs / 1000000L;
    }

    int getRunsPerTimeout() {
        return this.runsPerTimeout;
    }

    public String verifyConfigurationMatches(Configuration conf) {
        StringBuilder bld = new StringBuilder();
        int cacheSize = conf.getInt("dfs.client.mmap.cache.size", 1024);
        if (this.cacheSize != cacheSize) {
            bld.append("You specified a cache size of ").append(cacheSize).append(", but the existing cache size is ").append(this.cacheSize).append(".  ");
        }
        long timeoutMs = conf.getLong("dfs.client.mmap.cache.timeout.ms", 900000L);
        if (this.getTimeoutMs() != timeoutMs) {
            bld.append("You specified a cache timeout of ").append(timeoutMs).append(" ms, but the existing cache timeout is ").append(this.getTimeoutMs()).append("ms").append(".  ");
        }
        int runsPerTimeout = conf.getInt("dfs.client.mmap.cache.thread.runs.per.timeout", 4);
        if (this.getRunsPerTimeout() != runsPerTimeout) {
            bld.append("You specified ").append(runsPerTimeout).append(" runs per timeout, but the existing runs per timeout is ").append(this.getTimeoutMs()).append(".  ");
        }
        return bld.toString();
    }

    private void evictStaleEntries(long curTime) {
        if (this.closed) {
            return;
        }
        Iterator<Map.Entry<Long, ClientMmap>> iter = this.evictable.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<Long, ClientMmap> entry = iter.next();
            if (entry.getKey() + this.timeoutNs >= curTime) {
                return;
            }
            ClientMmap mmap = entry.getValue();
            Key key = new Key(mmap.getBlock(), mmap.getDatanodeID());
            this.mmaps.remove(key);
            iter.remove();
            mmap.unmap();
        }
    }

    private boolean evictOne() {
        Map.Entry<Long, ClientMmap> entry = this.evictable.pollFirstEntry();
        if (entry == null) {
            return false;
        }
        ClientMmap evictedMmap = entry.getValue();
        Key evictedKey = new Key(evictedMmap.getBlock(), evictedMmap.getDatanodeID());
        this.mmaps.remove(evictedKey);
        evictedMmap.unmap();
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ClientMmap create(Key key, FileInputStream in) throws IOException {
        if (this.mmaps.size() + 1 > this.cacheSize && !this.evictOne()) {
            LOG.warn((Object)("mmap cache is full (with " + this.cacheSize + " elements) and " + "nothing is evictable.  Ignoring request for mmap with " + "datanodeID=" + key.datanode + ", " + "block=" + key.block));
            return null;
        }
        Waitable<ClientMmap> waitable = new Waitable<ClientMmap>(this.lock.newCondition());
        this.mmaps.put(key, waitable);
        boolean success = false;
        ClientMmap mmap = null;
        try {
            try {
                this.lock.unlock();
                mmap = ClientMmap.load(this, in, key.block, key.datanode);
            }
            finally {
                this.lock.lock();
            }
            if (this.cacheCleaner == null) {
                this.cacheCleaner = new CacheCleaner(this);
                ScheduledFuture<?> future = this.executor.scheduleAtFixedRate(this.cacheCleaner, this.timeoutNs, this.timeoutNs / (long)this.runsPerTimeout, TimeUnit.NANOSECONDS);
                this.cacheCleaner.setFuture(future);
            }
            success = true;
        }
        finally {
            if (!success) {
                LOG.warn((Object)("failed to create mmap for datanodeID=" + key.datanode + ", " + "block=" + key.block));
                this.mmaps.remove(key);
            }
            waitable.provide(mmap);
        }
        if (LOG.isDebugEnabled()) {
            LOG.info((Object)("created a new ClientMmap for block " + key.block + " on datanode " + key.datanode));
        }
        return mmap;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ClientMmap fetch(DatanodeID datanodeID, ExtendedBlock block, FileInputStream in) throws IOException, InterruptedException {
        LOG.debug((Object)("fetching mmap with datanodeID=" + datanodeID + ", " + "block=" + block));
        Key key = new Key(block, datanodeID);
        ClientMmap mmap = null;
        try {
            this.lock.lock();
            if (this.closed) {
                throw new IOException("ClientMmapManager is closed.");
            }
            while (mmap == null) {
                Waitable<ClientMmap> entry = this.mmaps.get(key);
                if (entry == null) {
                    ClientMmap clientMmap = this.create(key, in);
                    return clientMmap;
                }
                mmap = entry.await();
            }
            if (mmap.ref() == 1) {
                this.evictable.remove(mmap.getLastEvictableTimeNs());
            }
        }
        finally {
            this.lock.unlock();
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("reusing existing mmap with datanodeID=" + datanodeID + ", " + "block=" + block));
        }
        return mmap;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void makeEvictable(ClientMmap mmap) {
        try {
            this.lock.lock();
            if (this.closed) {
                mmap.unmap();
                return;
            }
            long now = System.nanoTime();
            while (this.evictable.containsKey(now)) {
                ++now;
            }
            mmap.setLastEvictableTimeNs(now);
            this.evictable.put(now, mmap);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        try {
            this.lock.lock();
            this.closed = true;
            IOUtils.cleanup((Log)LOG, (Closeable[])new Closeable[]{this.cacheCleaner});
            this.evictStaleEntries(Long.MAX_VALUE);
            this.executor.shutdown();
        }
        finally {
            this.lock.unlock();
        }
    }

    @VisibleForTesting
    public synchronized void visitMmaps(ClientMmapVisitor visitor) throws InterruptedException {
        for (Waitable<ClientMmap> entry : this.mmaps.values()) {
            visitor.accept(entry.await());
        }
    }

    public void visitEvictable(ClientMmapVisitor visitor) throws InterruptedException {
        for (ClientMmap mmap : this.evictable.values()) {
            visitor.accept(mmap);
        }
    }

    @VisibleForTesting
    public static interface ClientMmapVisitor {
        public void accept(ClientMmap var1);
    }

    private static class CacheCleaner
    implements Runnable,
    Closeable {
        private WeakReference<ClientMmapManager> managerRef;
        private ScheduledFuture<?> future;

        CacheCleaner(ClientMmapManager manager) {
            this.managerRef = new WeakReference<ClientMmapManager>(manager);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            ClientMmapManager manager = (ClientMmapManager)this.managerRef.get();
            if (manager == null) {
                return;
            }
            long curTime = System.nanoTime();
            try {
                manager.lock.lock();
                manager.evictStaleEntries(curTime);
            }
            finally {
                manager.lock.unlock();
            }
        }

        void setFuture(ScheduledFuture<?> future) {
            this.future = future;
        }

        @Override
        public void close() throws IOException {
            this.future.cancel(false);
        }
    }

    private static class Key
    implements Comparable<Key> {
        private final ExtendedBlock block;
        private final DatanodeID datanode;

        Key(ExtendedBlock block, DatanodeID datanode) {
            this.block = block;
            this.datanode = datanode;
        }

        @Override
        public int compareTo(Key o) {
            return ComparisonChain.start().compare(this.block.getBlockId(), o.block.getBlockId()).compare(this.block.getGenerationStamp(), o.block.getGenerationStamp()).compare((Comparable)((Object)this.block.getBlockPoolId()), (Comparable)((Object)o.block.getBlockPoolId())).compare((Comparable)this.datanode, (Comparable)o.datanode).result();
        }

        public boolean equals(Object rhs) {
            if (rhs == null) {
                return false;
            }
            try {
                Key o = (Key)rhs;
                return this.compareTo(o) == 0;
            }
            catch (ClassCastException e) {
                return false;
            }
        }

        public int hashCode() {
            return this.block.hashCode() ^ this.datanode.hashCode();
        }
    }

    private static class Waitable<T> {
        private T val = null;
        private final Condition cond;

        public Waitable(Condition cond) {
            this.cond = cond;
        }

        public T await() throws InterruptedException {
            while (this.val == null) {
                this.cond.await();
            }
            return this.val;
        }

        public void provide(T val) {
            this.val = val;
            this.cond.signalAll();
        }
    }
}

