/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.discovery.zen;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchIllegalArgumentException;
import org.elasticsearch.ElasticsearchIllegalStateException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.ClusterName;
import org.elasticsearch.cluster.ClusterService;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateNonMasterUpdateTask;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.ProcessedClusterStateNonMasterUpdateTask;
import org.elasticsearch.cluster.ProcessedClusterStateUpdateTask;
import org.elasticsearch.cluster.block.ClusterBlocks;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeService;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.RoutingService;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.cluster.settings.ClusterDynamicSettings;
import org.elasticsearch.cluster.settings.DynamicSettings;
import org.elasticsearch.cluster.settings.Validator;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.base.Objects;
import org.elasticsearch.common.collect.Lists;
import org.elasticsearch.common.collect.Sets;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.component.Lifecycle;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.inject.internal.Nullable;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.discovery.Discovery;
import org.elasticsearch.discovery.DiscoverySettings;
import org.elasticsearch.discovery.InitialStateDiscoveryListener;
import org.elasticsearch.discovery.zen.elect.ElectMasterService;
import org.elasticsearch.discovery.zen.fd.MasterFaultDetection;
import org.elasticsearch.discovery.zen.fd.NodesFaultDetection;
import org.elasticsearch.discovery.zen.membership.MembershipAction;
import org.elasticsearch.discovery.zen.ping.PingContextProvider;
import org.elasticsearch.discovery.zen.ping.ZenPing;
import org.elasticsearch.discovery.zen.ping.ZenPingService;
import org.elasticsearch.discovery.zen.publish.PublishClusterStateAction;
import org.elasticsearch.node.service.NodeService;
import org.elasticsearch.node.settings.NodeSettingsService;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.BaseTransportRequestHandler;
import org.elasticsearch.transport.EmptyTransportResponseHandler;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportService;

public class ZenDiscovery
extends AbstractLifecycleComponent<Discovery>
implements Discovery,
PingContextProvider {
    public static final String SETTING_REJOIN_ON_MASTER_GONE = "discovery.zen.rejoin_on_master_gone";
    public static final String SETTING_PING_TIMEOUT = "discovery.zen.ping.timeout";
    public static final String SETTING_JOIN_TIMEOUT = "discovery.zen.join_timeout";
    public static final String SETTING_JOIN_RETRY_ATTEMPTS = "discovery.zen.join_retry_attempts";
    public static final String SETTING_JOIN_RETRY_DELAY = "discovery.zen.join_retry_delay";
    public static final String SETTING_MAX_PINGS_FROM_ANOTHER_MASTER = "discovery.zen.max_pings_from_another_master";
    public static final String SETTING_SEND_LEAVE_REQUEST = "discovery.zen.send_leave_request";
    public static final String SETTING_MASTER_ELECTION_FILTER_CLIENT = "discovery.zen.master_election.filter_client";
    public static final String SETTING_MASTER_ELECTION_FILTER_DATA = "discovery.zen.master_election.filter_data";
    public static final String DISCOVERY_REJOIN_ACTION_NAME = "internal:discovery/zen/rejoin";
    private final TransportService transportService;
    private final ClusterService clusterService;
    private RoutingService routingService;
    private final ClusterName clusterName;
    private final DiscoveryNodeService discoveryNodeService;
    private final DiscoverySettings discoverySettings;
    private final ZenPingService pingService;
    private final MasterFaultDetection masterFD;
    private final NodesFaultDetection nodesFD;
    private final PublishClusterStateAction publishClusterState;
    private final MembershipAction membership;
    private final TimeValue pingTimeout;
    private final TimeValue joinTimeout;
    private final int joinRetryAttempts;
    private final TimeValue joinRetryDelay;
    private final int maxPingsFromAnotherMaster;
    private final boolean sendLeaveRequest;
    private final ElectMasterService electMaster;
    private final boolean masterElectionFilterClientNodes;
    private final boolean masterElectionFilterDataNodes;
    private final CopyOnWriteArrayList<InitialStateDiscoveryListener> initialStateListeners = new CopyOnWriteArrayList();
    private final JoinThreadControl joinThreadControl;
    private final AtomicBoolean initialStateSent = new AtomicBoolean();
    private volatile boolean rejoinOnMasterGone;
    private final AtomicLong clusterJoinsCounter = new AtomicLong();
    @Nullable
    private NodeService nodeService;
    private final BlockingQueue<Tuple<DiscoveryNode, MembershipAction.JoinCallback>> processJoinRequests = ConcurrentCollections.newBlockingQueue();
    private final BlockingQueue<ProcessClusterState> processNewClusterStates = ConcurrentCollections.newBlockingQueue();

    @Inject
    public ZenDiscovery(Settings settings, ClusterName clusterName, ThreadPool threadPool, TransportService transportService, final ClusterService clusterService, NodeSettingsService nodeSettingsService, DiscoveryNodeService discoveryNodeService, ZenPingService pingService, ElectMasterService electMasterService, DiscoverySettings discoverySettings, @ClusterDynamicSettings DynamicSettings dynamicSettings) {
        super(settings);
        this.clusterName = clusterName;
        this.clusterService = clusterService;
        this.transportService = transportService;
        this.discoveryNodeService = discoveryNodeService;
        this.discoverySettings = discoverySettings;
        this.pingService = pingService;
        this.electMaster = electMasterService;
        TimeValue pingTimeout = this.componentSettings.getAsTime("initial_ping_timeout", TimeValue.timeValueSeconds(3L));
        pingTimeout = this.componentSettings.getAsTime("ping_timeout", pingTimeout);
        pingTimeout = settings.getAsTime("discovery.zen.ping_timeout", pingTimeout);
        this.pingTimeout = settings.getAsTime(SETTING_PING_TIMEOUT, pingTimeout);
        this.joinTimeout = settings.getAsTime(SETTING_JOIN_TIMEOUT, TimeValue.timeValueMillis(pingTimeout.millis() * 20L));
        this.joinRetryAttempts = settings.getAsInt(SETTING_JOIN_RETRY_ATTEMPTS, (Integer)3);
        this.joinRetryDelay = settings.getAsTime(SETTING_JOIN_RETRY_DELAY, TimeValue.timeValueMillis(100L));
        this.maxPingsFromAnotherMaster = settings.getAsInt(SETTING_MAX_PINGS_FROM_ANOTHER_MASTER, (Integer)3);
        this.sendLeaveRequest = settings.getAsBoolean(SETTING_SEND_LEAVE_REQUEST, (Boolean)true);
        this.masterElectionFilterClientNodes = settings.getAsBoolean(SETTING_MASTER_ELECTION_FILTER_CLIENT, (Boolean)true);
        this.masterElectionFilterDataNodes = settings.getAsBoolean(SETTING_MASTER_ELECTION_FILTER_DATA, (Boolean)false);
        this.rejoinOnMasterGone = settings.getAsBoolean(SETTING_REJOIN_ON_MASTER_GONE, (Boolean)true);
        if (this.joinRetryAttempts < 1) {
            throw new ElasticsearchIllegalArgumentException("'discovery.zen.join_retry_attempts' must be a positive number. got [discovery.zen.join_retry_attempts]");
        }
        if (this.maxPingsFromAnotherMaster < 1) {
            throw new ElasticsearchIllegalArgumentException("'discovery.zen.max_pings_from_another_master' must be a positive number. got [" + this.maxPingsFromAnotherMaster + "]");
        }
        this.logger.debug("using ping.timeout [{}], join.timeout [{}], master_election.filter_client [{}], master_election.filter_data [{}]", pingTimeout, this.joinTimeout, this.masterElectionFilterClientNodes, this.masterElectionFilterDataNodes);
        nodeSettingsService.addListener(new ApplySettings());
        this.masterFD = new MasterFaultDetection(settings, threadPool, transportService, clusterName, clusterService);
        this.masterFD.addListener(new MasterNodeFailureListener());
        this.nodesFD = new NodesFaultDetection(settings, threadPool, transportService, clusterName);
        this.nodesFD.addListener(new NodeFaultDetectionListener());
        this.publishClusterState = new PublishClusterStateAction(settings, transportService, this, new NewClusterStateListener(), discoverySettings, clusterName);
        this.pingService.setPingContextProvider(this);
        this.membership = new MembershipAction(settings, clusterService, transportService, this, new MembershipListener());
        this.joinThreadControl = new JoinThreadControl(threadPool);
        transportService.registerHandler(DISCOVERY_REJOIN_ACTION_NAME, new RejoinClusterRequestHandler());
        dynamicSettings.addDynamicSetting("discovery.zen.minimum_master_nodes", new Validator(){

            @Override
            public String validate(String setting, String value) {
                int intValue;
                try {
                    intValue = Integer.parseInt(value);
                }
                catch (NumberFormatException ex) {
                    return "cannot parse value [" + value + "] as an integer";
                }
                int masterNodes = clusterService.state().nodes().masterNodes().size();
                if (intValue > masterNodes) {
                    return "cannot set discovery.zen.minimum_master_nodes to more than the current master nodes count [" + masterNodes + "]";
                }
                return null;
            }
        });
    }

    @Override
    public void setNodeService(@Nullable NodeService nodeService) {
        this.nodeService = nodeService;
    }

    @Override
    public void setRoutingService(RoutingService routingService) {
        this.routingService = routingService;
    }

    @Override
    protected void doStart() throws ElasticsearchException {
        this.nodesFD.setLocalNode(this.clusterService.localNode());
        this.joinThreadControl.start();
        this.pingService.start();
        this.clusterService.submitStateUpdateTask("initial_join", new ClusterStateNonMasterUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                ZenDiscovery.this.joinThreadControl.startNewThreadIfNotRunning();
                return currentState;
            }

            @Override
            public void onFailure(String source, @org.elasticsearch.common.Nullable Throwable t) {
                ZenDiscovery.this.logger.warn("failed to start initial join process", t, new Object[0]);
            }
        });
    }

    @Override
    protected void doStop() throws ElasticsearchException {
        this.joinThreadControl.stop();
        this.pingService.stop();
        this.masterFD.stop("zen disco stop");
        this.nodesFD.stop();
        this.initialStateSent.set(false);
        DiscoveryNodes nodes = this.nodes();
        if (this.sendLeaveRequest && nodes.masterNode() != null) {
            if (!nodes.localNodeMaster()) {
                try {
                    this.membership.sendLeaveRequestBlocking(nodes.masterNode(), nodes.localNode(), TimeValue.timeValueSeconds(1L));
                }
                catch (Exception e) {
                    this.logger.debug("failed to send leave request to master [{}]", e, nodes.masterNode());
                }
            } else {
                DiscoveryNode[] possibleMasters;
                for (DiscoveryNode possibleMaster : possibleMasters = this.electMaster.nextPossibleMasters(nodes.nodes().values(), 5)) {
                    if (nodes.localNode().equals(possibleMaster)) continue;
                    try {
                        this.membership.sendLeaveRequest(nodes.localNode(), possibleMaster);
                    }
                    catch (Exception e) {
                        this.logger.debug("failed to send leave request from master [{}] to possible master [{}]", e, nodes.masterNode(), possibleMaster);
                    }
                }
            }
        }
    }

    @Override
    protected void doClose() throws ElasticsearchException {
        this.masterFD.close();
        this.nodesFD.close();
        this.publishClusterState.close();
        this.membership.close();
        this.pingService.close();
    }

    @Override
    public DiscoveryNode localNode() {
        return this.clusterService.localNode();
    }

    @Override
    public void addListener(InitialStateDiscoveryListener listener) {
        this.initialStateListeners.add(listener);
    }

    @Override
    public void removeListener(InitialStateDiscoveryListener listener) {
        this.initialStateListeners.remove(listener);
    }

    @Override
    public String nodeDescription() {
        return this.clusterName.value() + "/" + this.clusterService.localNode().id();
    }

    @Override
    public DiscoveryNodes nodes() {
        return this.clusterService.state().nodes();
    }

    @Override
    public NodeService nodeService() {
        return this.nodeService;
    }

    @Override
    public boolean nodeHasJoinedClusterOnce() {
        return this.clusterJoinsCounter.get() > 0L;
    }

    @Override
    public void publish(ClusterState clusterState, Discovery.AckListener ackListener) {
        if (!clusterState.getNodes().localNodeMaster()) {
            throw new ElasticsearchIllegalStateException("Shouldn't publish state when not master");
        }
        this.nodesFD.updateNodesAndPing(clusterState);
        this.publishClusterState.publish(clusterState, ackListener);
    }

    public boolean joiningCluster() {
        return this.joinThreadControl.joinThreadActive();
    }

    private void innerJoinCluster() {
        DiscoveryNode masterNode = null;
        final Thread currentThread = Thread.currentThread();
        while (masterNode == null && this.joinThreadControl.joinThreadActive(currentThread)) {
            masterNode = this.findMaster();
        }
        if (!this.joinThreadControl.joinThreadActive(currentThread)) {
            this.logger.trace("thread is no longer in currentJoinThread. Stopping.", new Object[0]);
            return;
        }
        if (this.clusterService.localNode().equals(masterNode)) {
            this.clusterService.submitStateUpdateTask("zen-disco-join (elected_as_master)", Priority.IMMEDIATE, new ProcessedClusterStateNonMasterUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) {
                    if (currentState.nodes().masterNode() != null) {
                        ZenDiscovery.this.logger.trace("join thread elected local node as master, but there is already a master in place: {}", currentState.nodes().masterNode());
                        return currentState;
                    }
                    DiscoveryNodes.Builder builder = new DiscoveryNodes.Builder(currentState.nodes()).masterNodeId(currentState.nodes().localNode().id());
                    ClusterBlocks clusterBlocks = ClusterBlocks.builder().blocks(currentState.blocks()).removeGlobalBlock(ZenDiscovery.this.discoverySettings.getNoMasterBlock()).build();
                    currentState = ClusterState.builder(currentState).nodes(builder).blocks(clusterBlocks).build();
                    RoutingAllocation.Result result = ZenDiscovery.this.routingService.getAllocationService().reroute(currentState);
                    return ClusterState.builder(currentState).routingResult(result).build();
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    ZenDiscovery.this.logger.error("unexpected failure during [{}]", t, source);
                    ZenDiscovery.this.joinThreadControl.markThreadAsDoneAndStartNew(currentThread);
                }

                @Override
                public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                    if (newState.nodes().localNodeMaster()) {
                        ZenDiscovery.this.joinThreadControl.markThreadAsDone(currentThread);
                        ZenDiscovery.this.nodesFD.updateNodesAndPing(newState);
                    } else {
                        ZenDiscovery.this.joinThreadControl.markThreadAsDoneAndStartNew(currentThread);
                    }
                    ZenDiscovery.this.sendInitialStateEventIfNeeded();
                    long count = ZenDiscovery.this.clusterJoinsCounter.incrementAndGet();
                    ZenDiscovery.this.logger.trace("cluster joins counter set to [{}] (elected as master)", count);
                }
            });
        } else {
            final boolean success = this.joinElectedMaster(masterNode);
            final DiscoveryNode finalMasterNode = masterNode;
            this.clusterService.submitStateUpdateTask("finalize_join (" + masterNode + ")", new ClusterStateNonMasterUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) throws Exception {
                    if (!success) {
                        ZenDiscovery.this.joinThreadControl.markThreadAsDoneAndStartNew(currentThread);
                        return currentState;
                    }
                    if (currentState.getNodes().masterNode() == null) {
                        ZenDiscovery.this.logger.debug("no master node is set, despite of join request completing. retrying pings.", new Object[0]);
                        ZenDiscovery.this.joinThreadControl.markThreadAsDoneAndStartNew(currentThread);
                        return currentState;
                    }
                    if (!currentState.getNodes().masterNode().equals(finalMasterNode)) {
                        return ZenDiscovery.this.joinThreadControl.stopRunningThreadAndRejoin(currentState, "master_switched_while_finalizing_join");
                    }
                    ZenDiscovery.this.joinThreadControl.markThreadAsDone(currentThread);
                    return currentState;
                }

                @Override
                public void onFailure(String source, @Nullable Throwable t) {
                    ZenDiscovery.this.logger.error("unexpected error while trying to finalize cluster join", t, new Object[0]);
                    ZenDiscovery.this.joinThreadControl.markThreadAsDoneAndStartNew(currentThread);
                }
            });
        }
    }

    private boolean joinElectedMaster(DiscoveryNode masterNode) {
        try {
            this.transportService.connectToNode(masterNode);
        }
        catch (Exception e) {
            this.logger.warn("failed to connect to master [{}], retrying...", e, masterNode);
            return false;
        }
        int joinAttempt = 0;
        while (true) {
            try {
                this.logger.trace("joining master {}", masterNode);
                this.membership.sendJoinRequestBlocking(masterNode, this.clusterService.localNode(), this.joinTimeout);
                return true;
            }
            catch (Throwable t) {
                Throwable unwrap = ExceptionsHelper.unwrapCause(t);
                if (unwrap instanceof ElasticsearchIllegalStateException && unwrap.getMessage().contains("not master for join request")) {
                    if (++joinAttempt == this.joinRetryAttempts) {
                        this.logger.info("failed to send join request to master [{}], reason [{}], tried [{}] times", masterNode, ExceptionsHelper.detailedMessage(t), joinAttempt);
                        return false;
                    }
                } else {
                    if (this.logger.isTraceEnabled()) {
                        this.logger.trace("failed to send join request to master [{}]", t, masterNode);
                    } else {
                        this.logger.info("failed to send join request to master [{}], reason [{}]", masterNode, ExceptionsHelper.detailedMessage(t));
                    }
                    return false;
                }
                this.logger.trace("master {} failed with [{}]. retrying... (attempts done: [{}])", masterNode, ExceptionsHelper.detailedMessage(t), joinAttempt);
                try {
                    Thread.sleep(this.joinRetryDelay.millis());
                    continue;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    continue;
                }
            }
            break;
        }
    }

    private void handleLeaveRequest(final DiscoveryNode node) {
        if (this.lifecycleState() != Lifecycle.State.STARTED) {
            return;
        }
        if (this.localNodeMaster()) {
            this.clusterService.submitStateUpdateTask("zen-disco-node_left(" + node + ")", Priority.IMMEDIATE, new ClusterStateUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) {
                    DiscoveryNodes.Builder builder = DiscoveryNodes.builder(currentState.nodes()).remove(node.id());
                    currentState = ClusterState.builder(currentState).nodes(builder).build();
                    if (!ZenDiscovery.this.electMaster.hasEnoughMasterNodes(currentState.nodes())) {
                        return ZenDiscovery.this.rejoin(currentState, "not enough master nodes");
                    }
                    RoutingAllocation.Result routingResult = ZenDiscovery.this.routingService.getAllocationService().reroute(ClusterState.builder(currentState).build());
                    return ClusterState.builder(currentState).routingResult(routingResult).build();
                }

                @Override
                public void onNoLongerMaster(String source) {
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    ZenDiscovery.this.logger.error("unexpected failure during [{}]", t, source);
                }
            });
        } else if (node.equals(this.nodes().masterNode())) {
            this.handleMasterGone(node, "shut_down");
        }
    }

    private void handleNodeFailure(final DiscoveryNode node, String reason) {
        if (this.lifecycleState() != Lifecycle.State.STARTED) {
            return;
        }
        if (!this.localNodeMaster()) {
            return;
        }
        this.clusterService.submitStateUpdateTask("zen-disco-node_failed(" + node + "), reason " + reason, Priority.IMMEDIATE, new ProcessedClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                if (currentState.nodes().get(node.id()) == null) {
                    ZenDiscovery.this.logger.debug("node [{}] already removed from cluster state. ignoring.", node);
                    return currentState;
                }
                DiscoveryNodes.Builder builder = DiscoveryNodes.builder(currentState.nodes()).remove(node.id());
                currentState = ClusterState.builder(currentState).nodes(builder).build();
                if (!ZenDiscovery.this.electMaster.hasEnoughMasterNodes(currentState.nodes())) {
                    return ZenDiscovery.this.rejoin(currentState, "not enough master nodes");
                }
                RoutingAllocation.Result routingResult = ZenDiscovery.this.routingService.getAllocationService().reroute(ClusterState.builder(currentState).build());
                return ClusterState.builder(currentState).routingResult(routingResult).build();
            }

            @Override
            public void onNoLongerMaster(String source) {
            }

            @Override
            public void onFailure(String source, Throwable t) {
                ZenDiscovery.this.logger.error("unexpected failure during [{}]", t, source);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                ZenDiscovery.this.sendInitialStateEventIfNeeded();
            }
        });
    }

    private void handleMinimumMasterNodesChanged(final int minimumMasterNodes) {
        if (this.lifecycleState() != Lifecycle.State.STARTED) {
            return;
        }
        final int prevMinimumMasterNode = this.electMaster.minimumMasterNodes();
        this.electMaster.minimumMasterNodes(minimumMasterNodes);
        if (!this.localNodeMaster()) {
            return;
        }
        this.clusterService.submitStateUpdateTask("zen-disco-minimum_master_nodes_changed", Priority.IMMEDIATE, new ProcessedClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                if (!ZenDiscovery.this.electMaster.hasEnoughMasterNodes(currentState.nodes())) {
                    return ZenDiscovery.this.rejoin(currentState, "not enough master nodes on change of minimum_master_nodes from [" + prevMinimumMasterNode + "] to [" + minimumMasterNodes + "]");
                }
                return currentState;
            }

            @Override
            public void onNoLongerMaster(String source) {
            }

            @Override
            public void onFailure(String source, Throwable t) {
                ZenDiscovery.this.logger.error("unexpected failure during [{}]", t, source);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                ZenDiscovery.this.sendInitialStateEventIfNeeded();
            }
        });
    }

    private void handleMasterGone(final DiscoveryNode masterNode, final String reason) {
        if (this.lifecycleState() != Lifecycle.State.STARTED) {
            return;
        }
        if (this.localNodeMaster()) {
            return;
        }
        this.logger.info("master_left [{}], reason [{}]", masterNode, reason);
        this.clusterService.submitStateUpdateTask("zen-disco-master_failed (" + masterNode + ")", Priority.IMMEDIATE, new ProcessedClusterStateNonMasterUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                if (!masterNode.id().equals(currentState.nodes().masterNodeId())) {
                    return currentState;
                }
                DiscoveryNodes discoveryNodes = DiscoveryNodes.builder(currentState.nodes()).remove(masterNode.id()).masterNodeId(null).build();
                ArrayList pendingNewClusterStates = new ArrayList();
                ZenDiscovery.this.processNewClusterStates.drainTo(pendingNewClusterStates);
                ZenDiscovery.this.logger.trace("removed [{}] pending cluster states", pendingNewClusterStates.size());
                boolean rejoin = true;
                if (!ZenDiscovery.this.rejoinOnMasterGone) {
                    ZenDiscovery.this.logger.debug("not rejoining cluster due to rejoinOnMasterGone [{}]", ZenDiscovery.this.rejoinOnMasterGone);
                    rejoin = false;
                } else if (discoveryNodes.smallestNonClientNodeVersion().before(Version.V_1_4_0_Beta1)) {
                    ZenDiscovery.this.logger.debug("not rejoining cluster due a minimum node version of [{}]", discoveryNodes.smallestNonClientNodeVersion());
                    rejoin = false;
                }
                if (rejoin) {
                    return ZenDiscovery.this.rejoin(ClusterState.builder(currentState).nodes(discoveryNodes).build(), "master left (reason = " + reason + ")");
                }
                if (!ZenDiscovery.this.electMaster.hasEnoughMasterNodes(discoveryNodes)) {
                    return ZenDiscovery.this.rejoin(ClusterState.builder(currentState).nodes(discoveryNodes).build(), "not enough master nodes after master left (reason = " + reason + ")");
                }
                DiscoveryNode electedMaster = ZenDiscovery.this.electMaster.electMaster(discoveryNodes);
                DiscoveryNode localNode = currentState.nodes().localNode();
                if (localNode.equals(electedMaster)) {
                    ZenDiscovery.this.masterFD.stop("got elected as new master since master left (reason = " + reason + ")");
                    discoveryNodes = DiscoveryNodes.builder(discoveryNodes).masterNodeId(localNode.id()).build();
                    ClusterState newState = ClusterState.builder(currentState).nodes(discoveryNodes).build();
                    ZenDiscovery.this.nodesFD.updateNodesAndPing(newState);
                    return newState;
                }
                ZenDiscovery.this.nodesFD.stop();
                if (electedMaster != null) {
                    discoveryNodes = DiscoveryNodes.builder(discoveryNodes).masterNodeId(electedMaster.id()).build();
                    ZenDiscovery.this.masterFD.restart(electedMaster, "possible elected master since master left (reason = " + reason + ")");
                    return ClusterState.builder(currentState).nodes(discoveryNodes).build();
                }
                return ZenDiscovery.this.rejoin(ClusterState.builder(currentState).nodes(discoveryNodes).build(), "master_left and no other node elected to become master");
            }

            @Override
            public void onFailure(String source, Throwable t) {
                ZenDiscovery.this.logger.error("unexpected failure during [{}]", t, source);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                ZenDiscovery.this.sendInitialStateEventIfNeeded();
            }
        });
    }

    void handleNewClusterStateFromMaster(ClusterState newClusterState, final PublishClusterStateAction.NewClusterStateListener.NewStateProcessed newStateProcessed) {
        ClusterName incomingClusterName = newClusterState.getClusterName();
        if (incomingClusterName != null && !incomingClusterName.equals(this.clusterName)) {
            this.logger.warn("received cluster state from [{}] which is also master but with a different cluster name [{}]", newClusterState.nodes().masterNode(), incomingClusterName);
            newStateProcessed.onNewClusterStateFailed(new ElasticsearchIllegalStateException("received state from a node that is not part of the cluster"));
            return;
        }
        if (this.localNodeMaster()) {
            this.logger.debug("received cluster state from [{}] which is also master with cluster name [{}]", newClusterState.nodes().masterNode(), incomingClusterName);
            final ClusterState newState = newClusterState;
            this.clusterService.submitStateUpdateTask("zen-disco-master_receive_cluster_state_from_another_master [" + newState.nodes().masterNode() + "]", Priority.URGENT, new ProcessedClusterStateUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) {
                    return ZenDiscovery.this.handleAnotherMaster(currentState, newState.nodes().masterNode(), newState.version(), "via a new cluster state");
                }

                @Override
                public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState2) {
                    newStateProcessed.onNewClusterStateProcessed();
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    ZenDiscovery.this.logger.error("unexpected failure during [{}]", t, source);
                    newStateProcessed.onNewClusterStateFailed(t);
                }
            });
        } else if (newClusterState.nodes().localNode() == null) {
            this.logger.warn("received a cluster state from [{}] and not part of the cluster, should not happen", newClusterState.nodes().masterNode());
            newStateProcessed.onNewClusterStateFailed(new ElasticsearchIllegalStateException("received state from a node that is not part of the cluster"));
        } else {
            final ProcessClusterState processClusterState = new ProcessClusterState(newClusterState, newStateProcessed);
            this.processNewClusterStates.add(processClusterState);
            assert (newClusterState.nodes().masterNode() != null) : "received a cluster state without a master";
            assert (!newClusterState.blocks().hasGlobalBlock(this.discoverySettings.getNoMasterBlock())) : "received a cluster state with a master block";
            this.clusterService.submitStateUpdateTask("zen-disco-receive(from master [" + newClusterState.nodes().masterNode() + "])", Priority.URGENT, new ProcessedClusterStateNonMasterUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) {
                    if (processClusterState.processed) {
                        return currentState;
                    }
                    ClusterState updatedState = ZenDiscovery.selectNextStateToProcess(ZenDiscovery.this.processNewClusterStates);
                    if (updatedState == null) {
                        updatedState = currentState;
                    }
                    if (ZenDiscovery.shouldIgnoreOrRejectNewClusterState(ZenDiscovery.this.logger, currentState, updatedState)) {
                        return currentState;
                    }
                    if (ZenDiscovery.this.masterFD.masterNode() == null || !ZenDiscovery.this.masterFD.masterNode().equals(updatedState.nodes().masterNode())) {
                        ZenDiscovery.this.masterFD.restart(updatedState.nodes().masterNode(), "new cluster state received and we are monitoring the wrong master [" + ZenDiscovery.this.masterFD.masterNode() + "]");
                    }
                    if (currentState.blocks().hasGlobalBlock(ZenDiscovery.this.discoverySettings.getNoMasterBlock())) {
                        ZenDiscovery.this.logger.debug("got first state from fresh master [{}]", updatedState.nodes().masterNodeId());
                        long count = ZenDiscovery.this.clusterJoinsCounter.incrementAndGet();
                        ZenDiscovery.this.logger.trace("updated cluster join cluster to [{}]", count);
                        return updatedState;
                    }
                    ClusterState.Builder builder = ClusterState.builder(updatedState);
                    if (updatedState.routingTable().version() == currentState.routingTable().version()) {
                        builder.routingTable(currentState.routingTable());
                    }
                    if (updatedState.metaData().version() == currentState.metaData().version()) {
                        builder.metaData(currentState.metaData());
                    } else {
                        MetaData.Builder metaDataBuilder = MetaData.builder(updatedState.metaData()).removeAllIndices();
                        for (IndexMetaData indexMetaData : updatedState.metaData()) {
                            IndexMetaData currentIndexMetaData = currentState.metaData().index(indexMetaData.index());
                            if (currentIndexMetaData != null && currentIndexMetaData.isSameUUID(indexMetaData.uuid()) && currentIndexMetaData.version() == indexMetaData.version()) {
                                metaDataBuilder.put(currentIndexMetaData, false);
                                continue;
                            }
                            metaDataBuilder.put(indexMetaData, false);
                        }
                        builder.metaData(metaDataBuilder);
                    }
                    return builder.build();
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    ZenDiscovery.this.logger.error("unexpected failure during [{}]", t, source);
                    newStateProcessed.onNewClusterStateFailed(t);
                }

                @Override
                public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                    ZenDiscovery.this.sendInitialStateEventIfNeeded();
                    newStateProcessed.onNewClusterStateProcessed();
                }
            });
        }
    }

    static ClusterState selectNextStateToProcess(Queue<ProcessClusterState> processNewClusterStates) {
        ProcessClusterState potentialState;
        ProcessClusterState stateToProcess = processNewClusterStates.poll();
        if (stateToProcess == null) {
            return null;
        }
        stateToProcess.processed = true;
        while ((potentialState = processNewClusterStates.peek()) != null && Objects.equal(stateToProcess.clusterState.nodes().masterNodeId(), potentialState.clusterState.nodes().masterNodeId()) && (potentialState = processNewClusterStates.poll()) != null) {
            potentialState.processed = true;
            if (potentialState.clusterState.version() <= stateToProcess.clusterState.version()) continue;
            stateToProcess = potentialState;
        }
        return stateToProcess.clusterState;
    }

    static boolean shouldIgnoreOrRejectNewClusterState(ESLogger logger, ClusterState currentState, ClusterState newClusterState) {
        if (currentState.nodes().masterNodeId() == null) {
            return false;
        }
        if (!currentState.nodes().masterNodeId().equals(newClusterState.nodes().masterNodeId())) {
            logger.warn("received a cluster state from a different master then the current one, rejecting (received {}, current {})", newClusterState.nodes().masterNode(), currentState.nodes().masterNode());
            throw new ElasticsearchIllegalStateException("cluster state from a different master then the current one, rejecting (received " + newClusterState.nodes().masterNode() + ", current " + currentState.nodes().masterNode() + ")");
        }
        if (newClusterState.version() < currentState.version()) {
            logger.debug("received a cluster state that has a lower version than the current one, ignoring (received {}, current {})", newClusterState.version(), currentState.version());
            return true;
        }
        return false;
    }

    private void handleJoinRequest(final DiscoveryNode node, MembershipAction.JoinCallback callback) {
        if (!this.transportService.addressSupported(node.address().getClass())) {
            this.logger.warn("received a wrong address type from [{}], ignoring...", node);
        } else {
            this.transportService.connectToNode(node);
            this.membership.sendValidateJoinRequestBlocking(node, this.joinTimeout);
            this.processJoinRequests.add(new Tuple<DiscoveryNode, MembershipAction.JoinCallback>(node, callback));
            this.clusterService.submitStateUpdateTask("zen-disco-receive(join from node[" + node + "])", Priority.URGENT, new ProcessedClusterStateUpdateTask(){
                private final List<Tuple<DiscoveryNode, MembershipAction.JoinCallback>> drainedJoinRequests = new ArrayList<Tuple<DiscoveryNode, MembershipAction.JoinCallback>>();
                private boolean nodeAdded = false;

                @Override
                public ClusterState execute(ClusterState currentState) {
                    ZenDiscovery.this.processJoinRequests.drainTo(this.drainedJoinRequests);
                    if (this.drainedJoinRequests.isEmpty()) {
                        return currentState;
                    }
                    DiscoveryNodes.Builder nodesBuilder = DiscoveryNodes.builder(currentState.nodes());
                    for (Tuple<DiscoveryNode, MembershipAction.JoinCallback> task : this.drainedJoinRequests) {
                        DiscoveryNode node2 = task.v1();
                        if (currentState.nodes().nodeExists(node2.id())) {
                            ZenDiscovery.this.logger.debug("received a join request for an existing node [{}]", node2);
                            continue;
                        }
                        this.nodeAdded = true;
                        nodesBuilder.put(node2);
                        for (DiscoveryNode existingNode : currentState.nodes()) {
                            if (!node2.address().equals(existingNode.address())) continue;
                            nodesBuilder.remove(existingNode.id());
                            ZenDiscovery.this.logger.warn("received join request from node [{}], but found existing node {} with same address, removing existing node", node2, existingNode);
                        }
                    }
                    ClusterState.Builder newState = ClusterState.builder(currentState);
                    if (this.nodeAdded) {
                        newState.nodes(nodesBuilder);
                    }
                    return newState.build();
                }

                @Override
                public void onNoLongerMaster(String source) {
                    ZenDiscovery.this.processJoinRequests.drainTo(this.drainedJoinRequests);
                    ElasticsearchIllegalStateException e = new ElasticsearchIllegalStateException("Node [" + ZenDiscovery.this.clusterService.localNode() + "] not master for join request from [" + node + "]");
                    this.innerOnFailure(e);
                }

                void innerOnFailure(Throwable t) {
                    for (Tuple<DiscoveryNode, MembershipAction.JoinCallback> drainedTask : this.drainedJoinRequests) {
                        try {
                            drainedTask.v2().onFailure(t);
                        }
                        catch (Exception e) {
                            ZenDiscovery.this.logger.error("error during task failure", e, new Object[0]);
                        }
                    }
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    ZenDiscovery.this.logger.error("unexpected failure during [{}]", t, source);
                    this.innerOnFailure(t);
                }

                @Override
                public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                    if (this.nodeAdded) {
                        ZenDiscovery.this.routingService.reroute("post_node_add");
                    }
                    for (Tuple<DiscoveryNode, MembershipAction.JoinCallback> drainedTask : this.drainedJoinRequests) {
                        try {
                            drainedTask.v2().onSuccess();
                        }
                        catch (Exception e) {
                            ZenDiscovery.this.logger.error("unexpected error during [{}]", e, source);
                        }
                    }
                }
            });
        }
    }

    private DiscoveryNode findMaster() {
        this.logger.trace("starting to ping", new Object[0]);
        ZenPing.PingResponse[] fullPingResponses = this.pingService.pingAndWait(this.pingTimeout);
        if (fullPingResponses == null) {
            this.logger.trace("No full ping responses", new Object[0]);
            return null;
        }
        if (this.logger.isTraceEnabled()) {
            StringBuilder sb = new StringBuilder("full ping responses:");
            if (fullPingResponses.length == 0) {
                sb.append(" {none}");
            } else {
                for (ZenPing.PingResponse pingResponse : fullPingResponses) {
                    sb.append("\n\t--> ").append(pingResponse);
                }
            }
            this.logger.trace(sb.toString(), new Object[0]);
        }
        ArrayList<ZenPing.PingResponse> pingResponses = Lists.newArrayList();
        for (ZenPing.PingResponse pingResponse : fullPingResponses) {
            DiscoveryNode node = pingResponse.node();
            if (this.masterElectionFilterClientNodes && (node.clientNode() || !node.masterNode() && !node.dataNode()) || this.masterElectionFilterDataNodes && !node.masterNode() && node.dataNode()) continue;
            pingResponses.add(pingResponse);
        }
        if (this.logger.isDebugEnabled()) {
            StringBuilder sb = new StringBuilder("filtered ping responses: (filter_client[").append(this.masterElectionFilterClientNodes).append("], filter_data[").append(this.masterElectionFilterDataNodes).append("])");
            if (pingResponses.isEmpty()) {
                sb.append(" {none}");
            } else {
                for (ZenPing.PingResponse pingResponse : pingResponses) {
                    sb.append("\n\t--> ").append(pingResponse);
                }
            }
            this.logger.debug(sb.toString(), new Object[0]);
        }
        DiscoveryNode localNode = this.clusterService.localNode();
        ArrayList<DiscoveryNode> pingMasters = Lists.newArrayList();
        for (ZenPing.PingResponse pingResponse : pingResponses) {
            if (pingResponse.master() == null || localNode.equals(pingResponse.master())) continue;
            pingMasters.add(pingResponse.master());
        }
        HashSet<DiscoveryNode> activeNodes = Sets.newHashSet();
        HashSet<DiscoveryNode> joinedOnceActiveNodes = Sets.newHashSet();
        if (localNode.masterNode()) {
            activeNodes.add(localNode);
            long joinsCounter = this.clusterJoinsCounter.get();
            if (joinsCounter > 0L) {
                this.logger.trace("adding local node to the list of active nodes who has previously joined the cluster (joins counter is [{}})", joinsCounter);
                joinedOnceActiveNodes.add(localNode);
            }
        }
        Version minimumPingVersion = localNode.version();
        for (ZenPing.PingResponse pingResponse : pingResponses) {
            activeNodes.add(pingResponse.node());
            minimumPingVersion = Version.smallest(pingResponse.node().version(), minimumPingVersion);
            if (pingResponse.hasJoinedOnce() == null || !pingResponse.hasJoinedOnce().booleanValue()) continue;
            assert (pingResponse.node().getVersion().onOrAfter(Version.V_1_4_0_Beta1)) : "ping version [" + pingResponse.node().version() + "]< 1.4.0 while having hasJoinedOnce == true";
            joinedOnceActiveNodes.add(pingResponse.node());
        }
        if (minimumPingVersion.before(Version.V_1_4_0_Beta1)) {
            this.logger.trace("ignoring joined once flags in ping responses, minimum ping version [{}]", minimumPingVersion);
            joinedOnceActiveNodes.clear();
        }
        if (pingMasters.isEmpty()) {
            if (this.electMaster.hasEnoughMasterNodes(activeNodes)) {
                DiscoveryNode master = this.electMaster.electMaster(joinedOnceActiveNodes);
                if (master != null) {
                    return master;
                }
                return this.electMaster.electMaster(activeNodes);
            }
            this.logger.trace("not enough master nodes [{}]", activeNodes);
            return null;
        }
        assert (!pingMasters.contains(localNode)) : "local node should never be elected as master when other nodes indicate an active master";
        return this.electMaster.electMaster(pingMasters);
    }

    private ClusterState rejoin(ClusterState clusterState, String reason) {
        assert (Thread.currentThread().getName().contains("clusterService#updateTask"));
        this.logger.warn(reason + ", current nodes: {}", clusterState.nodes());
        this.nodesFD.stop();
        this.masterFD.stop(reason);
        ClusterBlocks clusterBlocks = ClusterBlocks.builder().blocks(clusterState.blocks()).addGlobalBlock(this.discoverySettings.getNoMasterBlock()).build();
        DiscoveryNodes discoveryNodes = new DiscoveryNodes.Builder(clusterState.nodes()).masterNodeId(null).build();
        this.joinThreadControl.startNewThreadIfNotRunning();
        return ClusterState.builder(clusterState).blocks(clusterBlocks).nodes(discoveryNodes).build();
    }

    private boolean localNodeMaster() {
        return this.nodes().localNodeMaster();
    }

    private ClusterState handleAnotherMaster(ClusterState localClusterState, final DiscoveryNode otherMaster, long otherClusterStateVersion, String reason) {
        assert (localClusterState.nodes().localNodeMaster()) : "handleAnotherMaster called but current node is not a master";
        assert (Thread.currentThread().getName().contains("clusterService#updateTask")) : "not called from the cluster state update thread";
        if (otherClusterStateVersion > localClusterState.version()) {
            return this.rejoin(localClusterState, "zen-disco-discovered another master with a new cluster_state [" + otherMaster + "][" + reason + "]");
        }
        this.logger.warn("discovered [{}] which is also master but with an older cluster_state, telling [{}] to rejoin the cluster ([{}])", otherMaster, otherMaster, reason);
        try {
            this.transportService.connectToNode(otherMaster);
            this.transportService.sendRequest(otherMaster, DISCOVERY_REJOIN_ACTION_NAME, new RejoinClusterRequest(localClusterState.nodes().localNodeId()), new EmptyTransportResponseHandler("same"){

                @Override
                public void handleException(TransportException exp) {
                    ZenDiscovery.this.logger.warn("failed to send rejoin request to [{}]", exp, otherMaster);
                }
            });
        }
        catch (Exception e) {
            this.logger.warn("failed to send rejoin request to [{}]", e, otherMaster);
        }
        return localClusterState;
    }

    private void sendInitialStateEventIfNeeded() {
        if (this.initialStateSent.compareAndSet(false, true)) {
            for (InitialStateDiscoveryListener listener : this.initialStateListeners) {
                listener.initialStateProcessed();
            }
        }
    }

    boolean isRejoinOnMasterGone() {
        return this.rejoinOnMasterGone;
    }

    private class JoinThreadControl {
        private final ThreadPool threadPool;
        private final AtomicBoolean running = new AtomicBoolean(false);
        private final AtomicReference<Thread> currentJoinThread = new AtomicReference();

        public JoinThreadControl(ThreadPool threadPool) {
            this.threadPool = threadPool;
        }

        public boolean joinThreadActive() {
            Thread currentThread = this.currentJoinThread.get();
            return this.running.get() && currentThread != null && currentThread.isAlive();
        }

        public boolean joinThreadActive(Thread joinThread) {
            return this.running.get() && joinThread.equals(this.currentJoinThread.get());
        }

        public ClusterState stopRunningThreadAndRejoin(ClusterState clusterState, String reason) {
            this.assertClusterStateThread();
            this.currentJoinThread.set(null);
            return ZenDiscovery.this.rejoin(clusterState, reason);
        }

        public void startNewThreadIfNotRunning() {
            this.assertClusterStateThread();
            if (this.joinThreadActive()) {
                return;
            }
            this.threadPool.generic().execute(new Runnable(){

                @Override
                public void run() {
                    Thread currentThread = Thread.currentThread();
                    if (!JoinThreadControl.this.currentJoinThread.compareAndSet(null, currentThread)) {
                        return;
                    }
                    while (JoinThreadControl.this.running.get() && JoinThreadControl.this.joinThreadActive(currentThread)) {
                        try {
                            ZenDiscovery.this.innerJoinCluster();
                            return;
                        }
                        catch (Exception e) {
                            ZenDiscovery.this.logger.error("unexpected error while joining cluster, trying again", e, new Object[0]);
                            assert (ExceptionsHelper.reThrowIfNotNull(e));
                        }
                    }
                }
            });
        }

        public void markThreadAsDoneAndStartNew(Thread joinThread) {
            this.assertClusterStateThread();
            if (!this.markThreadAsDone(joinThread)) {
                return;
            }
            this.startNewThreadIfNotRunning();
        }

        public boolean markThreadAsDone(Thread joinThread) {
            this.assertClusterStateThread();
            return this.currentJoinThread.compareAndSet(joinThread, null);
        }

        public void stop() {
            this.running.set(false);
            Thread joinThread = this.currentJoinThread.getAndSet(null);
            if (joinThread != null) {
                joinThread.interrupt();
            }
        }

        public void start() {
            this.running.set(true);
        }

        private void assertClusterStateThread() {
            assert (Thread.currentThread().getName().contains("clusterService#updateTask")) : "not called from the cluster state update thread";
        }
    }

    class ApplySettings
    implements NodeSettingsService.Listener {
        ApplySettings() {
        }

        @Override
        public void onRefreshSettings(Settings settings) {
            boolean rejoinOnMasterGone;
            int minimumMasterNodes = settings.getAsInt("discovery.zen.minimum_master_nodes", (Integer)ZenDiscovery.this.electMaster.minimumMasterNodes());
            if (minimumMasterNodes != ZenDiscovery.this.electMaster.minimumMasterNodes()) {
                ZenDiscovery.this.logger.info("updating {} from [{}] to [{}]", "discovery.zen.minimum_master_nodes", ZenDiscovery.this.electMaster.minimumMasterNodes(), minimumMasterNodes);
                ZenDiscovery.this.handleMinimumMasterNodesChanged(minimumMasterNodes);
            }
            if ((rejoinOnMasterGone = settings.getAsBoolean(ZenDiscovery.SETTING_REJOIN_ON_MASTER_GONE, (Boolean)ZenDiscovery.this.rejoinOnMasterGone).booleanValue()) != ZenDiscovery.this.rejoinOnMasterGone) {
                ZenDiscovery.this.logger.info("updating {} from [{}] to [{}]", ZenDiscovery.SETTING_REJOIN_ON_MASTER_GONE, ZenDiscovery.this.rejoinOnMasterGone, rejoinOnMasterGone);
                ZenDiscovery.this.rejoinOnMasterGone = rejoinOnMasterGone;
            }
        }
    }

    class RejoinClusterRequestHandler
    extends BaseTransportRequestHandler<RejoinClusterRequest> {
        RejoinClusterRequestHandler() {
        }

        @Override
        public RejoinClusterRequest newInstance() {
            return new RejoinClusterRequest();
        }

        @Override
        public void messageReceived(final RejoinClusterRequest request, final TransportChannel channel) throws Exception {
            ZenDiscovery.this.clusterService.submitStateUpdateTask("received a request to rejoin the cluster from [" + request.fromNodeId + "]", Priority.IMMEDIATE, new ClusterStateNonMasterUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) {
                    try {
                        channel.sendResponse(TransportResponse.Empty.INSTANCE);
                    }
                    catch (Exception e) {
                        ZenDiscovery.this.logger.warn("failed to send response on rejoin cluster request handling", e, new Object[0]);
                    }
                    return ZenDiscovery.this.rejoin(currentState, "received a request to rejoin the cluster from [" + request.fromNodeId + "]");
                }

                @Override
                public void onNoLongerMaster(String source) {
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    ZenDiscovery.this.logger.error("unexpected failure during [{}]", t, source);
                }
            });
        }

        @Override
        public String executor() {
            return "same";
        }
    }

    static class RejoinClusterRequest
    extends TransportRequest {
        private String fromNodeId;

        RejoinClusterRequest(String fromNodeId) {
            this.fromNodeId = fromNodeId;
        }

        RejoinClusterRequest() {
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.fromNodeId = in.readOptionalString();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeOptionalString(this.fromNodeId);
        }
    }

    private class MasterNodeFailureListener
    implements MasterFaultDetection.Listener {
        private MasterNodeFailureListener() {
        }

        @Override
        public void onMasterFailure(DiscoveryNode masterNode, String reason) {
            ZenDiscovery.this.handleMasterGone(masterNode, reason);
        }
    }

    private class NodeFaultDetectionListener
    extends NodesFaultDetection.Listener {
        private final AtomicInteger pingsWhileMaster = new AtomicInteger(0);

        private NodeFaultDetectionListener() {
        }

        @Override
        public void onNodeFailure(DiscoveryNode node, String reason) {
            ZenDiscovery.this.handleNodeFailure(node, reason);
        }

        @Override
        public void onPingReceived(final NodesFaultDetection.PingRequest pingRequest) {
            if (!ZenDiscovery.this.localNodeMaster()) {
                this.pingsWhileMaster.set(0);
                return;
            }
            if (pingRequest.masterNode() == null) {
                return;
            }
            if (this.pingsWhileMaster.incrementAndGet() < ZenDiscovery.this.maxPingsFromAnotherMaster) {
                ZenDiscovery.this.logger.trace("got a ping from another master {}. current ping count: [{}]", pingRequest.masterNode(), this.pingsWhileMaster.get());
                return;
            }
            ZenDiscovery.this.logger.debug("got a ping from another master {}. resolving who should rejoin. current ping count: [{}]", pingRequest.masterNode(), this.pingsWhileMaster.get());
            ZenDiscovery.this.clusterService.submitStateUpdateTask("ping from another master", Priority.IMMEDIATE, new ClusterStateUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) throws Exception {
                    NodeFaultDetectionListener.this.pingsWhileMaster.set(0);
                    return ZenDiscovery.this.handleAnotherMaster(currentState, pingRequest.masterNode(), pingRequest.clusterStateVersion(), "node fd ping");
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    ZenDiscovery.this.logger.debug("unexpected error during cluster state update task after pings from another master", t, new Object[0]);
                }
            });
        }
    }

    private class MembershipListener
    implements MembershipAction.MembershipListener {
        private MembershipListener() {
        }

        @Override
        public void onJoin(DiscoveryNode node, MembershipAction.JoinCallback callback) {
            ZenDiscovery.this.handleJoinRequest(node, callback);
        }

        @Override
        public void onLeave(DiscoveryNode node) {
            ZenDiscovery.this.handleLeaveRequest(node);
        }
    }

    private class NewClusterStateListener
    implements PublishClusterStateAction.NewClusterStateListener {
        private NewClusterStateListener() {
        }

        @Override
        public void onNewClusterState(ClusterState clusterState, PublishClusterStateAction.NewClusterStateListener.NewStateProcessed newStateProcessed) {
            ZenDiscovery.this.handleNewClusterStateFromMaster(clusterState, newStateProcessed);
        }
    }

    static class ProcessClusterState {
        final ClusterState clusterState;
        final PublishClusterStateAction.NewClusterStateListener.NewStateProcessed newStateProcessed;
        volatile boolean processed;

        ProcessClusterState(ClusterState clusterState, PublishClusterStateAction.NewClusterStateListener.NewStateProcessed newStateProcessed) {
            this.clusterState = clusterState;
            this.newStateProcessed = newStateProcessed;
        }
    }
}

