@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.alicloud.bastionhost.kotlin

import com.pulumi.alicloud.bastionhost.HostAccountShareKeyAttachmentArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Provides a Bastion Host Account Share Key Attachment resource.
 * For information about Bastion Host Host Account Share Key Attachment and how to use it, see [What is Host Account Share Key Attachment](https://www.alibabacloud.com/help/en/bastion-host/latest/attachhostaccountstohostsharekey).
 * > **NOTE:** Available since v1.165.0.
 * ## Example Usage
 * Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as alicloud from "@pulumi/alicloud";
 * const config = new pulumi.Config();
 * const name = config.get("name") || "tf_example";
 * const default = alicloud.getZones({
 *     availableResourceCreation: "VSwitch",
 * });
 * const defaultGetNetworks = alicloud.vpc.getNetworks({
 *     nameRegex: "^default-NODELETING$",
 *     cidrBlock: "10.4.0.0/16",
 * });
 * const defaultGetSwitches = Promise.all([defaultGetNetworks, _default])&#46;then(([defaultGetNetworks, _default]) => alicloud.vpc.getSwitches({
 *     cidrBlock: "10.4.0.0/24",
 *     vpcId: defaultGetNetworks.ids?.[0],
 *     zoneId: _default.zones?.[0]?.id,
 * }));
 * const defaultSecurityGroup = new alicloud.ecs.SecurityGroup("default", {vpcId: defaultGetNetworks.then(defaultGetNetworks => defaultGetNetworks.ids?.[0])});
 * const defaultInstance = new alicloud.bastionhost.Instance("default", {
 *     description: name,
 *     licenseCode: "bhah_ent_50_asset",
 *     planCode: "cloudbastion",
 *     storage: "5",
 *     bandwidth: "5",
 *     period: 1,
 *     vswitchId: defaultGetSwitches.then(defaultGetSwitches => defaultGetSwitches.ids?.[0]),
 *     securityGroupIds: [defaultSecurityGroup&#46;id],
 * });
 * const defaultHost = new alicloud.bastionhost.Host("default", {
 *     instanceId: defaultInstance.id,
 *     hostName: name,
 *     activeAddressType: "Private",
 *     hostPrivateAddress: "172.16.0.10",
 *     osType: "Linux",
 *     source: "Local",
 * });
 * const defaultHostAccount = new alicloud.bastionhost.HostAccount("default", {
 *     hostAccountName: name,
 *     hostId: defaultHost.hostId,
 *     instanceId: defaultHost.instanceId,
 *     protocolName: "SSH",
 *     password: "YourPassword12345",
 * });
 * const privateKey = config.get("privateKey") || "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";
 * const defaultHostShareKey = new alicloud.bastionhost.HostShareKey("default", {
 *     hostShareKeyName: name,
 *     instanceId: defaultInstance.id,
 *     privateKey: privateKey,
 * });
 * const defaultHostAccountShareKeyAttachment = new alicloud.bastionhost.HostAccountShareKeyAttachment("default", {
 *     instanceId: defaultInstance.id,
 *     hostShareKeyId: defaultHostShareKey.hostShareKeyId,
 *     hostAccountId: defaultHostAccount.hostAccountId,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_alicloud as alicloud
 * config = pulumi.Config()
 * name = config.get("name")
 * if name is None:
 *     name = "tf_example"
 * default = alicloud.get_zones(available_resource_creation="VSwitch")
 * default_get_networks = alicloud.vpc.get_networks(name_regex="^default-NODELETING$",
 *     cidr_block="10.4.0.0/16")
 * default_get_switches = alicloud.vpc.get_switches(cidr_block="10.4.0.0/24",
 *     vpc_id=default_get_networks.ids[0],
 *     zone_id=default.zones[0].id)
 * default_security_group = alicloud.ecs.SecurityGroup("default", vpc_id=default_get_networks.ids[0])
 * default_instance = alicloud.bastionhost.Instance("default",
 *     description=name,
 *     license_code="bhah_ent_50_asset",
 *     plan_code="cloudbastion",
 *     storage="5",
 *     bandwidth="5",
 *     period=1,
 *     vswitch_id=default_get_switches.ids[0],
 *     security_group_ids=[default_security_group&#46;id])
 * default_host = alicloud.bastionhost.Host("default",
 *     instance_id=default_instance.id,
 *     host_name=name,
 *     active_address_type="Private",
 *     host_private_address="172.16.0.10",
 *     os_type="Linux",
 *     source="Local")
 * default_host_account = alicloud.bastionhost.HostAccount("default",
 *     host_account_name=name,
 *     host_id=default_host.host_id,
 *     instance_id=default_host.instance_id,
 *     protocol_name="SSH",
 *     password="YourPassword12345")
 * private_key = config.get("privateKey")
 * if private_key is None:
 *     private_key = "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"
 * default_host_share_key = alicloud.bastionhost.HostShareKey("default",
 *     host_share_key_name=name,
 *     instance_id=default_instance.id,
 *     private_key=private_key)
 * default_host_account_share_key_attachment = alicloud.bastionhost.HostAccountShareKeyAttachment("default",
 *     instance_id=default_instance.id,
 *     host_share_key_id=default_host_share_key.host_share_key_id,
 *     host_account_id=default_host_account.host_account_id)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AliCloud = Pulumi.AliCloud;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var name = config.Get("name") ?? "tf_example";
 *     var @default = AliCloud.GetZones.Invoke(new()
 *     {
 *         AvailableResourceCreation = "VSwitch",
 *     });
 *     var defaultGetNetworks = AliCloud.Vpc.GetNetworks.Invoke(new()
 *     {
 *         NameRegex = "^default-NODELETING$",
 *         CidrBlock = "10.4.0.0/16",
 *     });
 *     var defaultGetSwitches = AliCloud.Vpc.GetSwitches.Invoke(new()
 *     {
 *         CidrBlock = "10.4.0.0/24",
 *         VpcId = defaultGetNetworks.Apply(getNetworksResult => getNetworksResult.Ids[0]),
 *         ZoneId = @default.Apply(getZonesResult => getZonesResult.Zones[0]?.Id),
 *     });
 *     var defaultSecurityGroup = new AliCloud.Ecs.SecurityGroup("default", new()
 *     {
 *         VpcId = defaultGetNetworks.Apply(getNetworksResult => getNetworksResult.Ids[0]),
 *     });
 *     var defaultInstance = new AliCloud.BastionHost.Instance("default", new()
 *     {
 *         Description = name,
 *         LicenseCode = "bhah_ent_50_asset",
 *         PlanCode = "cloudbastion",
 *         Storage = "5",
 *         Bandwidth = "5",
 *         Period = 1,
 *         VswitchId = defaultGetSwitches.Apply(getSwitchesResult => getSwitchesResult.Ids[0]),
 *         SecurityGroupIds = new[]
 *         {
 *             defaultSecurityGroup.Id,
 *         },
 *     });
 *     var defaultHost = new AliCloud.BastionHost.Host("default", new()
 *     {
 *         InstanceId = defaultInstance.Id,
 *         HostName = name,
 *         ActiveAddressType = "Private",
 *         HostPrivateAddress = "172.16.0.10",
 *         OsType = "Linux",
 *         Source = "Local",
 *     });
 *     var defaultHostAccount = new AliCloud.BastionHost.HostAccount("default", new()
 *     {
 *         HostAccountName = name,
 *         HostId = defaultHost.HostId,
 *         InstanceId = defaultHost.InstanceId,
 *         ProtocolName = "SSH",
 *         Password = "YourPassword12345",
 *     });
 *     var privateKey = config.Get("privateKey") ?? "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";
 *     var defaultHostShareKey = new AliCloud.BastionHost.HostShareKey("default", new()
 *     {
 *         HostShareKeyName = name,
 *         InstanceId = defaultInstance.Id,
 *         PrivateKey = privateKey,
 *     });
 *     var defaultHostAccountShareKeyAttachment = new AliCloud.BastionHost.HostAccountShareKeyAttachment("default", new()
 *     {
 *         InstanceId = defaultInstance.Id,
 *         HostShareKeyId = defaultHostShareKey.HostShareKeyId,
 *         HostAccountId = defaultHostAccount.HostAccountId,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/bastionhost"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/ecs"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/vpc"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		cfg := config.New(ctx, "")
 * 		name := "tf_example"
 * 		if param := cfg.Get("name"); param != "" {
 * 			name = param
 * 		}
 * 		_default, err := alicloud.GetZones(ctx, &alicloud.GetZonesArgs{
 * 			AvailableResourceCreation: pulumi.StringRef("VSwitch"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultGetNetworks, err := vpc.GetNetworks(ctx, &vpc.GetNetworksArgs{
 * 			NameRegex: pulumi.StringRef("^default-NODELETING$"),
 * 			CidrBlock: pulumi.StringRef("10.4.0.0/16"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultGetSwitches, err := vpc.GetSwitches(ctx, &vpc.GetSwitchesArgs{
 * 			CidrBlock: pulumi.StringRef("10.4.0.0/24"),
 * 			VpcId:     pulumi.StringRef(defaultGetNetworks.Ids[0]),
 * 			ZoneId:    pulumi.StringRef(_default.Zones[0].Id),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultSecurityGroup, err := ecs.NewSecurityGroup(ctx, "default", &ecs.SecurityGroupArgs{
 * 			VpcId: pulumi.String(defaultGetNetworks.Ids[0]),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultInstance, err := bastionhost.NewInstance(ctx, "default", &bastionhost.InstanceArgs{
 * 			Description: pulumi.String(name),
 * 			LicenseCode: pulumi.String("bhah_ent_50_asset"),
 * 			PlanCode:    pulumi.String("cloudbastion"),
 * 			Storage:     pulumi.String("5"),
 * 			Bandwidth:   pulumi.String("5"),
 * 			Period:      pulumi.Int(1),
 * 			VswitchId:   pulumi.String(defaultGetSwitches.Ids[0]),
 * 			SecurityGroupIds: pulumi.StringArray{
 * 				defaultSecurityGroup.ID(),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultHost, err := bastionhost.NewHost(ctx, "default", &bastionhost.HostArgs{
 * 			InstanceId:         defaultInstance.ID(),
 * 			HostName:           pulumi.String(name),
 * 			ActiveAddressType:  pulumi.String("Private"),
 * 			HostPrivateAddress: pulumi.String("172.16.0.10"),
 * 			OsType:             pulumi.String("Linux"),
 * 			Source:             pulumi.String("Local"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultHostAccount, err := bastionhost.NewHostAccount(ctx, "default", &bastionhost.HostAccountArgs{
 * 			HostAccountName: pulumi.String(name),
 * 			HostId:          defaultHost.HostId,
 * 			InstanceId:      defaultHost.InstanceId,
 * 			ProtocolName:    pulumi.String("SSH"),
 * 			Password:        pulumi.String("YourPassword12345"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		privateKey := "LS0tLS1CRUdJTiBSU0EgUFJJVkFURSBLRVktLS0tLQpNSUlFcEFJQkFBS0NBUUVBc25oc29SSVVwVXltSG1FVHJXUGxDbkhMa3c3N0JYTm44ZHcvbDg3eG10SUhjd2syCkRybjFDZk5jZkpJV0tSdkFaYkdKMlZTS1RiRDhPTmcyT3JvUHFGUHBLOHJ5QjJRb1NYQVRsaUVHWFhNeW1tRm8KeDBmem12THFscUxpNGZnOExhcTc5UC85aGxLU1djTWhJU0pYVTNHMS9KdEFBUmEyQXc4cXEzSVQvMkZ5NktrdwowMU9MdDdLN2pGUFRPaHhtdmNoSkZ1SVo1YXI0cW5HUlFHQnpCL2hoRHVIWEMwRlhJZ2ozd0NXMDZ4R2V2WjJyCmNCWWwwN1luL2lvZk95MU0wRjZZN0JrMU95N21vYndzM1JsalUyL2FpZlhLMmNOUlk2Qjl5WXNvd1RBZmQ5OTQKQ2YxSlF3TlhsaUZCeTZueEJLQk1YbDhIU1grS1o3L29PUlIwVXdJREFRQUJBb0lCQVFDbU5JSXR5ckhSY3oxdApJMGo0L0FQc295ZE1EL0owRkJMa2FoSUxKWjFaYW1tbmx4ZHh4WHBQUndXRnVXTEw2OTFVbDI5aUoxb1ptazU1Ci9ka2EvZlhnOUN3OUxXWVN2aExLdVlaMEZOTmhxZ3VoUEVBZy9uLytlR1ZCM2ZYZkxaZVZpK0E0L1VHMG15ZFMKVXVlQ2ZRSElZeWh4VkgvWnc3WER5WmNhVFVZVVdMUWlYcVN0Y2JRbnZFOXpwOGc5TWh5UkhBcWYwbEt2UTRqdwphUnNKTnlob3lhZWcvUXlFeHVYNGdBR1lIc1lTSDRFVmtXOHl5WE1aOHpRdk1OSUNiYXhmUkRuSngybUh6a09rCnFHczVXbFp5L3VrQk5jWTQwd2Y0eTY2bEVJaVpKbiswaFhtSTF4Tk5SdHRwMjZnY3ROOXZWbmVicTdLTnpjTDgKeFQrMXZJaEpBb0dCQU9iMVM1YlE4NVRFWDBoZTRmdXc2R3ExbnhRLzJUSU03emZhK2VhZThPQlh2eVNFV3JpdwpPZzM3RFhVUDFNVU1iTEJRenE0STl1dE5YSVZadEFLR0h6ZDR6WmtQeGxORjZPN0FyWnJEWElDNEdKZHdmSEhxCjJZcDkxUDlWekJlOVhkTVdZVGFCNkMzWVdtYzQwM08vYWdyRCtNb2JnL0hqMSt0d2xZR2hjdlV2QW9HQkFNWFMKT2VnWHc5VUF3VEZabFBtZzZKeDI3TzNXUFBHd1E3QzRnYitFZzZkR1pLRnJVR1ZId2VUUG1HaGtwN1BmYU5ESwplaFVoUWFnNm9XOTF4dkE2YldZZ29SQmczUWkxc01MblRWeTExeVN1UEVFSCtMT2s1N3d2akNLSk5XZnM0SjVyCmg1NGw0QXZ6UVhyWWN0UlZkSmYrNjFacGFnTkdZMVBvWVJMTHJMSWRBb0dBTndydzErRzJtNWJ0YW04S2hwU1QKMzVLbmRnajlkM3N6cStrcE03aGZpZWYvcXZGTU9jWHVJQlRjRVRFVHNWNlRyTFdsZkQ2d3NrVitybDFCbEhSbwpqaXpoT3dCU2NOZ3hlbTA3TXE0cXBwYTViYVltVW5QNUlwTjRwdDNJeFVPaFQ4UitxS0h2TnJYZ1hjZGlSYXl4CjFoejhkeFoxckxselpTNHd3M001MVlzQ2dZRUFpUDEwTEUySXg5Q2wrTTdZWTZZU2I0Zkx1MGhKRy9XOGFuemIKSFExZlBrOTVFRytJVlJyRUl2ZS95MHNvOTE4VzdyL0lteWxVbG5ORHFEUWZkK3grSmVNaXBuenRsRUorRGZxdgprQ3c4dUtJUUI5akZXV0l4T0JpVktyVnB6bll6ZG9Gd2dRd3BneDBKazFDZzlIblpMQWpVWUJyUDEwUy9ORFFRClJUdldjK0VDZ1lBeGRIZWxQNG1RdkJaS1oxMlNKbHlLbFVLeW43aHhzSHVkMkphMVNtS3FWeHBERDNlR0w0Y3QKZXA1QTZ5NkF4eGViZkI0aDdYNEZ0QTBBRURPdkZDR0J1QlRvZ3ZBdUNDVUtqK2JIUG1SNG53UVYzcWZ2M3loRAp0TGkwU2FHVElta2wvbUNCUDhZaW9JUys2N0xjby9kbHphUTNGVDlxTnJieFdFWjJlaS9LVlE9PQotLS0tLUVORCBSU0EgUFJJVkFURSBLRVktLS0tLQ=="
 * 		if param := cfg.Get("privateKey"); param != "" {
 * 			privateKey = param
 * 		}
 * 		defaultHostShareKey, err := bastionhost.NewHostShareKey(ctx, "default", &bastionhost.HostShareKeyArgs{
 * 			HostShareKeyName: pulumi.String(name),
 * 			InstanceId:       defaultInstance.ID(),
 * 			PrivateKey:       pulumi.String(privateKey),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = bastionhost.NewHostAccountShareKeyAttachment(ctx, "default", &bastionhost.HostAccountShareKeyAttachmentArgs{
 * 			InstanceId:     defaultInstance.ID(),
 * 			HostShareKeyId: defaultHostShareKey.HostShareKeyId,
 * 			HostAccountId:  defaultHostAccount.HostAccountId,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.alicloud.AlicloudFunctions;
 * import com.pulumi.alicloud.inputs.GetZonesArgs;
 * import com.pulumi.alicloud.vpc.VpcFunctions;
 * import com.pulumi.alicloud.vpc.inputs.GetNetworksArgs;
 * import com.pulumi.alicloud.vpc.inputs.GetSwitchesArgs;
 * import com.pulumi.alicloud.ecs.SecurityGroup;
 * import com.pulumi.alicloud.ecs.SecurityGroupArgs;
 * import com.pulumi.alicloud.bastionhost.Instance;
 * import com.pulumi.alicloud.bastionhost.InstanceArgs;
 * import com.pulumi.alicloud.bastionhost.Host;
 * import com.pulumi.alicloud.bastionhost.HostArgs;
 * import com.pulumi.alicloud.bastionhost.HostAccount;
 * import com.pulumi.alicloud.bastionhost.HostAccountArgs;
 * import com.pulumi.alicloud.bastionhost.HostShareKey;
 * import com.pulumi.alicloud.bastionhost.HostShareKeyArgs;
 * import com.pulumi.alicloud.bastionhost.HostAccountShareKeyAttachment;
 * import com.pulumi.alicloud.bastionhost.HostAccountShareKeyAttachmentArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var name = config.get("name").orElse("tf_example");
 *         final var default = AlicloudFunctions.getZones(GetZonesArgs.builder()
 *             .availableResourceCreation("VSwitch")
 *             .build());
 *         final var defaultGetNetworks = VpcFunctions.getNetworks(GetNetworksArgs.builder()
 *             .nameRegex("^default-NODELETING$")
 *             .cidrBlock("10.4.0.0/16")
 *             .build());
 *         final var defaultGetSwitches = VpcFunctions.getSwitches(GetSwitchesArgs.builder()
 *             .cidrBlock("10.4.0.0/24")
 *             .vpcId(defaultGetNetworks.applyValue(getNetworksResult -> getNetworksResult.ids()[0]))
 *             .zoneId(default_.zones()[0].id())
 *             .build());
 *         var defaultSecurityGroup = new SecurityGroup("defaultSecurityGroup", SecurityGroupArgs.builder()
 *             .vpcId(defaultGetNetworks.applyValue(getNetworksResult -> getNetworksResult.ids()[0]))
 *             .build());
 *         var defaultInstance = new Instance("defaultInstance", InstanceArgs.builder()
 *             .description(name)
 *             .licenseCode("bhah_ent_50_asset")
 *             .planCode("cloudbastion")
 *             .storage("5")
 *             .bandwidth("5")
 *             .period("1")
 *             .vswitchId(defaultGetSwitches.applyValue(getSwitchesResult -> getSwitchesResult.ids()[0]))
 *             .securityGroupIds(defaultSecurityGroup.id())
 *             .build());
 *         var defaultHost = new Host("defaultHost", HostArgs.builder()
 *             .instanceId(defaultInstance.id())
 *             .hostName(name)
 *             .activeAddressType("Private")
 *             .hostPrivateAddress("172.16.0.10")
 *             .osType("Linux")
 *             .source("Local")
 *             .build());
 *         var defaultHostAccount = new HostAccount("defaultHostAccount", HostAccountArgs.builder()
 *             .hostAccountName(name)
 *             .hostId(defaultHost.hostId())
 *             .instanceId(defaultHost.instanceId())
 *             .protocolName("SSH")
 *             .password("YourPassword12345")
 *             .build());
 *         final var privateKey = config.get("privateKey").orElse("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");
 *         var defaultHostShareKey = new HostShareKey("defaultHostShareKey", HostShareKeyArgs.builder()
 *             .hostShareKeyName(name)
 *             .instanceId(defaultInstance.id())
 *             .privateKey(privateKey)
 *             .build());
 *         var defaultHostAccountShareKeyAttachment = new HostAccountShareKeyAttachment("defaultHostAccountShareKeyAttachment", HostAccountShareKeyAttachmentArgs.builder()
 *             .instanceId(defaultInstance.id())
 *             .hostShareKeyId(defaultHostShareKey.hostShareKeyId())
 *             .hostAccountId(defaultHostAccount.hostAccountId())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * configuration:
 *   name:
 *     type: string
 *     default: tf_example
 *   privateKey:
 *     type: string
 *     default: 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
 * resources:
 *   defaultSecurityGroup:
 *     type: alicloud:ecs:SecurityGroup
 *     name: default
 *     properties:
 *       vpcId: ${defaultGetNetworks.ids[0]}
 *   defaultInstance:
 *     type: alicloud:bastionhost:Instance
 *     name: default
 *     properties:
 *       description: ${name}
 *       licenseCode: bhah_ent_50_asset
 *       planCode: cloudbastion
 *       storage: '5'
 *       bandwidth: '5'
 *       period: '1'
 *       vswitchId: ${defaultGetSwitches.ids[0]}
 *       securityGroupIds:
 *         - ${defaultSecurityGroup.id}
 *   defaultHost:
 *     type: alicloud:bastionhost:Host
 *     name: default
 *     properties:
 *       instanceId: ${defaultInstance.id}
 *       hostName: ${name}
 *       activeAddressType: Private
 *       hostPrivateAddress: 172.16.0.10
 *       osType: Linux
 *       source: Local
 *   defaultHostAccount:
 *     type: alicloud:bastionhost:HostAccount
 *     name: default
 *     properties:
 *       hostAccountName: ${name}
 *       hostId: ${defaultHost.hostId}
 *       instanceId: ${defaultHost.instanceId}
 *       protocolName: SSH
 *       password: YourPassword12345
 *   defaultHostShareKey:
 *     type: alicloud:bastionhost:HostShareKey
 *     name: default
 *     properties:
 *       hostShareKeyName: ${name}
 *       instanceId: ${defaultInstance.id}
 *       privateKey: ${privateKey}
 *   defaultHostAccountShareKeyAttachment:
 *     type: alicloud:bastionhost:HostAccountShareKeyAttachment
 *     name: default
 *     properties:
 *       instanceId: ${defaultInstance.id}
 *       hostShareKeyId: ${defaultHostShareKey.hostShareKeyId}
 *       hostAccountId: ${defaultHostAccount.hostAccountId}
 * variables:
 *   default:
 *     fn::invoke:
 *       Function: alicloud:getZones
 *       Arguments:
 *         availableResourceCreation: VSwitch
 *   defaultGetNetworks:
 *     fn::invoke:
 *       Function: alicloud:vpc:getNetworks
 *       Arguments:
 *         nameRegex: ^default-NODELETING$
 *         cidrBlock: 10.4.0.0/16
 *   defaultGetSwitches:
 *     fn::invoke:
 *       Function: alicloud:vpc:getSwitches
 *       Arguments:
 *         cidrBlock: 10.4.0.0/24
 *         vpcId: ${defaultGetNetworks.ids[0]}
 *         zoneId: ${default.zones[0].id}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Bastion Host Account Share Key Attachment can be imported using the id, e.g.
 * ```sh
 * $ pulumi import alicloud:bastionhost/hostAccountShareKeyAttachment:HostAccountShareKeyAttachment example <instance_id>:<host_share_key_id>:<host_account_id>
 * ```
 * @property hostAccountId The ID list of the host account.
 * @property hostShareKeyId The ID of the host shared key.
 * @property instanceId The ID of the Bastion machine instance.
 */
public data class HostAccountShareKeyAttachmentArgs(
    public val hostAccountId: Output<String>? = null,
    public val hostShareKeyId: Output<String>? = null,
    public val instanceId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.alicloud.bastionhost.HostAccountShareKeyAttachmentArgs> {
    override fun toJava(): com.pulumi.alicloud.bastionhost.HostAccountShareKeyAttachmentArgs =
        com.pulumi.alicloud.bastionhost.HostAccountShareKeyAttachmentArgs.builder()
            .hostAccountId(hostAccountId?.applyValue({ args0 -> args0 }))
            .hostShareKeyId(hostShareKeyId?.applyValue({ args0 -> args0 }))
            .instanceId(instanceId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [HostAccountShareKeyAttachmentArgs].
 */
@PulumiTagMarker
public class HostAccountShareKeyAttachmentArgsBuilder internal constructor() {
    private var hostAccountId: Output<String>? = null

    private var hostShareKeyId: Output<String>? = null

    private var instanceId: Output<String>? = null

    /**
     * @param value The ID list of the host account.
     */
    @JvmName("ninhfmwsbrfkkhnh")
    public suspend fun hostAccountId(`value`: Output<String>) {
        this.hostAccountId = value
    }

    /**
     * @param value The ID of the host shared key.
     */
    @JvmName("wjbdpksqpgxvmofm")
    public suspend fun hostShareKeyId(`value`: Output<String>) {
        this.hostShareKeyId = value
    }

    /**
     * @param value The ID of the Bastion machine instance.
     */
    @JvmName("ybwtjwwvbjshhxvf")
    public suspend fun instanceId(`value`: Output<String>) {
        this.instanceId = value
    }

    /**
     * @param value The ID list of the host account.
     */
    @JvmName("aylitckqieblgypr")
    public suspend fun hostAccountId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.hostAccountId = mapped
    }

    /**
     * @param value The ID of the host shared key.
     */
    @JvmName("wkvnxtvogwqhoafj")
    public suspend fun hostShareKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.hostShareKeyId = mapped
    }

    /**
     * @param value The ID of the Bastion machine instance.
     */
    @JvmName("tpimkrerbronkgbu")
    public suspend fun instanceId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceId = mapped
    }

    internal fun build(): HostAccountShareKeyAttachmentArgs = HostAccountShareKeyAttachmentArgs(
        hostAccountId = hostAccountId,
        hostShareKeyId = hostShareKeyId,
        instanceId = instanceId,
    )
}
