@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.alicloud.bastionhost.kotlin

import com.pulumi.core.Output
import com.pulumi.kotlin.KotlinCustomResource
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.ResourceMapper
import com.pulumi.kotlin.options.CustomResourceOptions
import com.pulumi.kotlin.options.CustomResourceOptionsBuilder
import com.pulumi.resources.Resource
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit

/**
 * Builder for [HostShareKey].
 */
@PulumiTagMarker
public class HostShareKeyResourceBuilder internal constructor() {
    public var name: String? = null

    public var args: HostShareKeyArgs = HostShareKeyArgs()

    public var opts: CustomResourceOptions = CustomResourceOptions()

    /**
     * @param name The _unique_ name of the resulting resource.
     */
    public fun name(`value`: String) {
        this.name = value
    }

    /**
     * @param block The arguments to use to populate this resource's properties.
     */
    public suspend fun args(block: suspend HostShareKeyArgsBuilder.() -> Unit) {
        val builder = HostShareKeyArgsBuilder()
        block(builder)
        this.args = builder.build()
    }

    /**
     * @param block A bag of options that control this resource's behavior.
     */
    public suspend fun opts(block: suspend CustomResourceOptionsBuilder.() -> Unit) {
        this.opts = com.pulumi.kotlin.options.CustomResourceOptions.opts(block)
    }

    internal fun build(): HostShareKey {
        val builtJavaResource = com.pulumi.alicloud.bastionhost.HostShareKey(
            this.name,
            this.args.toJava(),
            this.opts.toJava(),
        )
        return HostShareKey(builtJavaResource)
    }
}

/**
 * Provides a Bastion Host Share Key resource.
 * For information about Bastion Host Host Share Key and how to use it, see [What is Host Share Key](https://www.alibabacloud.com/help/en/bastion-host/latest/createhostsharekey).
 * > **NOTE:** Available since v1.165.0.
 * ## Example Usage
 * Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as alicloud from "@pulumi/alicloud";
 * export = async () => {
 *     const config = new pulumi.Config();
 *     const name = config.get("name") || "tf_example";
 *     const default = await alicloud.bastionhost.getInstances({});
 *     const defaultGetZones = await alicloud.getZones({
 *         availableResourceCreation: "VSwitch",
 *     });
 *     const defaultGetNetworks = await alicloud.vpc.getNetworks({
 *         nameRegex: "^default-NODELETING$",
 *         cidrBlock: "10.4.0.0/16",
 *     });
 *     const defaultGetSwitches = await alicloud.vpc.getSwitches({
 *         cidrBlock: "10.4.0.0/24",
 *         vpcId: defaultGetNetworks.ids?.[0],
 *         zoneId: defaultGetZones.zones?.[0]?.id,
 *     });
 *     const defaultSecurityGroup: alicloud.ecs.SecurityGroup[] = [];
 *     for (const range = {value: 0}; range.value < (_default.ids.length > 0 ? 0 : 1); range.value++) {
 *         defaultSecurityGroup.push(new alicloud.ecs.SecurityGroup(`default-${range.value}`, {vpcId: defaultGetNetworks.ids?.[0]}));
 *     }
 *     const defaultInstance: alicloud.bastionhost.Instance[] = [];
 *     for (const range = {value: 0}; range.value < (_default.ids.length > 0 ? 0 : 1); range.value++) {
 *         defaultInstance.push(new alicloud.bastionhost.Instance(`default-${range.value}`, {
 *             description: name,
 *             licenseCode: "bhah_ent_50_asset",
 *             planCode: "cloudbastion",
 *             storage: "5",
 *             bandwidth: "5",
 *             period: 1,
 *             vswitchId: defaultGetSwitches.ids?.[0],
 *             securityGroupIds: [defaultSecurityGroup[0]&#46;id],
 *         }));
 *     }
 *     const instanceId = _default.ids.length > 0 ? _default.ids?.[0] : defaultInstance[0].id;
 *     const privateKey = config.get("privateKey") || "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";
 *     const defaultHostShareKey = new alicloud.bastionhost.HostShareKey("default", {
 *         hostShareKeyName: name,
 *         instanceId: instanceId,
 *         privateKey: privateKey,
 *     });
 * }
 * ```
 * ```python
 * import pulumi
 * import pulumi_alicloud as alicloud
 * config = pulumi.Config()
 * name = config.get("name")
 * if name is None:
 *     name = "tf_example"
 * default = alicloud.bastionhost.get_instances()
 * default_get_zones = alicloud.get_zones(available_resource_creation="VSwitch")
 * default_get_networks = alicloud.vpc.get_networks(name_regex="^default-NODELETING$",
 *     cidr_block="10.4.0.0/16")
 * default_get_switches = alicloud.vpc.get_switches(cidr_block="10.4.0.0/24",
 *     vpc_id=default_get_networks.ids[0],
 *     zone_id=default_get_zones.zones[0].id)
 * default_security_group = []
 * for range in [{"value": i} for i in range(0, 0 if len(default&#46;ids) > 0 else 1)]:
 *     default_security_group.append(alicloud.ecs.SecurityGroup(f"default-{range['value']}", vpc_id=default_get_networks&#46;ids[0]))
 * default_instance = []
 * for range in [{"value": i} for i in range(0, 0 if len(default&#46;ids) > 0 else 1)]:
 *     default_instance.append(alicloud.bastionhost.Instance(f"default-{range['value']}",
 *         description=name,
 *         license_code="bhah_ent_50_asset",
 *         plan_code="cloudbastion",
 *         storage="5",
 *         bandwidth="5",
 *         period=1,
 *         vswitch_id=default_get_switches.ids[0],
 *         security_group_ids=[default_security_group[0]&#46;id]))
 * instance_id = default.ids[0] if len(default&#46;ids) > 0 else default_instance[0].id
 * private_key = config.get("privateKey")
 * if private_key is None:
 *     private_key = "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"
 * default_host_share_key = alicloud.bastionhost.HostShareKey("default",
 *     host_share_key_name=name,
 *     instance_id=instance_id,
 *     private_key=private_key)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using System.Threading.Tasks;
 * using Pulumi;
 * using AliCloud = Pulumi.AliCloud;
 * return await Deployment.RunAsync(async() =>
 * {
 *     var config = new Config();
 *     var name = config.Get("name") ?? "tf_example";
 *     var @default = await AliCloud.BastionHost.GetInstances.InvokeAsync();
 *     var defaultGetZones = await AliCloud.GetZones.InvokeAsync(new()
 *     {
 *         AvailableResourceCreation = "VSwitch",
 *     });
 *     var defaultGetNetworks = await AliCloud.Vpc.GetNetworks.InvokeAsync(new()
 *     {
 *         NameRegex = "^default-NODELETING$",
 *         CidrBlock = "10.4.0.0/16",
 *     });
 *     var defaultGetSwitches = await AliCloud.Vpc.GetSwitches.InvokeAsync(new()
 *     {
 *         CidrBlock = "10.4.0.0/24",
 *         VpcId = defaultGetNetworks.Ids[0],
 *         ZoneId = defaultGetZones.Zones[0]?.Id,
 *     });
 *     var defaultSecurityGroup = new List<AliCloud.Ecs.SecurityGroup>();
 *     for (var rangeIndex = 0; rangeIndex < @default.Ids.Length > 0 ? 0 : 1; rangeIndex++)
 *     {
 *         var range = new { Value = rangeIndex };
 *         defaultSecurityGroup.Add(new AliCloud.Ecs.SecurityGroup($"default-{range.Value}", new()
 *         {
 *             VpcId = defaultGetNetworks.Ids[0],
 *         }));
 *     }
 *     var defaultInstance = new List<AliCloud.BastionHost.Instance>();
 *     for (var rangeIndex = 0; rangeIndex < @default.Ids.Length > 0 ? 0 : 1; rangeIndex++)
 *     {
 *         var range = new { Value = rangeIndex };
 *         defaultInstance.Add(new AliCloud.BastionHost.Instance($"default-{range.Value}", new()
 *         {
 *             Description = name,
 *             LicenseCode = "bhah_ent_50_asset",
 *             PlanCode = "cloudbastion",
 *             Storage = "5",
 *             Bandwidth = "5",
 *             Period = 1,
 *             VswitchId = defaultGetSwitches.Ids[0],
 *             SecurityGroupIds = new[]
 *             {
 *                 defaultSecurityGroup[0].Id,
 *             },
 *         }));
 *     }
 *     var instanceId = @default.Ids.Length > 0 ? @default.Ids[0] : defaultInstance[0].Id;
 *     var privateKey = config.Get("privateKey") ?? "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";
 *     var defaultHostShareKey = new AliCloud.BastionHost.HostShareKey("default", new()
 *     {
 *         HostShareKeyName = name,
 *         InstanceId = instanceId,
 *         PrivateKey = privateKey,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/bastionhost"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/ecs"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/vpc"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		cfg := config.New(ctx, "")
 * 		name := "tf_example"
 * 		if param := cfg.Get("name"); param != "" {
 * 			name = param
 * 		}
 * 		_default, err := bastionhost.GetInstances(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultGetZones, err := alicloud.GetZones(ctx, &alicloud.GetZonesArgs{
 * 			AvailableResourceCreation: pulumi.StringRef("VSwitch"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultGetNetworks, err := vpc.GetNetworks(ctx, &vpc.GetNetworksArgs{
 * 			NameRegex: pulumi.StringRef("^default-NODELETING$"),
 * 			CidrBlock: pulumi.StringRef("10.4.0.0/16"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultGetSwitches, err := vpc.GetSwitches(ctx, &vpc.GetSwitchesArgs{
 * 			CidrBlock: pulumi.StringRef("10.4.0.0/24"),
 * 			VpcId:     pulumi.StringRef(defaultGetNetworks.Ids[0]),
 * 			ZoneId:    pulumi.StringRef(defaultGetZones.Zones[0].Id),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		var tmp0 float64
 * 		if pulumi.Float64(len(_default.Ids)) > 0 {
 * 			tmp0 = 0
 * 		} else {
 * 			tmp0 = 1
 * 		}
 * 		var defaultSecurityGroup []*ecs.SecurityGroup
 * 		for index := 0; index < tmp0; index++ {
 * 			key0 := index
 * 			_ := index
 * 			__res, err := ecs.NewSecurityGroup(ctx, fmt.Sprintf("default-%v", key0), &ecs.SecurityGroupArgs{
 * 				VpcId: pulumi.String(defaultGetNetworks.Ids[0]),
 * 			})
 * 			if err != nil {
 * 				return err
 * 			}
 * 			defaultSecurityGroup = append(defaultSecurityGroup, __res)
 * 		}
 * 		var tmp1 float64
 * 		if pulumi.Float64(len(_default.Ids)) > 0 {
 * 			tmp1 = 0
 * 		} else {
 * 			tmp1 = 1
 * 		}
 * 		var defaultInstance []*bastionhost.Instance
 * 		for index := 0; index < tmp1; index++ {
 * 			key0 := index
 * 			_ := index
 * 			__res, err := bastionhost.NewInstance(ctx, fmt.Sprintf("default-%v", key0), &bastionhost.InstanceArgs{
 * 				Description: pulumi.String(name),
 * 				LicenseCode: pulumi.String("bhah_ent_50_asset"),
 * 				PlanCode:    pulumi.String("cloudbastion"),
 * 				Storage:     pulumi.String("5"),
 * 				Bandwidth:   pulumi.String("5"),
 * 				Period:      pulumi.Int(1),
 * 				VswitchId:   pulumi.String(defaultGetSwitches.Ids[0]),
 * 				SecurityGroupIds: pulumi.StringArray{
 * 					defaultSecurityGroup[0].ID(),
 * 				},
 * 			})
 * 			if err != nil {
 * 				return err
 * 			}
 * 			defaultInstance = append(defaultInstance, __res)
 * 		}
 * 		var tmp2 *string
 * 		if pulumi.Float64(len(_default.Ids)) > 0 {
 * 			tmp2 = _default.Ids[0]
 * 		} else {
 * 			tmp2 = defaultInstance[0].ID()
 * 		}
 * 		instanceId := tmp2
 * 		privateKey := "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"
 * 		if param := cfg.Get("privateKey"); param != "" {
 * 			privateKey = param
 * 		}
 * 		_, err = bastionhost.NewHostShareKey(ctx, "default", &bastionhost.HostShareKeyArgs{
 * 			HostShareKeyName: pulumi.String(name),
 * 			InstanceId:       pulumi.String(instanceId),
 * 			PrivateKey:       pulumi.String(privateKey),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.alicloud.bastionhost.BastionhostFunctions;
 * import com.pulumi.alicloud.bastionhost.inputs.GetInstancesArgs;
 * import com.pulumi.alicloud.AlicloudFunctions;
 * import com.pulumi.alicloud.inputs.GetZonesArgs;
 * import com.pulumi.alicloud.vpc.VpcFunctions;
 * import com.pulumi.alicloud.vpc.inputs.GetNetworksArgs;
 * import com.pulumi.alicloud.vpc.inputs.GetSwitchesArgs;
 * import com.pulumi.alicloud.ecs.SecurityGroup;
 * import com.pulumi.alicloud.ecs.SecurityGroupArgs;
 * import com.pulumi.alicloud.bastionhost.Instance;
 * import com.pulumi.alicloud.bastionhost.InstanceArgs;
 * import com.pulumi.alicloud.bastionhost.HostShareKey;
 * import com.pulumi.alicloud.bastionhost.HostShareKeyArgs;
 * import com.pulumi.codegen.internal.KeyedValue;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var name = config.get("name").orElse("tf_example");
 *         final var default = BastionhostFunctions.getInstances();
 *         final var defaultGetZones = AlicloudFunctions.getZones(GetZonesArgs.builder()
 *             .availableResourceCreation("VSwitch")
 *             .build());
 *         final var defaultGetNetworks = VpcFunctions.getNetworks(GetNetworksArgs.builder()
 *             .nameRegex("^default-NODELETING$")
 *             .cidrBlock("10.4.0.0/16")
 *             .build());
 *         final var defaultGetSwitches = VpcFunctions.getSwitches(GetSwitchesArgs.builder()
 *             .cidrBlock("10.4.0.0/24")
 *             .vpcId(defaultGetNetworks.applyValue(getNetworksResult -> getNetworksResult.ids()[0]))
 *             .zoneId(defaultGetZones.applyValue(getZonesResult -> getZonesResult.zones()[0].id()))
 *             .build());
 *         for (var i = 0; i < default_.ids().length() > 0 ? 0 : 1; i++) {
 *             new SecurityGroup("defaultSecurityGroup-" + i, SecurityGroupArgs.builder()
 *                 .vpcId(defaultGetNetworks.applyValue(getNetworksResult -> getNetworksResult.ids()[0]))
 *                 .build());
 * }
 *         for (var i = 0; i < default_.ids().length() > 0 ? 0 : 1; i++) {
 *             new Instance("defaultInstance-" + i, InstanceArgs.builder()
 *                 .description(name)
 *                 .licenseCode("bhah_ent_50_asset")
 *                 .planCode("cloudbastion")
 *                 .storage("5")
 *                 .bandwidth("5")
 *                 .period("1")
 *                 .vswitchId(defaultGetSwitches.applyValue(getSwitchesResult -> getSwitchesResult.ids()[0]))
 *                 .securityGroupIds(defaultSecurityGroup[0].id())
 *                 .build());
 * }
 *         final var instanceId = default_.ids().length() > 0 ? default_.ids()[0] : defaultInstance[0].id();
 *         final var privateKey = config.get("privateKey").orElse("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");
 *         var defaultHostShareKey = new HostShareKey("defaultHostShareKey", HostShareKeyArgs.builder()
 *             .hostShareKeyName(name)
 *             .instanceId(instanceId)
 *             .privateKey(privateKey)
 *             .build());
 *     }
 * }
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Bastion Host Share Key can be imported using the id, e.g.
 * ```sh
 * $ pulumi import alicloud:bastionhost/hostShareKey:HostShareKey example <instance_id>:<host_share_key_id>
 * ```
 */
public class HostShareKey internal constructor(
    override val javaResource: com.pulumi.alicloud.bastionhost.HostShareKey,
) : KotlinCustomResource(javaResource, HostShareKeyMapper) {
    /**
     * The first ID of the resource.
     */
    public val hostShareKeyId: Output<String>
        get() = javaResource.hostShareKeyId().applyValue({ args0 -> args0 })

    /**
     * The name of the host shared key to be added. The name can be a maximum of 128 characters in length.
     */
    public val hostShareKeyName: Output<String>
        get() = javaResource.hostShareKeyName().applyValue({ args0 -> args0 })

    /**
     * The ID of the Bastion instance.
     */
    public val instanceId: Output<String>
        get() = javaResource.instanceId().applyValue({ args0 -> args0 })

    /**
     * The password of the private key. The value is a Base64-encoded string.
     */
    public val passPhrase: Output<String>?
        get() = javaResource.passPhrase().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * The private key. The value is a Base64-encoded string.
     */
    public val privateKey: Output<String>
        get() = javaResource.privateKey().applyValue({ args0 -> args0 })

    /**
     * The fingerprint of the private key.
     */
    public val privateKeyFingerPrint: Output<String>
        get() = javaResource.privateKeyFingerPrint().applyValue({ args0 -> args0 })
}

public object HostShareKeyMapper : ResourceMapper<HostShareKey> {
    override fun supportsMappingOfType(javaResource: Resource): Boolean =
        com.pulumi.alicloud.bastionhost.HostShareKey::class == javaResource::class

    override fun map(javaResource: Resource): HostShareKey = HostShareKey(
        javaResource as
            com.pulumi.alicloud.bastionhost.HostShareKey,
    )
}

/**
 * @see [HostShareKey].
 * @param name The _unique_ name of the resulting resource.
 * @param block Builder for [HostShareKey].
 */
public suspend fun hostShareKey(
    name: String,
    block: suspend HostShareKeyResourceBuilder.() -> Unit,
): HostShareKey {
    val builder = HostShareKeyResourceBuilder()
    builder.name(name)
    block(builder)
    return builder.build()
}

/**
 * @see [HostShareKey].
 * @param name The _unique_ name of the resulting resource.
 */
public fun hostShareKey(name: String): HostShareKey {
    val builder = HostShareKeyResourceBuilder()
    builder.name(name)
    return builder.build()
}
