@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.alicloud.cen.kotlin

import com.pulumi.alicloud.cen.RouteMapArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * This topic provides an overview of the route map function of Cloud Enterprise Networks (CENs).
 * You can use the route map function to filter routes and modify route attributes.
 * By doing so, you can manage the communication between networks attached to a CEN.
 * For information about CEN Route Map and how to use it, see [Manage CEN Route Map](https://www.alibabacloud.com/help/en/cloud-enterprise-network/latest/api-cbn-2017-09-12-createcenroutemap).
 * > **NOTE:** Available since v1.82.0.
 * ## Example Usage
 * Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as alicloud from "@pulumi/alicloud";
 * const config = new pulumi.Config();
 * const sourceRegion = config.get("sourceRegion") || "cn-hangzhou";
 * const destinationRegion = config.get("destinationRegion") || "cn-shanghai";
 * const exampleHz = new alicloud.vpc.Network("example_hz", {
 *     vpcName: "tf_example",
 *     cidrBlock: "192.168.0.0/16",
 * });
 * const exampleSh = new alicloud.vpc.Network("example_sh", {
 *     vpcName: "tf_example",
 *     cidrBlock: "172.16.0.0/12",
 * });
 * const example = new alicloud.cen.Instance("example", {
 *     cenInstanceName: "tf_example",
 *     description: "an example for cen",
 * });
 * const exampleHzInstanceAttachment = new alicloud.cen.InstanceAttachment("example_hz", {
 *     instanceId: example.id,
 *     childInstanceId: exampleHz.id,
 *     childInstanceType: "VPC",
 *     childInstanceRegionId: sourceRegion,
 * });
 * const exampleShInstanceAttachment = new alicloud.cen.InstanceAttachment("example_sh", {
 *     instanceId: example.id,
 *     childInstanceId: exampleSh.id,
 *     childInstanceType: "VPC",
 *     childInstanceRegionId: destinationRegion,
 * });
 * const _default = new alicloud.cen.RouteMap("default", {
 *     cenRegionId: sourceRegion,
 *     cenId: example.id,
 *     description: "tf_example",
 *     priority: 1,
 *     transmitDirection: "RegionIn",
 *     mapResult: "Permit",
 *     nextPriority: 1,
 *     sourceRegionIds: [sourceRegion],
 *     sourceInstanceIds: [exampleHzInstanceAttachment&#46;childInstanceId],
 *     sourceInstanceIdsReverseMatch: false,
 *     destinationInstanceIds: [exampleShInstanceAttachment&#46;childInstanceId],
 *     destinationInstanceIdsReverseMatch: false,
 *     sourceRouteTableIds: [exampleHz&#46;routeTableId],
 *     destinationRouteTableIds: [exampleSh&#46;routeTableId],
 *     sourceChildInstanceTypes: ["VPC"],
 *     destinationChildInstanceTypes: ["VPC"],
 *     destinationCidrBlocks: [exampleSh&#46;cidrBlock],
 *     cidrMatchMode: "Include",
 *     routeTypes: ["System"],
 *     matchAsns: ["65501"],
 *     asPathMatchMode: "Include",
 *     matchCommunitySets: ["65501:1"],
 *     communityMatchMode: "Include",
 *     communityOperateMode: "Additive",
 *     operateCommunitySets: ["65501:1"],
 *     preference: 20,
 *     prependAsPaths: ["65501"],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_alicloud as alicloud
 * config = pulumi.Config()
 * source_region = config.get("sourceRegion")
 * if source_region is None:
 *     source_region = "cn-hangzhou"
 * destination_region = config.get("destinationRegion")
 * if destination_region is None:
 *     destination_region = "cn-shanghai"
 * example_hz = alicloud.vpc.Network("example_hz",
 *     vpc_name="tf_example",
 *     cidr_block="192.168.0.0/16")
 * example_sh = alicloud.vpc.Network("example_sh",
 *     vpc_name="tf_example",
 *     cidr_block="172.16.0.0/12")
 * example = alicloud.cen.Instance("example",
 *     cen_instance_name="tf_example",
 *     description="an example for cen")
 * example_hz_instance_attachment = alicloud.cen.InstanceAttachment("example_hz",
 *     instance_id=example.id,
 *     child_instance_id=example_hz.id,
 *     child_instance_type="VPC",
 *     child_instance_region_id=source_region)
 * example_sh_instance_attachment = alicloud.cen.InstanceAttachment("example_sh",
 *     instance_id=example.id,
 *     child_instance_id=example_sh.id,
 *     child_instance_type="VPC",
 *     child_instance_region_id=destination_region)
 * default = alicloud.cen.RouteMap("default",
 *     cen_region_id=source_region,
 *     cen_id=example.id,
 *     description="tf_example",
 *     priority=1,
 *     transmit_direction="RegionIn",
 *     map_result="Permit",
 *     next_priority=1,
 *     source_region_ids=[source_region],
 *     source_instance_ids=[example_hz_instance_attachment&#46;child_instance_id],
 *     source_instance_ids_reverse_match=False,
 *     destination_instance_ids=[example_sh_instance_attachment&#46;child_instance_id],
 *     destination_instance_ids_reverse_match=False,
 *     source_route_table_ids=[example_hz&#46;route_table_id],
 *     destination_route_table_ids=[example_sh&#46;route_table_id],
 *     source_child_instance_types=["VPC"],
 *     destination_child_instance_types=["VPC"],
 *     destination_cidr_blocks=[example_sh&#46;cidr_block],
 *     cidr_match_mode="Include",
 *     route_types=["System"],
 *     match_asns=["65501"],
 *     as_path_match_mode="Include",
 *     match_community_sets=["65501:1"],
 *     community_match_mode="Include",
 *     community_operate_mode="Additive",
 *     operate_community_sets=["65501:1"],
 *     preference=20,
 *     prepend_as_paths=["65501"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AliCloud = Pulumi.AliCloud;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var sourceRegion = config.Get("sourceRegion") ?? "cn-hangzhou";
 *     var destinationRegion = config.Get("destinationRegion") ?? "cn-shanghai";
 *     var exampleHz = new AliCloud.Vpc.Network("example_hz", new()
 *     {
 *         VpcName = "tf_example",
 *         CidrBlock = "192.168.0.0/16",
 *     });
 *     var exampleSh = new AliCloud.Vpc.Network("example_sh", new()
 *     {
 *         VpcName = "tf_example",
 *         CidrBlock = "172.16.0.0/12",
 *     });
 *     var example = new AliCloud.Cen.Instance("example", new()
 *     {
 *         CenInstanceName = "tf_example",
 *         Description = "an example for cen",
 *     });
 *     var exampleHzInstanceAttachment = new AliCloud.Cen.InstanceAttachment("example_hz", new()
 *     {
 *         InstanceId = example.Id,
 *         ChildInstanceId = exampleHz.Id,
 *         ChildInstanceType = "VPC",
 *         ChildInstanceRegionId = sourceRegion,
 *     });
 *     var exampleShInstanceAttachment = new AliCloud.Cen.InstanceAttachment("example_sh", new()
 *     {
 *         InstanceId = example.Id,
 *         ChildInstanceId = exampleSh.Id,
 *         ChildInstanceType = "VPC",
 *         ChildInstanceRegionId = destinationRegion,
 *     });
 *     var @default = new AliCloud.Cen.RouteMap("default", new()
 *     {
 *         CenRegionId = sourceRegion,
 *         CenId = example.Id,
 *         Description = "tf_example",
 *         Priority = 1,
 *         TransmitDirection = "RegionIn",
 *         MapResult = "Permit",
 *         NextPriority = 1,
 *         SourceRegionIds = new[]
 *         {
 *             sourceRegion,
 *         },
 *         SourceInstanceIds = new[]
 *         {
 *             exampleHzInstanceAttachment.ChildInstanceId,
 *         },
 *         SourceInstanceIdsReverseMatch = false,
 *         DestinationInstanceIds = new[]
 *         {
 *             exampleShInstanceAttachment.ChildInstanceId,
 *         },
 *         DestinationInstanceIdsReverseMatch = false,
 *         SourceRouteTableIds = new[]
 *         {
 *             exampleHz.RouteTableId,
 *         },
 *         DestinationRouteTableIds = new[]
 *         {
 *             exampleSh.RouteTableId,
 *         },
 *         SourceChildInstanceTypes = new[]
 *         {
 *             "VPC",
 *         },
 *         DestinationChildInstanceTypes = new[]
 *         {
 *             "VPC",
 *         },
 *         DestinationCidrBlocks = new[]
 *         {
 *             exampleSh.CidrBlock,
 *         },
 *         CidrMatchMode = "Include",
 *         RouteTypes = new[]
 *         {
 *             "System",
 *         },
 *         MatchAsns = new[]
 *         {
 *             "65501",
 *         },
 *         AsPathMatchMode = "Include",
 *         MatchCommunitySets = new[]
 *         {
 *             "65501:1",
 *         },
 *         CommunityMatchMode = "Include",
 *         CommunityOperateMode = "Additive",
 *         OperateCommunitySets = new[]
 *         {
 *             "65501:1",
 *         },
 *         Preference = 20,
 *         PrependAsPaths = new[]
 *         {
 *             "65501",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/cen"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/vpc"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		cfg := config.New(ctx, "")
 * 		sourceRegion := "cn-hangzhou"
 * 		if param := cfg.Get("sourceRegion"); param != "" {
 * 			sourceRegion = param
 * 		}
 * 		destinationRegion := "cn-shanghai"
 * 		if param := cfg.Get("destinationRegion"); param != "" {
 * 			destinationRegion = param
 * 		}
 * 		exampleHz, err := vpc.NewNetwork(ctx, "example_hz", &vpc.NetworkArgs{
 * 			VpcName:   pulumi.String("tf_example"),
 * 			CidrBlock: pulumi.String("192.168.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleSh, err := vpc.NewNetwork(ctx, "example_sh", &vpc.NetworkArgs{
 * 			VpcName:   pulumi.String("tf_example"),
 * 			CidrBlock: pulumi.String("172.16.0.0/12"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := cen.NewInstance(ctx, "example", &cen.InstanceArgs{
 * 			CenInstanceName: pulumi.String("tf_example"),
 * 			Description:     pulumi.String("an example for cen"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleHzInstanceAttachment, err := cen.NewInstanceAttachment(ctx, "example_hz", &cen.InstanceAttachmentArgs{
 * 			InstanceId:            example.ID(),
 * 			ChildInstanceId:       exampleHz.ID(),
 * 			ChildInstanceType:     pulumi.String("VPC"),
 * 			ChildInstanceRegionId: pulumi.String(sourceRegion),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleShInstanceAttachment, err := cen.NewInstanceAttachment(ctx, "example_sh", &cen.InstanceAttachmentArgs{
 * 			InstanceId:            example.ID(),
 * 			ChildInstanceId:       exampleSh.ID(),
 * 			ChildInstanceType:     pulumi.String("VPC"),
 * 			ChildInstanceRegionId: pulumi.String(destinationRegion),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cen.NewRouteMap(ctx, "default", &cen.RouteMapArgs{
 * 			CenRegionId:       pulumi.String(sourceRegion),
 * 			CenId:             example.ID(),
 * 			Description:       pulumi.String("tf_example"),
 * 			Priority:          pulumi.Int(1),
 * 			TransmitDirection: pulumi.String("RegionIn"),
 * 			MapResult:         pulumi.String("Permit"),
 * 			NextPriority:      pulumi.Int(1),
 * 			SourceRegionIds: pulumi.StringArray{
 * 				pulumi.String(sourceRegion),
 * 			},
 * 			SourceInstanceIds: pulumi.StringArray{
 * 				exampleHzInstanceAttachment.ChildInstanceId,
 * 			},
 * 			SourceInstanceIdsReverseMatch: pulumi.Bool(false),
 * 			DestinationInstanceIds: pulumi.StringArray{
 * 				exampleShInstanceAttachment.ChildInstanceId,
 * 			},
 * 			DestinationInstanceIdsReverseMatch: pulumi.Bool(false),
 * 			SourceRouteTableIds: pulumi.StringArray{
 * 				exampleHz.RouteTableId,
 * 			},
 * 			DestinationRouteTableIds: pulumi.StringArray{
 * 				exampleSh.RouteTableId,
 * 			},
 * 			SourceChildInstanceTypes: pulumi.StringArray{
 * 				pulumi.String("VPC"),
 * 			},
 * 			DestinationChildInstanceTypes: pulumi.StringArray{
 * 				pulumi.String("VPC"),
 * 			},
 * 			DestinationCidrBlocks: pulumi.StringArray{
 * 				exampleSh.CidrBlock,
 * 			},
 * 			CidrMatchMode: pulumi.String("Include"),
 * 			RouteTypes: pulumi.StringArray{
 * 				pulumi.String("System"),
 * 			},
 * 			MatchAsns: pulumi.StringArray{
 * 				pulumi.String("65501"),
 * 			},
 * 			AsPathMatchMode: pulumi.String("Include"),
 * 			MatchCommunitySets: pulumi.StringArray{
 * 				pulumi.String("65501:1"),
 * 			},
 * 			CommunityMatchMode:   pulumi.String("Include"),
 * 			CommunityOperateMode: pulumi.String("Additive"),
 * 			OperateCommunitySets: pulumi.StringArray{
 * 				pulumi.String("65501:1"),
 * 			},
 * 			Preference: pulumi.Int(20),
 * 			PrependAsPaths: pulumi.StringArray{
 * 				pulumi.String("65501"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.alicloud.vpc.Network;
 * import com.pulumi.alicloud.vpc.NetworkArgs;
 * import com.pulumi.alicloud.cen.Instance;
 * import com.pulumi.alicloud.cen.InstanceArgs;
 * import com.pulumi.alicloud.cen.InstanceAttachment;
 * import com.pulumi.alicloud.cen.InstanceAttachmentArgs;
 * import com.pulumi.alicloud.cen.RouteMap;
 * import com.pulumi.alicloud.cen.RouteMapArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var sourceRegion = config.get("sourceRegion").orElse("cn-hangzhou");
 *         final var destinationRegion = config.get("destinationRegion").orElse("cn-shanghai");
 *         var exampleHz = new Network("exampleHz", NetworkArgs.builder()
 *             .vpcName("tf_example")
 *             .cidrBlock("192.168.0.0/16")
 *             .build());
 *         var exampleSh = new Network("exampleSh", NetworkArgs.builder()
 *             .vpcName("tf_example")
 *             .cidrBlock("172.16.0.0/12")
 *             .build());
 *         var example = new Instance("example", InstanceArgs.builder()
 *             .cenInstanceName("tf_example")
 *             .description("an example for cen")
 *             .build());
 *         var exampleHzInstanceAttachment = new InstanceAttachment("exampleHzInstanceAttachment", InstanceAttachmentArgs.builder()
 *             .instanceId(example.id())
 *             .childInstanceId(exampleHz.id())
 *             .childInstanceType("VPC")
 *             .childInstanceRegionId(sourceRegion)
 *             .build());
 *         var exampleShInstanceAttachment = new InstanceAttachment("exampleShInstanceAttachment", InstanceAttachmentArgs.builder()
 *             .instanceId(example.id())
 *             .childInstanceId(exampleSh.id())
 *             .childInstanceType("VPC")
 *             .childInstanceRegionId(destinationRegion)
 *             .build());
 *         var default_ = new RouteMap("default", RouteMapArgs.builder()
 *             .cenRegionId(sourceRegion)
 *             .cenId(example.id())
 *             .description("tf_example")
 *             .priority("1")
 *             .transmitDirection("RegionIn")
 *             .mapResult("Permit")
 *             .nextPriority("1")
 *             .sourceRegionIds(sourceRegion)
 *             .sourceInstanceIds(exampleHzInstanceAttachment.childInstanceId())
 *             .sourceInstanceIdsReverseMatch("false")
 *             .destinationInstanceIds(exampleShInstanceAttachment.childInstanceId())
 *             .destinationInstanceIdsReverseMatch("false")
 *             .sourceRouteTableIds(exampleHz.routeTableId())
 *             .destinationRouteTableIds(exampleSh.routeTableId())
 *             .sourceChildInstanceTypes("VPC")
 *             .destinationChildInstanceTypes("VPC")
 *             .destinationCidrBlocks(exampleSh.cidrBlock())
 *             .cidrMatchMode("Include")
 *             .routeTypes("System")
 *             .matchAsns("65501")
 *             .asPathMatchMode("Include")
 *             .matchCommunitySets("65501:1")
 *             .communityMatchMode("Include")
 *             .communityOperateMode("Additive")
 *             .operateCommunitySets("65501:1")
 *             .preference("20")
 *             .prependAsPaths("65501")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * configuration:
 *   sourceRegion:
 *     type: string
 *     default: cn-hangzhou
 *   destinationRegion:
 *     type: string
 *     default: cn-shanghai
 * resources:
 *   exampleHz:
 *     type: alicloud:vpc:Network
 *     name: example_hz
 *     properties:
 *       vpcName: tf_example
 *       cidrBlock: 192.168.0.0/16
 *   exampleSh:
 *     type: alicloud:vpc:Network
 *     name: example_sh
 *     properties:
 *       vpcName: tf_example
 *       cidrBlock: 172.16.0.0/12
 *   example:
 *     type: alicloud:cen:Instance
 *     properties:
 *       cenInstanceName: tf_example
 *       description: an example for cen
 *   exampleHzInstanceAttachment:
 *     type: alicloud:cen:InstanceAttachment
 *     name: example_hz
 *     properties:
 *       instanceId: ${example.id}
 *       childInstanceId: ${exampleHz.id}
 *       childInstanceType: VPC
 *       childInstanceRegionId: ${sourceRegion}
 *   exampleShInstanceAttachment:
 *     type: alicloud:cen:InstanceAttachment
 *     name: example_sh
 *     properties:
 *       instanceId: ${example.id}
 *       childInstanceId: ${exampleSh.id}
 *       childInstanceType: VPC
 *       childInstanceRegionId: ${destinationRegion}
 *   default:
 *     type: alicloud:cen:RouteMap
 *     properties:
 *       cenRegionId: ${sourceRegion}
 *       cenId: ${example.id}
 *       description: tf_example
 *       priority: '1'
 *       transmitDirection: RegionIn
 *       mapResult: Permit
 *       nextPriority: '1'
 *       sourceRegionIds:
 *         - ${sourceRegion}
 *       sourceInstanceIds:
 *         - ${exampleHzInstanceAttachment.childInstanceId}
 *       sourceInstanceIdsReverseMatch: 'false'
 *       destinationInstanceIds:
 *         - ${exampleShInstanceAttachment.childInstanceId}
 *       destinationInstanceIdsReverseMatch: 'false'
 *       sourceRouteTableIds:
 *         - ${exampleHz.routeTableId}
 *       destinationRouteTableIds:
 *         - ${exampleSh.routeTableId}
 *       sourceChildInstanceTypes:
 *         - VPC
 *       destinationChildInstanceTypes:
 *         - VPC
 *       destinationCidrBlocks:
 *         - ${exampleSh.cidrBlock}
 *       cidrMatchMode: Include
 *       routeTypes:
 *         - System
 *       matchAsns:
 *         - '65501'
 *       asPathMatchMode: Include
 *       matchCommunitySets:
 *         - 65501:1
 *       communityMatchMode: Include
 *       communityOperateMode: Additive
 *       operateCommunitySets:
 *         - 65501:1
 *       preference: '20'
 *       prependAsPaths:
 *         - '65501'
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * CEN RouteMap can be imported using the id, e.g.
 * ```sh
 * $ pulumi import alicloud:cen/routeMap:RouteMap default <cen_id>:<route_map_id>.
 * ```
 * @property asPathMatchMode A match statement. It indicates the mode in which the AS path attribute is matched. Valid values: ["Include", "Complete"].
 * @property cenId The ID of the CEN instance.
 * @property cenRegionId The ID of the region to which the CEN instance belongs.
 * @property cidrMatchMode A match statement. It indicates the mode in which the prefix attribute is matched. Valid values: ["Include", "Complete"].
 * @property communityMatchMode A match statement. It indicates the mode in which the community attribute is matched. Valid values: ["Include", "Complete"].
 * @property communityOperateMode An action statement. It indicates the mode in which the community attribute is operated. Valid values: ["Additive", "Replace"].
 * @property description The description of the route map.
 * @property destinationChildInstanceTypes A match statement that indicates the list of destination instance types. Valid values: ["VPC", "VBR", "CCN", "VPN"].
 * @property destinationCidrBlocks A match statement that indicates the prefix list. The prefix is in the CIDR format. You can enter a maximum of 32 CIDR blocks.
 * @property destinationInstanceIds A match statement that indicates the list of IDs of the destination instances.
 * @property destinationInstanceIdsReverseMatch Indicates whether to enable the reverse match method for the DestinationInstanceIds match condition. Valid values: ["false", "true"]. Default to "false".
 * @property destinationRouteTableIds A match statement that indicates the list of IDs of the destination route tables. You can enter a maximum of 32 route table IDs.
 * @property mapResult The action that is performed to a route if the route matches all the match conditions. Valid values: ["Permit", "Deny"].
 * @property matchAsns A match statement that indicates the AS path list. The AS path is a well-known mandatory attribute, which describes the numbers of the ASs that a BGP route passes through during transmission.
 * @property matchCommunitySets A match statement that indicates the community set. The format of each community is nn:nn, which ranges from 1 to 65535. You can enter a maximum of 32 communities. Communities must comply with RFC 1997. Large communities (RFC 8092) are not supported.
 * @property nextPriority The priority of the next route map that is associated with the current route map. Value range: 1 to 100.
 * @property operateCommunitySets An action statement that operates the community attribute. The format of each community is nn:nn, which ranges from 1 to 65535. You can enter a maximum of 32 communities. Communities must comply with RFC 1997. Large communities (RFC 8092) are not supported.
 * @property preference An action statement that modifies the priority of the route. Value range: 1 to 100. The default priority of a route is 50. A lower value indicates a higher preference.
 * @property prependAsPaths An action statement that indicates an AS path is prepended when the regional gateway receives or advertises a route.
 * @property priority The priority of the route map. Value range: 1 to 100. A lower value indicates a higher priority.
 * @property routeTypes A match statement that indicates the list of route types. Valid values: ["System", "Custom", "BGP"].
 * @property sourceChildInstanceTypes A match statement that indicates the list of source instance types. Valid values: ["VPC", "VBR", "CCN"].
 * @property sourceInstanceIds A match statement that indicates the list of IDs of the source instances.
 * @property sourceInstanceIdsReverseMatch Indicates whether to enable the reverse match method for the SourceInstanceIds match condition. Valid values: ["false", "true"]. Default to "false".
 * @property sourceRegionIds A match statement that indicates the list of IDs of the source regions. You can enter a maximum of 32 region IDs.
 * @property sourceRouteTableIds A match statement that indicates the list of IDs of the source route tables. You can enter a maximum of 32 route table IDs.
 * @property transitRouterRouteTableId The routing table ID of the forwarding router. If you do not enter the routing table ID, the routing policy is automatically associated with the default routing table of the forwarding router.
 * @property transmitDirection The direction in which the route map is applied. Valid values: ["RegionIn", "RegionOut"].
 */
public data class RouteMapArgs(
    public val asPathMatchMode: Output<String>? = null,
    public val cenId: Output<String>? = null,
    public val cenRegionId: Output<String>? = null,
    public val cidrMatchMode: Output<String>? = null,
    public val communityMatchMode: Output<String>? = null,
    public val communityOperateMode: Output<String>? = null,
    public val description: Output<String>? = null,
    public val destinationChildInstanceTypes: Output<List<String>>? = null,
    public val destinationCidrBlocks: Output<List<String>>? = null,
    public val destinationInstanceIds: Output<List<String>>? = null,
    public val destinationInstanceIdsReverseMatch: Output<Boolean>? = null,
    public val destinationRouteTableIds: Output<List<String>>? = null,
    public val mapResult: Output<String>? = null,
    public val matchAsns: Output<List<String>>? = null,
    public val matchCommunitySets: Output<List<String>>? = null,
    public val nextPriority: Output<Int>? = null,
    public val operateCommunitySets: Output<List<String>>? = null,
    public val preference: Output<Int>? = null,
    public val prependAsPaths: Output<List<String>>? = null,
    public val priority: Output<Int>? = null,
    public val routeTypes: Output<List<String>>? = null,
    public val sourceChildInstanceTypes: Output<List<String>>? = null,
    public val sourceInstanceIds: Output<List<String>>? = null,
    public val sourceInstanceIdsReverseMatch: Output<Boolean>? = null,
    public val sourceRegionIds: Output<List<String>>? = null,
    public val sourceRouteTableIds: Output<List<String>>? = null,
    public val transitRouterRouteTableId: Output<String>? = null,
    public val transmitDirection: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.alicloud.cen.RouteMapArgs> {
    override fun toJava(): com.pulumi.alicloud.cen.RouteMapArgs =
        com.pulumi.alicloud.cen.RouteMapArgs.builder()
            .asPathMatchMode(asPathMatchMode?.applyValue({ args0 -> args0 }))
            .cenId(cenId?.applyValue({ args0 -> args0 }))
            .cenRegionId(cenRegionId?.applyValue({ args0 -> args0 }))
            .cidrMatchMode(cidrMatchMode?.applyValue({ args0 -> args0 }))
            .communityMatchMode(communityMatchMode?.applyValue({ args0 -> args0 }))
            .communityOperateMode(communityOperateMode?.applyValue({ args0 -> args0 }))
            .description(description?.applyValue({ args0 -> args0 }))
            .destinationChildInstanceTypes(
                destinationChildInstanceTypes?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .destinationCidrBlocks(destinationCidrBlocks?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .destinationInstanceIds(
                destinationInstanceIds?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .destinationInstanceIdsReverseMatch(
                destinationInstanceIdsReverseMatch?.applyValue({ args0 ->
                    args0
                }),
            )
            .destinationRouteTableIds(
                destinationRouteTableIds?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .mapResult(mapResult?.applyValue({ args0 -> args0 }))
            .matchAsns(matchAsns?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .matchCommunitySets(matchCommunitySets?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .nextPriority(nextPriority?.applyValue({ args0 -> args0 }))
            .operateCommunitySets(operateCommunitySets?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .preference(preference?.applyValue({ args0 -> args0 }))
            .prependAsPaths(prependAsPaths?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .priority(priority?.applyValue({ args0 -> args0 }))
            .routeTypes(routeTypes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .sourceChildInstanceTypes(
                sourceChildInstanceTypes?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .sourceInstanceIds(sourceInstanceIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .sourceInstanceIdsReverseMatch(sourceInstanceIdsReverseMatch?.applyValue({ args0 -> args0 }))
            .sourceRegionIds(sourceRegionIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .sourceRouteTableIds(sourceRouteTableIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .transitRouterRouteTableId(transitRouterRouteTableId?.applyValue({ args0 -> args0 }))
            .transmitDirection(transmitDirection?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [RouteMapArgs].
 */
@PulumiTagMarker
public class RouteMapArgsBuilder internal constructor() {
    private var asPathMatchMode: Output<String>? = null

    private var cenId: Output<String>? = null

    private var cenRegionId: Output<String>? = null

    private var cidrMatchMode: Output<String>? = null

    private var communityMatchMode: Output<String>? = null

    private var communityOperateMode: Output<String>? = null

    private var description: Output<String>? = null

    private var destinationChildInstanceTypes: Output<List<String>>? = null

    private var destinationCidrBlocks: Output<List<String>>? = null

    private var destinationInstanceIds: Output<List<String>>? = null

    private var destinationInstanceIdsReverseMatch: Output<Boolean>? = null

    private var destinationRouteTableIds: Output<List<String>>? = null

    private var mapResult: Output<String>? = null

    private var matchAsns: Output<List<String>>? = null

    private var matchCommunitySets: Output<List<String>>? = null

    private var nextPriority: Output<Int>? = null

    private var operateCommunitySets: Output<List<String>>? = null

    private var preference: Output<Int>? = null

    private var prependAsPaths: Output<List<String>>? = null

    private var priority: Output<Int>? = null

    private var routeTypes: Output<List<String>>? = null

    private var sourceChildInstanceTypes: Output<List<String>>? = null

    private var sourceInstanceIds: Output<List<String>>? = null

    private var sourceInstanceIdsReverseMatch: Output<Boolean>? = null

    private var sourceRegionIds: Output<List<String>>? = null

    private var sourceRouteTableIds: Output<List<String>>? = null

    private var transitRouterRouteTableId: Output<String>? = null

    private var transmitDirection: Output<String>? = null

    /**
     * @param value A match statement. It indicates the mode in which the AS path attribute is matched. Valid values: ["Include", "Complete"].
     */
    @JvmName("cewefcpjmnlosjgr")
    public suspend fun asPathMatchMode(`value`: Output<String>) {
        this.asPathMatchMode = value
    }

    /**
     * @param value The ID of the CEN instance.
     */
    @JvmName("kktgqjnausuuuabn")
    public suspend fun cenId(`value`: Output<String>) {
        this.cenId = value
    }

    /**
     * @param value The ID of the region to which the CEN instance belongs.
     */
    @JvmName("wirdrfbefgadcxpf")
    public suspend fun cenRegionId(`value`: Output<String>) {
        this.cenRegionId = value
    }

    /**
     * @param value A match statement. It indicates the mode in which the prefix attribute is matched. Valid values: ["Include", "Complete"].
     */
    @JvmName("bcnelnmplhhiymel")
    public suspend fun cidrMatchMode(`value`: Output<String>) {
        this.cidrMatchMode = value
    }

    /**
     * @param value A match statement. It indicates the mode in which the community attribute is matched. Valid values: ["Include", "Complete"].
     */
    @JvmName("adcjctocoxnwhfxi")
    public suspend fun communityMatchMode(`value`: Output<String>) {
        this.communityMatchMode = value
    }

    /**
     * @param value An action statement. It indicates the mode in which the community attribute is operated. Valid values: ["Additive", "Replace"].
     */
    @JvmName("qgrtvlsfcxcmaayi")
    public suspend fun communityOperateMode(`value`: Output<String>) {
        this.communityOperateMode = value
    }

    /**
     * @param value The description of the route map.
     */
    @JvmName("jngdmxbynlwevxtb")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value A match statement that indicates the list of destination instance types. Valid values: ["VPC", "VBR", "CCN", "VPN"].
     */
    @JvmName("ewohuiniinhchcwx")
    public suspend fun destinationChildInstanceTypes(`value`: Output<List<String>>) {
        this.destinationChildInstanceTypes = value
    }

    @JvmName("sxqdfshimavyrmcv")
    public suspend fun destinationChildInstanceTypes(vararg values: Output<String>) {
        this.destinationChildInstanceTypes = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the list of destination instance types. Valid values: ["VPC", "VBR", "CCN", "VPN"].
     */
    @JvmName("lgfloijosfockmrk")
    public suspend fun destinationChildInstanceTypes(values: List<Output<String>>) {
        this.destinationChildInstanceTypes = Output.all(values)
    }

    /**
     * @param value A match statement that indicates the prefix list. The prefix is in the CIDR format. You can enter a maximum of 32 CIDR blocks.
     */
    @JvmName("ntfpxqgspssbugfm")
    public suspend fun destinationCidrBlocks(`value`: Output<List<String>>) {
        this.destinationCidrBlocks = value
    }

    @JvmName("jcfdqidhenhorxpb")
    public suspend fun destinationCidrBlocks(vararg values: Output<String>) {
        this.destinationCidrBlocks = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the prefix list. The prefix is in the CIDR format. You can enter a maximum of 32 CIDR blocks.
     */
    @JvmName("dphkbcehhhavwpoj")
    public suspend fun destinationCidrBlocks(values: List<Output<String>>) {
        this.destinationCidrBlocks = Output.all(values)
    }

    /**
     * @param value A match statement that indicates the list of IDs of the destination instances.
     */
    @JvmName("abncpugrqukcoyxs")
    public suspend fun destinationInstanceIds(`value`: Output<List<String>>) {
        this.destinationInstanceIds = value
    }

    @JvmName("wbwmokuiwdcdafyk")
    public suspend fun destinationInstanceIds(vararg values: Output<String>) {
        this.destinationInstanceIds = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the list of IDs of the destination instances.
     */
    @JvmName("amibqhbpcsvaqviq")
    public suspend fun destinationInstanceIds(values: List<Output<String>>) {
        this.destinationInstanceIds = Output.all(values)
    }

    /**
     * @param value Indicates whether to enable the reverse match method for the DestinationInstanceIds match condition. Valid values: ["false", "true"]. Default to "false".
     */
    @JvmName("laborehcspgrirqe")
    public suspend fun destinationInstanceIdsReverseMatch(`value`: Output<Boolean>) {
        this.destinationInstanceIdsReverseMatch = value
    }

    /**
     * @param value A match statement that indicates the list of IDs of the destination route tables. You can enter a maximum of 32 route table IDs.
     */
    @JvmName("raqoxbslcvpgtegx")
    public suspend fun destinationRouteTableIds(`value`: Output<List<String>>) {
        this.destinationRouteTableIds = value
    }

    @JvmName("uekuphltwhpqaaod")
    public suspend fun destinationRouteTableIds(vararg values: Output<String>) {
        this.destinationRouteTableIds = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the list of IDs of the destination route tables. You can enter a maximum of 32 route table IDs.
     */
    @JvmName("ycdilnwntnpjyges")
    public suspend fun destinationRouteTableIds(values: List<Output<String>>) {
        this.destinationRouteTableIds = Output.all(values)
    }

    /**
     * @param value The action that is performed to a route if the route matches all the match conditions. Valid values: ["Permit", "Deny"].
     */
    @JvmName("gdnoadglywensriy")
    public suspend fun mapResult(`value`: Output<String>) {
        this.mapResult = value
    }

    /**
     * @param value A match statement that indicates the AS path list. The AS path is a well-known mandatory attribute, which describes the numbers of the ASs that a BGP route passes through during transmission.
     */
    @JvmName("nqbyivmohaifsagx")
    public suspend fun matchAsns(`value`: Output<List<String>>) {
        this.matchAsns = value
    }

    @JvmName("gpjgdbdkirlhwufa")
    public suspend fun matchAsns(vararg values: Output<String>) {
        this.matchAsns = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the AS path list. The AS path is a well-known mandatory attribute, which describes the numbers of the ASs that a BGP route passes through during transmission.
     */
    @JvmName("txgbhktrghvflbuk")
    public suspend fun matchAsns(values: List<Output<String>>) {
        this.matchAsns = Output.all(values)
    }

    /**
     * @param value A match statement that indicates the community set. The format of each community is nn:nn, which ranges from 1 to 65535. You can enter a maximum of 32 communities. Communities must comply with RFC 1997. Large communities (RFC 8092) are not supported.
     */
    @JvmName("lyfaiuxltoqdrjht")
    public suspend fun matchCommunitySets(`value`: Output<List<String>>) {
        this.matchCommunitySets = value
    }

    @JvmName("ddssagscicjxhpmj")
    public suspend fun matchCommunitySets(vararg values: Output<String>) {
        this.matchCommunitySets = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the community set. The format of each community is nn:nn, which ranges from 1 to 65535. You can enter a maximum of 32 communities. Communities must comply with RFC 1997. Large communities (RFC 8092) are not supported.
     */
    @JvmName("fxfvolopipvhjiab")
    public suspend fun matchCommunitySets(values: List<Output<String>>) {
        this.matchCommunitySets = Output.all(values)
    }

    /**
     * @param value The priority of the next route map that is associated with the current route map. Value range: 1 to 100.
     */
    @JvmName("orthdoncynmneoej")
    public suspend fun nextPriority(`value`: Output<Int>) {
        this.nextPriority = value
    }

    /**
     * @param value An action statement that operates the community attribute. The format of each community is nn:nn, which ranges from 1 to 65535. You can enter a maximum of 32 communities. Communities must comply with RFC 1997. Large communities (RFC 8092) are not supported.
     */
    @JvmName("ehnuiptkhcwondvv")
    public suspend fun operateCommunitySets(`value`: Output<List<String>>) {
        this.operateCommunitySets = value
    }

    @JvmName("vjkjgvrdfbowirgx")
    public suspend fun operateCommunitySets(vararg values: Output<String>) {
        this.operateCommunitySets = Output.all(values.asList())
    }

    /**
     * @param values An action statement that operates the community attribute. The format of each community is nn:nn, which ranges from 1 to 65535. You can enter a maximum of 32 communities. Communities must comply with RFC 1997. Large communities (RFC 8092) are not supported.
     */
    @JvmName("wyornebcttlxdgng")
    public suspend fun operateCommunitySets(values: List<Output<String>>) {
        this.operateCommunitySets = Output.all(values)
    }

    /**
     * @param value An action statement that modifies the priority of the route. Value range: 1 to 100. The default priority of a route is 50. A lower value indicates a higher preference.
     */
    @JvmName("slgajiutlqraqxhm")
    public suspend fun preference(`value`: Output<Int>) {
        this.preference = value
    }

    /**
     * @param value An action statement that indicates an AS path is prepended when the regional gateway receives or advertises a route.
     */
    @JvmName("gmiurcojitukwfoy")
    public suspend fun prependAsPaths(`value`: Output<List<String>>) {
        this.prependAsPaths = value
    }

    @JvmName("wboudbcmyvofymyn")
    public suspend fun prependAsPaths(vararg values: Output<String>) {
        this.prependAsPaths = Output.all(values.asList())
    }

    /**
     * @param values An action statement that indicates an AS path is prepended when the regional gateway receives or advertises a route.
     */
    @JvmName("tddftiofrryjpnpf")
    public suspend fun prependAsPaths(values: List<Output<String>>) {
        this.prependAsPaths = Output.all(values)
    }

    /**
     * @param value The priority of the route map. Value range: 1 to 100. A lower value indicates a higher priority.
     */
    @JvmName("mxvyhlpjinpdipda")
    public suspend fun priority(`value`: Output<Int>) {
        this.priority = value
    }

    /**
     * @param value A match statement that indicates the list of route types. Valid values: ["System", "Custom", "BGP"].
     */
    @JvmName("okwxbtcdvkpdnauy")
    public suspend fun routeTypes(`value`: Output<List<String>>) {
        this.routeTypes = value
    }

    @JvmName("nynxxvnnmpedrtrr")
    public suspend fun routeTypes(vararg values: Output<String>) {
        this.routeTypes = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the list of route types. Valid values: ["System", "Custom", "BGP"].
     */
    @JvmName("sivyudbkyjjvpdjr")
    public suspend fun routeTypes(values: List<Output<String>>) {
        this.routeTypes = Output.all(values)
    }

    /**
     * @param value A match statement that indicates the list of source instance types. Valid values: ["VPC", "VBR", "CCN"].
     */
    @JvmName("labxmbpunrrnaawh")
    public suspend fun sourceChildInstanceTypes(`value`: Output<List<String>>) {
        this.sourceChildInstanceTypes = value
    }

    @JvmName("bebchmqgofdvlytq")
    public suspend fun sourceChildInstanceTypes(vararg values: Output<String>) {
        this.sourceChildInstanceTypes = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the list of source instance types. Valid values: ["VPC", "VBR", "CCN"].
     */
    @JvmName("kqbhrxjtnijxqyac")
    public suspend fun sourceChildInstanceTypes(values: List<Output<String>>) {
        this.sourceChildInstanceTypes = Output.all(values)
    }

    /**
     * @param value A match statement that indicates the list of IDs of the source instances.
     */
    @JvmName("gesatuigoelhqvlj")
    public suspend fun sourceInstanceIds(`value`: Output<List<String>>) {
        this.sourceInstanceIds = value
    }

    @JvmName("ptlnigvyedlaseur")
    public suspend fun sourceInstanceIds(vararg values: Output<String>) {
        this.sourceInstanceIds = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the list of IDs of the source instances.
     */
    @JvmName("egqduvswfvkbujom")
    public suspend fun sourceInstanceIds(values: List<Output<String>>) {
        this.sourceInstanceIds = Output.all(values)
    }

    /**
     * @param value Indicates whether to enable the reverse match method for the SourceInstanceIds match condition. Valid values: ["false", "true"]. Default to "false".
     */
    @JvmName("atcxmpxrxtwdoiib")
    public suspend fun sourceInstanceIdsReverseMatch(`value`: Output<Boolean>) {
        this.sourceInstanceIdsReverseMatch = value
    }

    /**
     * @param value A match statement that indicates the list of IDs of the source regions. You can enter a maximum of 32 region IDs.
     */
    @JvmName("ahgfdykenrhihsev")
    public suspend fun sourceRegionIds(`value`: Output<List<String>>) {
        this.sourceRegionIds = value
    }

    @JvmName("doarmdvvdjxmyhau")
    public suspend fun sourceRegionIds(vararg values: Output<String>) {
        this.sourceRegionIds = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the list of IDs of the source regions. You can enter a maximum of 32 region IDs.
     */
    @JvmName("xxsogvhvyqdnljrd")
    public suspend fun sourceRegionIds(values: List<Output<String>>) {
        this.sourceRegionIds = Output.all(values)
    }

    /**
     * @param value A match statement that indicates the list of IDs of the source route tables. You can enter a maximum of 32 route table IDs.
     */
    @JvmName("xnkccqwcmsfbbkpg")
    public suspend fun sourceRouteTableIds(`value`: Output<List<String>>) {
        this.sourceRouteTableIds = value
    }

    @JvmName("igsessyqvkmplxyn")
    public suspend fun sourceRouteTableIds(vararg values: Output<String>) {
        this.sourceRouteTableIds = Output.all(values.asList())
    }

    /**
     * @param values A match statement that indicates the list of IDs of the source route tables. You can enter a maximum of 32 route table IDs.
     */
    @JvmName("suelyqsbkvbjtnnu")
    public suspend fun sourceRouteTableIds(values: List<Output<String>>) {
        this.sourceRouteTableIds = Output.all(values)
    }

    /**
     * @param value The routing table ID of the forwarding router. If you do not enter the routing table ID, the routing policy is automatically associated with the default routing table of the forwarding router.
     */
    @JvmName("omkatwfdndmvbqlx")
    public suspend fun transitRouterRouteTableId(`value`: Output<String>) {
        this.transitRouterRouteTableId = value
    }

    /**
     * @param value The direction in which the route map is applied. Valid values: ["RegionIn", "RegionOut"].
     */
    @JvmName("ohhvlenlvjokatvt")
    public suspend fun transmitDirection(`value`: Output<String>) {
        this.transmitDirection = value
    }

    /**
     * @param value A match statement. It indicates the mode in which the AS path attribute is matched. Valid values: ["Include", "Complete"].
     */
    @JvmName("qwcgoexksurlpudd")
    public suspend fun asPathMatchMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.asPathMatchMode = mapped
    }

    /**
     * @param value The ID of the CEN instance.
     */
    @JvmName("afabkxpysgkddolq")
    public suspend fun cenId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cenId = mapped
    }

    /**
     * @param value The ID of the region to which the CEN instance belongs.
     */
    @JvmName("bpjnwvultahpacxb")
    public suspend fun cenRegionId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cenRegionId = mapped
    }

    /**
     * @param value A match statement. It indicates the mode in which the prefix attribute is matched. Valid values: ["Include", "Complete"].
     */
    @JvmName("xbbqvcdxtdalkyxb")
    public suspend fun cidrMatchMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cidrMatchMode = mapped
    }

    /**
     * @param value A match statement. It indicates the mode in which the community attribute is matched. Valid values: ["Include", "Complete"].
     */
    @JvmName("kbqkhdyiyrsugmdy")
    public suspend fun communityMatchMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.communityMatchMode = mapped
    }

    /**
     * @param value An action statement. It indicates the mode in which the community attribute is operated. Valid values: ["Additive", "Replace"].
     */
    @JvmName("kwwltqhtkqeywysg")
    public suspend fun communityOperateMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.communityOperateMode = mapped
    }

    /**
     * @param value The description of the route map.
     */
    @JvmName("cwowugejugwyuyha")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value A match statement that indicates the list of destination instance types. Valid values: ["VPC", "VBR", "CCN", "VPN"].
     */
    @JvmName("qcoerbeuachfsqma")
    public suspend fun destinationChildInstanceTypes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.destinationChildInstanceTypes = mapped
    }

    /**
     * @param values A match statement that indicates the list of destination instance types. Valid values: ["VPC", "VBR", "CCN", "VPN"].
     */
    @JvmName("yeodvtqjcjkqmpkt")
    public suspend fun destinationChildInstanceTypes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.destinationChildInstanceTypes = mapped
    }

    /**
     * @param value A match statement that indicates the prefix list. The prefix is in the CIDR format. You can enter a maximum of 32 CIDR blocks.
     */
    @JvmName("uyimtgctvsldkwbn")
    public suspend fun destinationCidrBlocks(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.destinationCidrBlocks = mapped
    }

    /**
     * @param values A match statement that indicates the prefix list. The prefix is in the CIDR format. You can enter a maximum of 32 CIDR blocks.
     */
    @JvmName("gfavogrukhwxdted")
    public suspend fun destinationCidrBlocks(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.destinationCidrBlocks = mapped
    }

    /**
     * @param value A match statement that indicates the list of IDs of the destination instances.
     */
    @JvmName("lrjaxwuiwrlhvvdo")
    public suspend fun destinationInstanceIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.destinationInstanceIds = mapped
    }

    /**
     * @param values A match statement that indicates the list of IDs of the destination instances.
     */
    @JvmName("xetrgsxppysqeyam")
    public suspend fun destinationInstanceIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.destinationInstanceIds = mapped
    }

    /**
     * @param value Indicates whether to enable the reverse match method for the DestinationInstanceIds match condition. Valid values: ["false", "true"]. Default to "false".
     */
    @JvmName("afuyvkdxeivtetng")
    public suspend fun destinationInstanceIdsReverseMatch(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.destinationInstanceIdsReverseMatch = mapped
    }

    /**
     * @param value A match statement that indicates the list of IDs of the destination route tables. You can enter a maximum of 32 route table IDs.
     */
    @JvmName("uwnscxrpvwlydwpo")
    public suspend fun destinationRouteTableIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.destinationRouteTableIds = mapped
    }

    /**
     * @param values A match statement that indicates the list of IDs of the destination route tables. You can enter a maximum of 32 route table IDs.
     */
    @JvmName("sdktesfkcjpyayql")
    public suspend fun destinationRouteTableIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.destinationRouteTableIds = mapped
    }

    /**
     * @param value The action that is performed to a route if the route matches all the match conditions. Valid values: ["Permit", "Deny"].
     */
    @JvmName("dwlqcggmusgscaes")
    public suspend fun mapResult(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.mapResult = mapped
    }

    /**
     * @param value A match statement that indicates the AS path list. The AS path is a well-known mandatory attribute, which describes the numbers of the ASs that a BGP route passes through during transmission.
     */
    @JvmName("kyiivwltugchbkvy")
    public suspend fun matchAsns(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.matchAsns = mapped
    }

    /**
     * @param values A match statement that indicates the AS path list. The AS path is a well-known mandatory attribute, which describes the numbers of the ASs that a BGP route passes through during transmission.
     */
    @JvmName("pqvshpgtcvdkchjy")
    public suspend fun matchAsns(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.matchAsns = mapped
    }

    /**
     * @param value A match statement that indicates the community set. The format of each community is nn:nn, which ranges from 1 to 65535. You can enter a maximum of 32 communities. Communities must comply with RFC 1997. Large communities (RFC 8092) are not supported.
     */
    @JvmName("foxyyeqnsstcecip")
    public suspend fun matchCommunitySets(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.matchCommunitySets = mapped
    }

    /**
     * @param values A match statement that indicates the community set. The format of each community is nn:nn, which ranges from 1 to 65535. You can enter a maximum of 32 communities. Communities must comply with RFC 1997. Large communities (RFC 8092) are not supported.
     */
    @JvmName("ljdkgfwthjocoopj")
    public suspend fun matchCommunitySets(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.matchCommunitySets = mapped
    }

    /**
     * @param value The priority of the next route map that is associated with the current route map. Value range: 1 to 100.
     */
    @JvmName("wsrrcidgkbterefd")
    public suspend fun nextPriority(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.nextPriority = mapped
    }

    /**
     * @param value An action statement that operates the community attribute. The format of each community is nn:nn, which ranges from 1 to 65535. You can enter a maximum of 32 communities. Communities must comply with RFC 1997. Large communities (RFC 8092) are not supported.
     */
    @JvmName("ypagbcyabaydcomc")
    public suspend fun operateCommunitySets(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.operateCommunitySets = mapped
    }

    /**
     * @param values An action statement that operates the community attribute. The format of each community is nn:nn, which ranges from 1 to 65535. You can enter a maximum of 32 communities. Communities must comply with RFC 1997. Large communities (RFC 8092) are not supported.
     */
    @JvmName("pbrsavyframbygpv")
    public suspend fun operateCommunitySets(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.operateCommunitySets = mapped
    }

    /**
     * @param value An action statement that modifies the priority of the route. Value range: 1 to 100. The default priority of a route is 50. A lower value indicates a higher preference.
     */
    @JvmName("fvvrumrrtllacqxs")
    public suspend fun preference(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preference = mapped
    }

    /**
     * @param value An action statement that indicates an AS path is prepended when the regional gateway receives or advertises a route.
     */
    @JvmName("wlepebdljnxeolkx")
    public suspend fun prependAsPaths(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.prependAsPaths = mapped
    }

    /**
     * @param values An action statement that indicates an AS path is prepended when the regional gateway receives or advertises a route.
     */
    @JvmName("tjxnofrudcbanwir")
    public suspend fun prependAsPaths(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.prependAsPaths = mapped
    }

    /**
     * @param value The priority of the route map. Value range: 1 to 100. A lower value indicates a higher priority.
     */
    @JvmName("uudesvlupajuxjth")
    public suspend fun priority(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.priority = mapped
    }

    /**
     * @param value A match statement that indicates the list of route types. Valid values: ["System", "Custom", "BGP"].
     */
    @JvmName("ktfgwsgctkjauowt")
    public suspend fun routeTypes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.routeTypes = mapped
    }

    /**
     * @param values A match statement that indicates the list of route types. Valid values: ["System", "Custom", "BGP"].
     */
    @JvmName("tybrhldnyrcprcsf")
    public suspend fun routeTypes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.routeTypes = mapped
    }

    /**
     * @param value A match statement that indicates the list of source instance types. Valid values: ["VPC", "VBR", "CCN"].
     */
    @JvmName("qmncumocerkhqgwn")
    public suspend fun sourceChildInstanceTypes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceChildInstanceTypes = mapped
    }

    /**
     * @param values A match statement that indicates the list of source instance types. Valid values: ["VPC", "VBR", "CCN"].
     */
    @JvmName("jooootxkbihqiwop")
    public suspend fun sourceChildInstanceTypes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.sourceChildInstanceTypes = mapped
    }

    /**
     * @param value A match statement that indicates the list of IDs of the source instances.
     */
    @JvmName("ybtudwonpychvvqv")
    public suspend fun sourceInstanceIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceInstanceIds = mapped
    }

    /**
     * @param values A match statement that indicates the list of IDs of the source instances.
     */
    @JvmName("jccmrdjwhyykfcgs")
    public suspend fun sourceInstanceIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.sourceInstanceIds = mapped
    }

    /**
     * @param value Indicates whether to enable the reverse match method for the SourceInstanceIds match condition. Valid values: ["false", "true"]. Default to "false".
     */
    @JvmName("mqssfhhnyxlneceo")
    public suspend fun sourceInstanceIdsReverseMatch(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceInstanceIdsReverseMatch = mapped
    }

    /**
     * @param value A match statement that indicates the list of IDs of the source regions. You can enter a maximum of 32 region IDs.
     */
    @JvmName("jsgpfmlrmobggrlc")
    public suspend fun sourceRegionIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceRegionIds = mapped
    }

    /**
     * @param values A match statement that indicates the list of IDs of the source regions. You can enter a maximum of 32 region IDs.
     */
    @JvmName("rfbglvqvfhhcxvhj")
    public suspend fun sourceRegionIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.sourceRegionIds = mapped
    }

    /**
     * @param value A match statement that indicates the list of IDs of the source route tables. You can enter a maximum of 32 route table IDs.
     */
    @JvmName("ercutbkornlpwdlg")
    public suspend fun sourceRouteTableIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceRouteTableIds = mapped
    }

    /**
     * @param values A match statement that indicates the list of IDs of the source route tables. You can enter a maximum of 32 route table IDs.
     */
    @JvmName("wutddnydknqqbbir")
    public suspend fun sourceRouteTableIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.sourceRouteTableIds = mapped
    }

    /**
     * @param value The routing table ID of the forwarding router. If you do not enter the routing table ID, the routing policy is automatically associated with the default routing table of the forwarding router.
     */
    @JvmName("hfcdxkgdexuqucub")
    public suspend fun transitRouterRouteTableId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.transitRouterRouteTableId = mapped
    }

    /**
     * @param value The direction in which the route map is applied. Valid values: ["RegionIn", "RegionOut"].
     */
    @JvmName("ryuepwxgkftckafb")
    public suspend fun transmitDirection(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.transmitDirection = mapped
    }

    internal fun build(): RouteMapArgs = RouteMapArgs(
        asPathMatchMode = asPathMatchMode,
        cenId = cenId,
        cenRegionId = cenRegionId,
        cidrMatchMode = cidrMatchMode,
        communityMatchMode = communityMatchMode,
        communityOperateMode = communityOperateMode,
        description = description,
        destinationChildInstanceTypes = destinationChildInstanceTypes,
        destinationCidrBlocks = destinationCidrBlocks,
        destinationInstanceIds = destinationInstanceIds,
        destinationInstanceIdsReverseMatch = destinationInstanceIdsReverseMatch,
        destinationRouteTableIds = destinationRouteTableIds,
        mapResult = mapResult,
        matchAsns = matchAsns,
        matchCommunitySets = matchCommunitySets,
        nextPriority = nextPriority,
        operateCommunitySets = operateCommunitySets,
        preference = preference,
        prependAsPaths = prependAsPaths,
        priority = priority,
        routeTypes = routeTypes,
        sourceChildInstanceTypes = sourceChildInstanceTypes,
        sourceInstanceIds = sourceInstanceIds,
        sourceInstanceIdsReverseMatch = sourceInstanceIdsReverseMatch,
        sourceRegionIds = sourceRegionIds,
        sourceRouteTableIds = sourceRouteTableIds,
        transitRouterRouteTableId = transitRouterRouteTableId,
        transmitDirection = transmitDirection,
    )
}
