@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.alicloud.dbs.kotlin

import com.pulumi.alicloud.dbs.BackupPlanArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Provides a DBS Backup Plan resource.
 * For information about DBS Backup Plan and how to use it, see [What is Backup Plan](https://www.alibabacloud.com/help/en/dbs/developer-reference/api-dbs-2019-03-06-createandstartbackupplan).
 * > **NOTE:** Available since v1.185.0.
 * ## Example Usage
 * Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as alicloud from "@pulumi/alicloud";
 * const config = new pulumi.Config();
 * const name = config.get("name") || "terraform-example";
 * const default = alicloud.resourcemanager.getResourceGroups({
 *     status: "OK",
 * });
 * const defaultGetZones = alicloud.rds.getZones({
 *     engine: "MySQL",
 *     engineVersion: "8.0",
 *     instanceChargeType: "PostPaid",
 *     category: "HighAvailability",
 *     dbInstanceStorageType: "cloud_essd",
 * });
 * const defaultGetInstanceClasses = defaultGetZones.then(defaultGetZones => alicloud.rds.getInstanceClasses({
 *     zoneId: defaultGetZones.zones?.[1]?.id,
 *     engine: "MySQL",
 *     engineVersion: "8.0",
 *     category: "HighAvailability",
 *     dbInstanceStorageType: "cloud_essd",
 *     instanceChargeType: "PostPaid",
 * }));
 * const defaultGetNetworks = alicloud.vpc.getNetworks({
 *     nameRegex: "^default-NODELETING",
 * });
 * const defaultGetSwitches = Promise.all([defaultGetNetworks, defaultGetZones])&#46;then(([defaultGetNetworks, defaultGetZones]) => alicloud.vpc.getSwitches({
 *     vpcId: defaultGetNetworks.ids?.[0],
 *     zoneId: defaultGetZones.zones?.[1]?.id,
 * }));
 * const vswitchId = defaultGetSwitches.then(defaultGetSwitches => defaultGetSwitches.ids?.[0]);
 * const zoneId = defaultGetZones.then(defaultGetZones => defaultGetZones.ids?.[1]);
 * const defaultSecurityGroup = new alicloud.ecs.SecurityGroup("default", {
 *     name: name,
 *     vpcId: defaultGetNetworks.then(defaultGetNetworks => defaultGetNetworks.ids?.[0]),
 * });
 * const defaultInstance = new alicloud.rds.Instance("default", {
 *     engine: "MySQL",
 *     engineVersion: "8.0",
 *     dbInstanceStorageType: "cloud_essd",
 *     instanceType: defaultGetInstanceClasses.then(defaultGetInstanceClasses => defaultGetInstanceClasses.instanceClasses?.[0]?.instanceClass),
 *     instanceStorage: defaultGetInstanceClasses.then(defaultGetInstanceClasses => defaultGetInstanceClasses.instanceClasses?.[0]?.storageRange?.min),
 *     vswitchId: vswitchId,
 *     instanceName: name,
 * });
 * const defaultDatabase = new alicloud.rds.Database("default", {
 *     instanceId: defaultInstance.id,
 *     name: "tfdatabase",
 * });
 * const defaultRdsAccount = new alicloud.rds.RdsAccount("default", {
 *     dbInstanceId: defaultInstance.id,
 *     accountName: "tfnormal000",
 *     accountPassword: "Test12345",
 * });
 * const defaultAccountPrivilege = new alicloud.rds.AccountPrivilege("default", {
 *     instanceId: defaultInstance.id,
 *     accountName: defaultRdsAccount.accountName,
 *     privilege: "ReadWrite",
 *     dbNames: [defaultDatabase&#46;name],
 * });
 * const defaultBackupPlan = new alicloud.dbs.BackupPlan("default", {
 *     backupPlanName: name,
 *     paymentType: "PayAsYouGo",
 *     instanceClass: "xlarge",
 *     backupMethod: "logical",
 *     databaseType: "MySQL",
 *     databaseRegion: "cn-hangzhou",
 *     storageRegion: "cn-hangzhou",
 *     instanceType: "RDS",
 *     sourceEndpointInstanceType: "RDS",
 *     resourceGroupId: _default.then(_default => _default.ids?.[0]),
 *     sourceEndpointRegion: "cn-hangzhou",
 *     sourceEndpointInstanceId: defaultInstance.id,
 *     sourceEndpointUserName: defaultAccountPrivilege.accountName,
 *     sourceEndpointPassword: defaultRdsAccount.accountPassword,
 *     backupObjects: pulumi.interpolate`[{"DBName":"${defaultDatabase&#46;name}"}]`,
 *     backupPeriod: "Monday",
 *     backupStartTime: "14:22",
 *     backupStorageType: "system",
 *     backupRetentionPeriod: 740,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_alicloud as alicloud
 * config = pulumi.Config()
 * name = config.get("name")
 * if name is None:
 *     name = "terraform-example"
 * default = alicloud.resourcemanager.get_resource_groups(status="OK")
 * default_get_zones = alicloud.rds.get_zones(engine="MySQL",
 *     engine_version="8.0",
 *     instance_charge_type="PostPaid",
 *     category="HighAvailability",
 *     db_instance_storage_type="cloud_essd")
 * default_get_instance_classes = alicloud.rds.get_instance_classes(zone_id=default_get_zones.zones[1].id,
 *     engine="MySQL",
 *     engine_version="8.0",
 *     category="HighAvailability",
 *     db_instance_storage_type="cloud_essd",
 *     instance_charge_type="PostPaid")
 * default_get_networks = alicloud.vpc.get_networks(name_regex="^default-NODELETING")
 * default_get_switches = alicloud.vpc.get_switches(vpc_id=default_get_networks.ids[0],
 *     zone_id=default_get_zones.zones[1].id)
 * vswitch_id = default_get_switches.ids[0]
 * zone_id = default_get_zones.ids[1]
 * default_security_group = alicloud.ecs.SecurityGroup("default",
 *     name=name,
 *     vpc_id=default_get_networks.ids[0])
 * default_instance = alicloud.rds.Instance("default",
 *     engine="MySQL",
 *     engine_version="8.0",
 *     db_instance_storage_type="cloud_essd",
 *     instance_type=default_get_instance_classes.instance_classes[0].instance_class,
 *     instance_storage=default_get_instance_classes.instance_classes[0].storage_range.min,
 *     vswitch_id=vswitch_id,
 *     instance_name=name)
 * default_database = alicloud.rds.Database("default",
 *     instance_id=default_instance.id,
 *     name="tfdatabase")
 * default_rds_account = alicloud.rds.RdsAccount("default",
 *     db_instance_id=default_instance.id,
 *     account_name="tfnormal000",
 *     account_password="Test12345")
 * default_account_privilege = alicloud.rds.AccountPrivilege("default",
 *     instance_id=default_instance.id,
 *     account_name=default_rds_account.account_name,
 *     privilege="ReadWrite",
 *     db_names=[default_database&#46;name])
 * default_backup_plan = alicloud.dbs.BackupPlan("default",
 *     backup_plan_name=name,
 *     payment_type="PayAsYouGo",
 *     instance_class="xlarge",
 *     backup_method="logical",
 *     database_type="MySQL",
 *     database_region="cn-hangzhou",
 *     storage_region="cn-hangzhou",
 *     instance_type="RDS",
 *     source_endpoint_instance_type="RDS",
 *     resource_group_id=default.ids[0],
 *     source_endpoint_region="cn-hangzhou",
 *     source_endpoint_instance_id=default_instance.id,
 *     source_endpoint_user_name=default_account_privilege.account_name,
 *     source_endpoint_password=default_rds_account.account_password,
 *     backup_objects=default_database.name.apply(lambda name: f"[{{\"DBName\":\"{name}\"}}]"),
 *     backup_period="Monday",
 *     backup_start_time="14:22",
 *     backup_storage_type="system",
 *     backup_retention_period=740)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AliCloud = Pulumi.AliCloud;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var name = config.Get("name") ?? "terraform-example";
 *     var @default = AliCloud.ResourceManager.GetResourceGroups.Invoke(new()
 *     {
 *         Status = "OK",
 *     });
 *     var defaultGetZones = AliCloud.Rds.GetZones.Invoke(new()
 *     {
 *         Engine = "MySQL",
 *         EngineVersion = "8.0",
 *         InstanceChargeType = "PostPaid",
 *         Category = "HighAvailability",
 *         DbInstanceStorageType = "cloud_essd",
 *     });
 *     var defaultGetInstanceClasses = AliCloud.Rds.GetInstanceClasses.Invoke(new()
 *     {
 *         ZoneId = defaultGetZones.Apply(getZonesResult => getZonesResult.Zones[1]?.Id),
 *         Engine = "MySQL",
 *         EngineVersion = "8.0",
 *         Category = "HighAvailability",
 *         DbInstanceStorageType = "cloud_essd",
 *         InstanceChargeType = "PostPaid",
 *     });
 *     var defaultGetNetworks = AliCloud.Vpc.GetNetworks.Invoke(new()
 *     {
 *         NameRegex = "^default-NODELETING",
 *     });
 *     var defaultGetSwitches = AliCloud.Vpc.GetSwitches.Invoke(new()
 *     {
 *         VpcId = defaultGetNetworks.Apply(getNetworksResult => getNetworksResult.Ids[0]),
 *         ZoneId = defaultGetZones.Apply(getZonesResult => getZonesResult.Zones[1]?.Id),
 *     });
 *     var vswitchId = defaultGetSwitches.Apply(getSwitchesResult => getSwitchesResult.Ids[0]);
 *     var zoneId = defaultGetZones.Apply(getZonesResult => getZonesResult.Ids[1]);
 *     var defaultSecurityGroup = new AliCloud.Ecs.SecurityGroup("default", new()
 *     {
 *         Name = name,
 *         VpcId = defaultGetNetworks.Apply(getNetworksResult => getNetworksResult.Ids[0]),
 *     });
 *     var defaultInstance = new AliCloud.Rds.Instance("default", new()
 *     {
 *         Engine = "MySQL",
 *         EngineVersion = "8.0",
 *         DbInstanceStorageType = "cloud_essd",
 *         InstanceType = defaultGetInstanceClasses.Apply(getInstanceClassesResult => getInstanceClassesResult.InstanceClasses[0]?.InstanceClass),
 *         InstanceStorage = defaultGetInstanceClasses.Apply(getInstanceClassesResult => getInstanceClassesResult.InstanceClasses[0]?.StorageRange?.Min),
 *         VswitchId = vswitchId,
 *         InstanceName = name,
 *     });
 *     var defaultDatabase = new AliCloud.Rds.Database("default", new()
 *     {
 *         InstanceId = defaultInstance.Id,
 *         Name = "tfdatabase",
 *     });
 *     var defaultRdsAccount = new AliCloud.Rds.RdsAccount("default", new()
 *     {
 *         DbInstanceId = defaultInstance.Id,
 *         AccountName = "tfnormal000",
 *         AccountPassword = "Test12345",
 *     });
 *     var defaultAccountPrivilege = new AliCloud.Rds.AccountPrivilege("default", new()
 *     {
 *         InstanceId = defaultInstance.Id,
 *         AccountName = defaultRdsAccount.AccountName,
 *         Privilege = "ReadWrite",
 *         DbNames = new[]
 *         {
 *             defaultDatabase.Name,
 *         },
 *     });
 *     var defaultBackupPlan = new AliCloud.DBS.BackupPlan("default", new()
 *     {
 *         BackupPlanName = name,
 *         PaymentType = "PayAsYouGo",
 *         InstanceClass = "xlarge",
 *         BackupMethod = "logical",
 *         DatabaseType = "MySQL",
 *         DatabaseRegion = "cn-hangzhou",
 *         StorageRegion = "cn-hangzhou",
 *         InstanceType = "RDS",
 *         SourceEndpointInstanceType = "RDS",
 *         ResourceGroupId = @default.Apply(@default => @default.Apply(getResourceGroupsResult => getResourceGroupsResult.Ids[0])),
 *         SourceEndpointRegion = "cn-hangzhou",
 *         SourceEndpointInstanceId = defaultInstance.Id,
 *         SourceEndpointUserName = defaultAccountPrivilege.AccountName,
 *         SourceEndpointPassword = defaultRdsAccount.AccountPassword,
 *         BackupObjects = defaultDatabase.Name.Apply(name => $"[{{\"DBName\":\"{name}\"}}]"),
 *         BackupPeriod = "Monday",
 *         BackupStartTime = "14:22",
 *         BackupStorageType = "system",
 *         BackupRetentionPeriod = 740,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/dbs"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/ecs"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/rds"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/resourcemanager"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/vpc"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		cfg := config.New(ctx, "")
 * 		name := "terraform-example"
 * 		if param := cfg.Get("name"); param != "" {
 * 			name = param
 * 		}
 * 		_default, err := resourcemanager.GetResourceGroups(ctx, &resourcemanager.GetResourceGroupsArgs{
 * 			Status: pulumi.StringRef("OK"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultGetZones, err := rds.GetZones(ctx, &rds.GetZonesArgs{
 * 			Engine:                pulumi.StringRef("MySQL"),
 * 			EngineVersion:         pulumi.StringRef("8.0"),
 * 			InstanceChargeType:    pulumi.StringRef("PostPaid"),
 * 			Category:              pulumi.StringRef("HighAvailability"),
 * 			DbInstanceStorageType: pulumi.StringRef("cloud_essd"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultGetInstanceClasses, err := rds.GetInstanceClasses(ctx, &rds.GetInstanceClassesArgs{
 * 			ZoneId:                pulumi.StringRef(defaultGetZones.Zones[1].Id),
 * 			Engine:                pulumi.StringRef("MySQL"),
 * 			EngineVersion:         pulumi.StringRef("8.0"),
 * 			Category:              pulumi.StringRef("HighAvailability"),
 * 			DbInstanceStorageType: pulumi.StringRef("cloud_essd"),
 * 			InstanceChargeType:    pulumi.StringRef("PostPaid"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultGetNetworks, err := vpc.GetNetworks(ctx, &vpc.GetNetworksArgs{
 * 			NameRegex: pulumi.StringRef("^default-NODELETING"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultGetSwitches, err := vpc.GetSwitches(ctx, &vpc.GetSwitchesArgs{
 * 			VpcId:  pulumi.StringRef(defaultGetNetworks.Ids[0]),
 * 			ZoneId: pulumi.StringRef(defaultGetZones.Zones[1].Id),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		vswitchId := defaultGetSwitches.Ids[0]
 * 		_ := defaultGetZones.Ids[1]
 * 		_, err = ecs.NewSecurityGroup(ctx, "default", &ecs.SecurityGroupArgs{
 * 			Name:  pulumi.String(name),
 * 			VpcId: pulumi.String(defaultGetNetworks.Ids[0]),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultInstance, err := rds.NewInstance(ctx, "default", &rds.InstanceArgs{
 * 			Engine:                pulumi.String("MySQL"),
 * 			EngineVersion:         pulumi.String("8.0"),
 * 			DbInstanceStorageType: pulumi.String("cloud_essd"),
 * 			InstanceType:          pulumi.String(defaultGetInstanceClasses.InstanceClasses[0].InstanceClass),
 * 			InstanceStorage:       pulumi.String(defaultGetInstanceClasses.InstanceClasses[0].StorageRange.Min),
 * 			VswitchId:             pulumi.String(vswitchId),
 * 			InstanceName:          pulumi.String(name),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultDatabase, err := rds.NewDatabase(ctx, "default", &rds.DatabaseArgs{
 * 			InstanceId: defaultInstance.ID(),
 * 			Name:       pulumi.String("tfdatabase"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultRdsAccount, err := rds.NewRdsAccount(ctx, "default", &rds.RdsAccountArgs{
 * 			DbInstanceId:    defaultInstance.ID(),
 * 			AccountName:     pulumi.String("tfnormal000"),
 * 			AccountPassword: pulumi.String("Test12345"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultAccountPrivilege, err := rds.NewAccountPrivilege(ctx, "default", &rds.AccountPrivilegeArgs{
 * 			InstanceId:  defaultInstance.ID(),
 * 			AccountName: defaultRdsAccount.AccountName,
 * 			Privilege:   pulumi.String("ReadWrite"),
 * 			DbNames: pulumi.StringArray{
 * 				defaultDatabase.Name,
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = dbs.NewBackupPlan(ctx, "default", &dbs.BackupPlanArgs{
 * 			BackupPlanName:             pulumi.String(name),
 * 			PaymentType:                pulumi.String("PayAsYouGo"),
 * 			InstanceClass:              pulumi.String("xlarge"),
 * 			BackupMethod:               pulumi.String("logical"),
 * 			DatabaseType:               pulumi.String("MySQL"),
 * 			DatabaseRegion:             pulumi.String("cn-hangzhou"),
 * 			StorageRegion:              pulumi.String("cn-hangzhou"),
 * 			InstanceType:               pulumi.String("RDS"),
 * 			SourceEndpointInstanceType: pulumi.String("RDS"),
 * 			ResourceGroupId:            pulumi.String(_default.Ids[0]),
 * 			SourceEndpointRegion:       pulumi.String("cn-hangzhou"),
 * 			SourceEndpointInstanceId:   defaultInstance.ID(),
 * 			SourceEndpointUserName:     defaultAccountPrivilege.AccountName,
 * 			SourceEndpointPassword:     defaultRdsAccount.AccountPassword,
 * 			BackupObjects: defaultDatabase.Name.ApplyT(func(name string) (string, error) {
 * 				return fmt.Sprintf("[{\"DBName\":\"%v\"}]", name), nil
 * 			}).(pulumi.StringOutput),
 * 			BackupPeriod:          pulumi.String("Monday"),
 * 			BackupStartTime:       pulumi.String("14:22"),
 * 			BackupStorageType:     pulumi.String("system"),
 * 			BackupRetentionPeriod: pulumi.Int(740),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.alicloud.resourcemanager.ResourcemanagerFunctions;
 * import com.pulumi.alicloud.resourcemanager.inputs.GetResourceGroupsArgs;
 * import com.pulumi.alicloud.rds.RdsFunctions;
 * import com.pulumi.alicloud.rds.inputs.GetZonesArgs;
 * import com.pulumi.alicloud.rds.inputs.GetInstanceClassesArgs;
 * import com.pulumi.alicloud.vpc.VpcFunctions;
 * import com.pulumi.alicloud.vpc.inputs.GetNetworksArgs;
 * import com.pulumi.alicloud.vpc.inputs.GetSwitchesArgs;
 * import com.pulumi.alicloud.ecs.SecurityGroup;
 * import com.pulumi.alicloud.ecs.SecurityGroupArgs;
 * import com.pulumi.alicloud.rds.Instance;
 * import com.pulumi.alicloud.rds.InstanceArgs;
 * import com.pulumi.alicloud.rds.Database;
 * import com.pulumi.alicloud.rds.DatabaseArgs;
 * import com.pulumi.alicloud.rds.RdsAccount;
 * import com.pulumi.alicloud.rds.RdsAccountArgs;
 * import com.pulumi.alicloud.rds.AccountPrivilege;
 * import com.pulumi.alicloud.rds.AccountPrivilegeArgs;
 * import com.pulumi.alicloud.dbs.BackupPlan;
 * import com.pulumi.alicloud.dbs.BackupPlanArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var name = config.get("name").orElse("terraform-example");
 *         final var default = ResourcemanagerFunctions.getResourceGroups(GetResourceGroupsArgs.builder()
 *             .status("OK")
 *             .build());
 *         final var defaultGetZones = RdsFunctions.getZones(GetZonesArgs.builder()
 *             .engine("MySQL")
 *             .engineVersion("8.0")
 *             .instanceChargeType("PostPaid")
 *             .category("HighAvailability")
 *             .dbInstanceStorageType("cloud_essd")
 *             .build());
 *         final var defaultGetInstanceClasses = RdsFunctions.getInstanceClasses(GetInstanceClassesArgs.builder()
 *             .zoneId(defaultGetZones.applyValue(getZonesResult -> getZonesResult.zones()[1].id()))
 *             .engine("MySQL")
 *             .engineVersion("8.0")
 *             .category("HighAvailability")
 *             .dbInstanceStorageType("cloud_essd")
 *             .instanceChargeType("PostPaid")
 *             .build());
 *         final var defaultGetNetworks = VpcFunctions.getNetworks(GetNetworksArgs.builder()
 *             .nameRegex("^default-NODELETING")
 *             .build());
 *         final var defaultGetSwitches = VpcFunctions.getSwitches(GetSwitchesArgs.builder()
 *             .vpcId(defaultGetNetworks.applyValue(getNetworksResult -> getNetworksResult.ids()[0]))
 *             .zoneId(defaultGetZones.applyValue(getZonesResult -> getZonesResult.zones()[1].id()))
 *             .build());
 *         final var vswitchId = defaultGetSwitches.applyValue(getSwitchesResult -> getSwitchesResult.ids()[0]);
 *         final var zoneId = defaultGetZones.applyValue(getZonesResult -> getZonesResult.ids()[1]);
 *         var defaultSecurityGroup = new SecurityGroup("defaultSecurityGroup", SecurityGroupArgs.builder()
 *             .name(name)
 *             .vpcId(defaultGetNetworks.applyValue(getNetworksResult -> getNetworksResult.ids()[0]))
 *             .build());
 *         var defaultInstance = new Instance("defaultInstance", InstanceArgs.builder()
 *             .engine("MySQL")
 *             .engineVersion("8.0")
 *             .dbInstanceStorageType("cloud_essd")
 *             .instanceType(defaultGetInstanceClasses.applyValue(getInstanceClassesResult -> getInstanceClassesResult.instanceClasses()[0].instanceClass()))
 *             .instanceStorage(defaultGetInstanceClasses.applyValue(getInstanceClassesResult -> getInstanceClassesResult.instanceClasses()[0].storageRange().min()))
 *             .vswitchId(vswitchId)
 *             .instanceName(name)
 *             .build());
 *         var defaultDatabase = new Database("defaultDatabase", DatabaseArgs.builder()
 *             .instanceId(defaultInstance.id())
 *             .name("tfdatabase")
 *             .build());
 *         var defaultRdsAccount = new RdsAccount("defaultRdsAccount", RdsAccountArgs.builder()
 *             .dbInstanceId(defaultInstance.id())
 *             .accountName("tfnormal000")
 *             .accountPassword("Test12345")
 *             .build());
 *         var defaultAccountPrivilege = new AccountPrivilege("defaultAccountPrivilege", AccountPrivilegeArgs.builder()
 *             .instanceId(defaultInstance.id())
 *             .accountName(defaultRdsAccount.accountName())
 *             .privilege("ReadWrite")
 *             .dbNames(defaultDatabase.name())
 *             .build());
 *         var defaultBackupPlan = new BackupPlan("defaultBackupPlan", BackupPlanArgs.builder()
 *             .backupPlanName(name)
 *             .paymentType("PayAsYouGo")
 *             .instanceClass("xlarge")
 *             .backupMethod("logical")
 *             .databaseType("MySQL")
 *             .databaseRegion("cn-hangzhou")
 *             .storageRegion("cn-hangzhou")
 *             .instanceType("RDS")
 *             .sourceEndpointInstanceType("RDS")
 *             .resourceGroupId(default_.ids()[0])
 *             .sourceEndpointRegion("cn-hangzhou")
 *             .sourceEndpointInstanceId(defaultInstance.id())
 *             .sourceEndpointUserName(defaultAccountPrivilege.accountName())
 *             .sourceEndpointPassword(defaultRdsAccount.accountPassword())
 *             .backupObjects(defaultDatabase.name().applyValue(name -> String.format("[{{\"DBName\":\"%s\"}}]", name)))
 *             .backupPeriod("Monday")
 *             .backupStartTime("14:22")
 *             .backupStorageType("system")
 *             .backupRetentionPeriod(740)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * configuration:
 *   name:
 *     type: string
 *     default: terraform-example
 * resources:
 *   defaultSecurityGroup:
 *     type: alicloud:ecs:SecurityGroup
 *     name: default
 *     properties:
 *       name: ${name}
 *       vpcId: ${defaultGetNetworks.ids[0]}
 *   defaultInstance:
 *     type: alicloud:rds:Instance
 *     name: default
 *     properties:
 *       engine: MySQL
 *       engineVersion: '8.0'
 *       dbInstanceStorageType: cloud_essd
 *       instanceType: ${defaultGetInstanceClasses.instanceClasses[0].instanceClass}
 *       instanceStorage: ${defaultGetInstanceClasses.instanceClasses[0].storageRange.min}
 *       vswitchId: ${vswitchId}
 *       instanceName: ${name}
 *   defaultDatabase:
 *     type: alicloud:rds:Database
 *     name: default
 *     properties:
 *       instanceId: ${defaultInstance.id}
 *       name: tfdatabase
 *   defaultRdsAccount:
 *     type: alicloud:rds:RdsAccount
 *     name: default
 *     properties:
 *       dbInstanceId: ${defaultInstance.id}
 *       accountName: tfnormal000
 *       accountPassword: Test12345
 *   defaultAccountPrivilege:
 *     type: alicloud:rds:AccountPrivilege
 *     name: default
 *     properties:
 *       instanceId: ${defaultInstance.id}
 *       accountName: ${defaultRdsAccount.accountName}
 *       privilege: ReadWrite
 *       dbNames:
 *         - ${defaultDatabase.name}
 *   defaultBackupPlan:
 *     type: alicloud:dbs:BackupPlan
 *     name: default
 *     properties:
 *       backupPlanName: ${name}
 *       paymentType: PayAsYouGo
 *       instanceClass: xlarge
 *       backupMethod: logical
 *       databaseType: MySQL
 *       databaseRegion: cn-hangzhou
 *       storageRegion: cn-hangzhou
 *       instanceType: RDS
 *       sourceEndpointInstanceType: RDS
 *       resourceGroupId: ${default.ids[0]}
 *       sourceEndpointRegion: cn-hangzhou
 *       sourceEndpointInstanceId: ${defaultInstance.id}
 *       sourceEndpointUserName: ${defaultAccountPrivilege.accountName}
 *       sourceEndpointPassword: ${defaultRdsAccount.accountPassword}
 *       backupObjects: '[{"DBName":"${defaultDatabase&#46;name}"}]'
 *       backupPeriod: Monday
 *       backupStartTime: 14:22
 *       backupStorageType: system
 *       backupRetentionPeriod: 740
 * variables:
 *   default:
 *     fn::invoke:
 *       Function: alicloud:resourcemanager:getResourceGroups
 *       Arguments:
 *         status: OK
 *   defaultGetZones:
 *     fn::invoke:
 *       Function: alicloud:rds:getZones
 *       Arguments:
 *         engine: MySQL
 *         engineVersion: '8.0'
 *         instanceChargeType: PostPaid
 *         category: HighAvailability
 *         dbInstanceStorageType: cloud_essd
 *   defaultGetInstanceClasses:
 *     fn::invoke:
 *       Function: alicloud:rds:getInstanceClasses
 *       Arguments:
 *         zoneId: ${defaultGetZones.zones[1].id}
 *         engine: MySQL
 *         engineVersion: '8.0'
 *         category: HighAvailability
 *         dbInstanceStorageType: cloud_essd
 *         instanceChargeType: PostPaid
 *   defaultGetNetworks:
 *     fn::invoke:
 *       Function: alicloud:vpc:getNetworks
 *       Arguments:
 *         nameRegex: ^default-NODELETING
 *   defaultGetSwitches:
 *     fn::invoke:
 *       Function: alicloud:vpc:getSwitches
 *       Arguments:
 *         vpcId: ${defaultGetNetworks.ids[0]}
 *         zoneId: ${defaultGetZones.zones[1].id}
 *   vswitchId: ${defaultGetSwitches.ids[0]}
 *   zoneId: ${defaultGetZones.ids[1]}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * DBS Backup Plan can be imported using the id, e.g.
 * ```sh
 * $ pulumi import alicloud:dbs/backupPlan:BackupPlan example <id>
 * ```
 * @property backupGatewayId The ID of the backup gateway. This parameter is required when the `source_endpoint_instance_type` is `Agent`.
 * @property backupLogIntervalSeconds The backup log interval seconds.
 * @property backupMethod Backup method. Valid values: `duplication`, `logical`, `physical`.
 * @property backupObjects The backup object.
 * @property backupPeriod Full backup cycle, Valid values: `Monday`, `Tuesday`, `Wednesday`, `Thursday`, `Friday`, `Saturday`, `Sunday`. supports the selection of multiple fetch values, separated by English commas (,).
 * @property backupPlanName The name of the resource.
 * @property backupRateLimit The backup rate limit.
 * @property backupRetentionPeriod The retention time of backup data. Valid values: 0 to 1825. Default value: 730 days.
 * @property backupSpeedLimit The backup speed limit.
 * @property backupStartTime The start time of full Backup. The format is `<I> HH:mm</I>` Z(UTC time).
 * @property backupStorageType Built-in storage type, Valid values: `system`.
 * @property backupStrategyType The backup strategy type. Valid values: `simple`, `manual`.
 * @property crossAliyunId The UID that is backed up across Alibaba cloud accounts.
 * @property crossRoleName The name of the RAM role that is backed up across Alibaba cloud accounts.
 * @property databaseRegion The database region.
 * @property databaseType Database type. Valid values: `DRDS`, `FIle`, `MSSQL`, `MariaDB`, `MongoDB`, `MySQL`, `Oracle`, `PPAS`, `PostgreSQL`, `Redis`.
 * @property duplicationArchivePeriod The storage time for conversion to archive cold standby is 365 days by default.
 * @property duplicationInfrequentAccessPeriod The storage time is converted to low-frequency access. The default time is 180 days.
 * @property enableBackupLog Whether to enable incremental log Backup.
 * @property instanceClass The instance class. Valid values: `large`, `medium`, `micro`, `small`, `xlarge`.
 * @property instanceType The instance type. Valid values: `RDS`, `PolarDB`, `DDS`, `Kvstore`, `Other`.
 * @property ossBucketName The OSS Bucket name. The system automatically generates a new name by default.
 * @property paymentType The payment type of the resource. Valid values: `PayAsYouGo`, `Subscription`.
 * @property period Specify that the prepaid instance is of the package year or monthly type. Valid values: `Month`, `Year`.
 * @property resourceGroupId The ID of the resource group.
 * @property sourceEndpointDatabaseName The name of the database. This parameter is required when the `database_type` is `PostgreSQL` or `MongoDB`.
 * @property sourceEndpointInstanceId The ID of the database instance. This parameter is required when the `source_endpoint_instance_type` is `RDS`, `ECS`, `DDS`, or `Express`.
 * @property sourceEndpointInstanceType The location of the database. Valid values: `RDS`, `ECS`, `Express`, `Agent`, `DDS`, `Other`.
 * @property sourceEndpointIp The source endpoint ip.
 * @property sourceEndpointOracleSid Oracle SID name. This parameter is required when the `database_type` is `Oracle`.
 * @property sourceEndpointPassword The source endpoint password.  This parameter is not required when the `database_type` is `Redis`, or when the `source_endpoint_instance_type` is `Agent` and the `database_type` is `MSSQL`. This parameter is required in other scenarios.
 * @property sourceEndpointPort The source endpoint port.
 * @property sourceEndpointRegion The region of the database. This parameter is required when the `source_endpoint_instance_type` is `RDS`, `ECS`, `DDS`, `Express`, or `Agent`.
 * @property sourceEndpointSid Oracle SID name. This parameter is required when the `database_type` is `Oracle`.
 * @property sourceEndpointUserName The source endpoint username. This parameter is not required when the `database_type` is `Redis`, or when the `source_endpoint_instance_type` is `Agent` and the `database_type` is `MSSQL`. This parameter is required in other scenarios.
 * @property status The status of the resource. Valid values: `pause`, `running`.
 * @property storageRegion The storage region.
 * @property usedTime Specify purchase duration. When the parameter `period` is `Year`, the `used_time` value is 1 to 9. When the parameter `period` is `Month`, the `used_time` value is 1 to 11.
 */
public data class BackupPlanArgs(
    public val backupGatewayId: Output<String>? = null,
    public val backupLogIntervalSeconds: Output<Int>? = null,
    public val backupMethod: Output<String>? = null,
    public val backupObjects: Output<String>? = null,
    public val backupPeriod: Output<String>? = null,
    public val backupPlanName: Output<String>? = null,
    public val backupRateLimit: Output<String>? = null,
    public val backupRetentionPeriod: Output<Int>? = null,
    public val backupSpeedLimit: Output<String>? = null,
    public val backupStartTime: Output<String>? = null,
    public val backupStorageType: Output<String>? = null,
    public val backupStrategyType: Output<String>? = null,
    public val crossAliyunId: Output<String>? = null,
    public val crossRoleName: Output<String>? = null,
    public val databaseRegion: Output<String>? = null,
    public val databaseType: Output<String>? = null,
    public val duplicationArchivePeriod: Output<Int>? = null,
    public val duplicationInfrequentAccessPeriod: Output<Int>? = null,
    public val enableBackupLog: Output<Boolean>? = null,
    public val instanceClass: Output<String>? = null,
    public val instanceType: Output<String>? = null,
    public val ossBucketName: Output<String>? = null,
    public val paymentType: Output<String>? = null,
    public val period: Output<String>? = null,
    public val resourceGroupId: Output<String>? = null,
    public val sourceEndpointDatabaseName: Output<String>? = null,
    public val sourceEndpointInstanceId: Output<String>? = null,
    public val sourceEndpointInstanceType: Output<String>? = null,
    public val sourceEndpointIp: Output<String>? = null,
    public val sourceEndpointOracleSid: Output<String>? = null,
    public val sourceEndpointPassword: Output<String>? = null,
    public val sourceEndpointPort: Output<Int>? = null,
    public val sourceEndpointRegion: Output<String>? = null,
    public val sourceEndpointSid: Output<String>? = null,
    public val sourceEndpointUserName: Output<String>? = null,
    public val status: Output<String>? = null,
    public val storageRegion: Output<String>? = null,
    public val usedTime: Output<Int>? = null,
) : ConvertibleToJava<com.pulumi.alicloud.dbs.BackupPlanArgs> {
    override fun toJava(): com.pulumi.alicloud.dbs.BackupPlanArgs =
        com.pulumi.alicloud.dbs.BackupPlanArgs.builder()
            .backupGatewayId(backupGatewayId?.applyValue({ args0 -> args0 }))
            .backupLogIntervalSeconds(backupLogIntervalSeconds?.applyValue({ args0 -> args0 }))
            .backupMethod(backupMethod?.applyValue({ args0 -> args0 }))
            .backupObjects(backupObjects?.applyValue({ args0 -> args0 }))
            .backupPeriod(backupPeriod?.applyValue({ args0 -> args0 }))
            .backupPlanName(backupPlanName?.applyValue({ args0 -> args0 }))
            .backupRateLimit(backupRateLimit?.applyValue({ args0 -> args0 }))
            .backupRetentionPeriod(backupRetentionPeriod?.applyValue({ args0 -> args0 }))
            .backupSpeedLimit(backupSpeedLimit?.applyValue({ args0 -> args0 }))
            .backupStartTime(backupStartTime?.applyValue({ args0 -> args0 }))
            .backupStorageType(backupStorageType?.applyValue({ args0 -> args0 }))
            .backupStrategyType(backupStrategyType?.applyValue({ args0 -> args0 }))
            .crossAliyunId(crossAliyunId?.applyValue({ args0 -> args0 }))
            .crossRoleName(crossRoleName?.applyValue({ args0 -> args0 }))
            .databaseRegion(databaseRegion?.applyValue({ args0 -> args0 }))
            .databaseType(databaseType?.applyValue({ args0 -> args0 }))
            .duplicationArchivePeriod(duplicationArchivePeriod?.applyValue({ args0 -> args0 }))
            .duplicationInfrequentAccessPeriod(
                duplicationInfrequentAccessPeriod?.applyValue({ args0 ->
                    args0
                }),
            )
            .enableBackupLog(enableBackupLog?.applyValue({ args0 -> args0 }))
            .instanceClass(instanceClass?.applyValue({ args0 -> args0 }))
            .instanceType(instanceType?.applyValue({ args0 -> args0 }))
            .ossBucketName(ossBucketName?.applyValue({ args0 -> args0 }))
            .paymentType(paymentType?.applyValue({ args0 -> args0 }))
            .period(period?.applyValue({ args0 -> args0 }))
            .resourceGroupId(resourceGroupId?.applyValue({ args0 -> args0 }))
            .sourceEndpointDatabaseName(sourceEndpointDatabaseName?.applyValue({ args0 -> args0 }))
            .sourceEndpointInstanceId(sourceEndpointInstanceId?.applyValue({ args0 -> args0 }))
            .sourceEndpointInstanceType(sourceEndpointInstanceType?.applyValue({ args0 -> args0 }))
            .sourceEndpointIp(sourceEndpointIp?.applyValue({ args0 -> args0 }))
            .sourceEndpointOracleSid(sourceEndpointOracleSid?.applyValue({ args0 -> args0 }))
            .sourceEndpointPassword(sourceEndpointPassword?.applyValue({ args0 -> args0 }))
            .sourceEndpointPort(sourceEndpointPort?.applyValue({ args0 -> args0 }))
            .sourceEndpointRegion(sourceEndpointRegion?.applyValue({ args0 -> args0 }))
            .sourceEndpointSid(sourceEndpointSid?.applyValue({ args0 -> args0 }))
            .sourceEndpointUserName(sourceEndpointUserName?.applyValue({ args0 -> args0 }))
            .status(status?.applyValue({ args0 -> args0 }))
            .storageRegion(storageRegion?.applyValue({ args0 -> args0 }))
            .usedTime(usedTime?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [BackupPlanArgs].
 */
@PulumiTagMarker
public class BackupPlanArgsBuilder internal constructor() {
    private var backupGatewayId: Output<String>? = null

    private var backupLogIntervalSeconds: Output<Int>? = null

    private var backupMethod: Output<String>? = null

    private var backupObjects: Output<String>? = null

    private var backupPeriod: Output<String>? = null

    private var backupPlanName: Output<String>? = null

    private var backupRateLimit: Output<String>? = null

    private var backupRetentionPeriod: Output<Int>? = null

    private var backupSpeedLimit: Output<String>? = null

    private var backupStartTime: Output<String>? = null

    private var backupStorageType: Output<String>? = null

    private var backupStrategyType: Output<String>? = null

    private var crossAliyunId: Output<String>? = null

    private var crossRoleName: Output<String>? = null

    private var databaseRegion: Output<String>? = null

    private var databaseType: Output<String>? = null

    private var duplicationArchivePeriod: Output<Int>? = null

    private var duplicationInfrequentAccessPeriod: Output<Int>? = null

    private var enableBackupLog: Output<Boolean>? = null

    private var instanceClass: Output<String>? = null

    private var instanceType: Output<String>? = null

    private var ossBucketName: Output<String>? = null

    private var paymentType: Output<String>? = null

    private var period: Output<String>? = null

    private var resourceGroupId: Output<String>? = null

    private var sourceEndpointDatabaseName: Output<String>? = null

    private var sourceEndpointInstanceId: Output<String>? = null

    private var sourceEndpointInstanceType: Output<String>? = null

    private var sourceEndpointIp: Output<String>? = null

    private var sourceEndpointOracleSid: Output<String>? = null

    private var sourceEndpointPassword: Output<String>? = null

    private var sourceEndpointPort: Output<Int>? = null

    private var sourceEndpointRegion: Output<String>? = null

    private var sourceEndpointSid: Output<String>? = null

    private var sourceEndpointUserName: Output<String>? = null

    private var status: Output<String>? = null

    private var storageRegion: Output<String>? = null

    private var usedTime: Output<Int>? = null

    /**
     * @param value The ID of the backup gateway. This parameter is required when the `source_endpoint_instance_type` is `Agent`.
     */
    @JvmName("sotoxmlbobgrwiad")
    public suspend fun backupGatewayId(`value`: Output<String>) {
        this.backupGatewayId = value
    }

    /**
     * @param value The backup log interval seconds.
     */
    @JvmName("htxgtjwdxedmvrlm")
    public suspend fun backupLogIntervalSeconds(`value`: Output<Int>) {
        this.backupLogIntervalSeconds = value
    }

    /**
     * @param value Backup method. Valid values: `duplication`, `logical`, `physical`.
     */
    @JvmName("ahmahhikgmbcnmlw")
    public suspend fun backupMethod(`value`: Output<String>) {
        this.backupMethod = value
    }

    /**
     * @param value The backup object.
     */
    @JvmName("yfwfrftmtwgvoxai")
    public suspend fun backupObjects(`value`: Output<String>) {
        this.backupObjects = value
    }

    /**
     * @param value Full backup cycle, Valid values: `Monday`, `Tuesday`, `Wednesday`, `Thursday`, `Friday`, `Saturday`, `Sunday`. supports the selection of multiple fetch values, separated by English commas (,).
     */
    @JvmName("gktacpdtjumrudeb")
    public suspend fun backupPeriod(`value`: Output<String>) {
        this.backupPeriod = value
    }

    /**
     * @param value The name of the resource.
     */
    @JvmName("qbrrqbedwthibhcs")
    public suspend fun backupPlanName(`value`: Output<String>) {
        this.backupPlanName = value
    }

    /**
     * @param value The backup rate limit.
     */
    @JvmName("kxoefrfwmkexwuak")
    public suspend fun backupRateLimit(`value`: Output<String>) {
        this.backupRateLimit = value
    }

    /**
     * @param value The retention time of backup data. Valid values: 0 to 1825. Default value: 730 days.
     */
    @JvmName("vmennfcpdrcqvtjj")
    public suspend fun backupRetentionPeriod(`value`: Output<Int>) {
        this.backupRetentionPeriod = value
    }

    /**
     * @param value The backup speed limit.
     */
    @JvmName("ikcnpksfvqnvpins")
    public suspend fun backupSpeedLimit(`value`: Output<String>) {
        this.backupSpeedLimit = value
    }

    /**
     * @param value The start time of full Backup. The format is `<I> HH:mm</I>` Z(UTC time).
     */
    @JvmName("lbfdaiawfprudfhf")
    public suspend fun backupStartTime(`value`: Output<String>) {
        this.backupStartTime = value
    }

    /**
     * @param value Built-in storage type, Valid values: `system`.
     */
    @JvmName("roacafnhnycrsywd")
    public suspend fun backupStorageType(`value`: Output<String>) {
        this.backupStorageType = value
    }

    /**
     * @param value The backup strategy type. Valid values: `simple`, `manual`.
     */
    @JvmName("uqeecclkvomtkhdr")
    public suspend fun backupStrategyType(`value`: Output<String>) {
        this.backupStrategyType = value
    }

    /**
     * @param value The UID that is backed up across Alibaba cloud accounts.
     */
    @JvmName("ihdrunayflhtjpwm")
    public suspend fun crossAliyunId(`value`: Output<String>) {
        this.crossAliyunId = value
    }

    /**
     * @param value The name of the RAM role that is backed up across Alibaba cloud accounts.
     */
    @JvmName("bdshyseetridcjij")
    public suspend fun crossRoleName(`value`: Output<String>) {
        this.crossRoleName = value
    }

    /**
     * @param value The database region.
     */
    @JvmName("ikdwbebcgmprahbl")
    public suspend fun databaseRegion(`value`: Output<String>) {
        this.databaseRegion = value
    }

    /**
     * @param value Database type. Valid values: `DRDS`, `FIle`, `MSSQL`, `MariaDB`, `MongoDB`, `MySQL`, `Oracle`, `PPAS`, `PostgreSQL`, `Redis`.
     */
    @JvmName("qrvtmegfsjwnylnn")
    public suspend fun databaseType(`value`: Output<String>) {
        this.databaseType = value
    }

    /**
     * @param value The storage time for conversion to archive cold standby is 365 days by default.
     */
    @JvmName("jscsrkhbpkybewmb")
    public suspend fun duplicationArchivePeriod(`value`: Output<Int>) {
        this.duplicationArchivePeriod = value
    }

    /**
     * @param value The storage time is converted to low-frequency access. The default time is 180 days.
     */
    @JvmName("yfnfxutilebagnux")
    public suspend fun duplicationInfrequentAccessPeriod(`value`: Output<Int>) {
        this.duplicationInfrequentAccessPeriod = value
    }

    /**
     * @param value Whether to enable incremental log Backup.
     */
    @JvmName("locppybgwawpqatf")
    public suspend fun enableBackupLog(`value`: Output<Boolean>) {
        this.enableBackupLog = value
    }

    /**
     * @param value The instance class. Valid values: `large`, `medium`, `micro`, `small`, `xlarge`.
     */
    @JvmName("gqaatwygxdqfpbup")
    public suspend fun instanceClass(`value`: Output<String>) {
        this.instanceClass = value
    }

    /**
     * @param value The instance type. Valid values: `RDS`, `PolarDB`, `DDS`, `Kvstore`, `Other`.
     */
    @JvmName("dkdhtiddfeujkrmp")
    public suspend fun instanceType(`value`: Output<String>) {
        this.instanceType = value
    }

    /**
     * @param value The OSS Bucket name. The system automatically generates a new name by default.
     */
    @JvmName("kqlbymnkgnqlifjh")
    public suspend fun ossBucketName(`value`: Output<String>) {
        this.ossBucketName = value
    }

    /**
     * @param value The payment type of the resource. Valid values: `PayAsYouGo`, `Subscription`.
     */
    @JvmName("pdkiutuobodddgkg")
    public suspend fun paymentType(`value`: Output<String>) {
        this.paymentType = value
    }

    /**
     * @param value Specify that the prepaid instance is of the package year or monthly type. Valid values: `Month`, `Year`.
     */
    @JvmName("cpodrcqxosfxsupe")
    public suspend fun period(`value`: Output<String>) {
        this.period = value
    }

    /**
     * @param value The ID of the resource group.
     */
    @JvmName("wmujhdwcnqrukxoa")
    public suspend fun resourceGroupId(`value`: Output<String>) {
        this.resourceGroupId = value
    }

    /**
     * @param value The name of the database. This parameter is required when the `database_type` is `PostgreSQL` or `MongoDB`.
     */
    @JvmName("wjfttnqpwecvqypt")
    public suspend fun sourceEndpointDatabaseName(`value`: Output<String>) {
        this.sourceEndpointDatabaseName = value
    }

    /**
     * @param value The ID of the database instance. This parameter is required when the `source_endpoint_instance_type` is `RDS`, `ECS`, `DDS`, or `Express`.
     */
    @JvmName("ekgwjojsghoawfhf")
    public suspend fun sourceEndpointInstanceId(`value`: Output<String>) {
        this.sourceEndpointInstanceId = value
    }

    /**
     * @param value The location of the database. Valid values: `RDS`, `ECS`, `Express`, `Agent`, `DDS`, `Other`.
     */
    @JvmName("qxluactqvvehkapl")
    public suspend fun sourceEndpointInstanceType(`value`: Output<String>) {
        this.sourceEndpointInstanceType = value
    }

    /**
     * @param value The source endpoint ip.
     */
    @JvmName("kuukkdhxiiloftnr")
    public suspend fun sourceEndpointIp(`value`: Output<String>) {
        this.sourceEndpointIp = value
    }

    /**
     * @param value Oracle SID name. This parameter is required when the `database_type` is `Oracle`.
     */
    @JvmName("wwfkshvpolhjptpb")
    public suspend fun sourceEndpointOracleSid(`value`: Output<String>) {
        this.sourceEndpointOracleSid = value
    }

    /**
     * @param value The source endpoint password.  This parameter is not required when the `database_type` is `Redis`, or when the `source_endpoint_instance_type` is `Agent` and the `database_type` is `MSSQL`. This parameter is required in other scenarios.
     */
    @JvmName("ilawcswujrcvilny")
    public suspend fun sourceEndpointPassword(`value`: Output<String>) {
        this.sourceEndpointPassword = value
    }

    /**
     * @param value The source endpoint port.
     */
    @JvmName("rqmbbkhrhlqggqhi")
    public suspend fun sourceEndpointPort(`value`: Output<Int>) {
        this.sourceEndpointPort = value
    }

    /**
     * @param value The region of the database. This parameter is required when the `source_endpoint_instance_type` is `RDS`, `ECS`, `DDS`, `Express`, or `Agent`.
     */
    @JvmName("hywtrupmvlhwpibx")
    public suspend fun sourceEndpointRegion(`value`: Output<String>) {
        this.sourceEndpointRegion = value
    }

    /**
     * @param value Oracle SID name. This parameter is required when the `database_type` is `Oracle`.
     */
    @JvmName("vtqckxahxewjodmg")
    public suspend fun sourceEndpointSid(`value`: Output<String>) {
        this.sourceEndpointSid = value
    }

    /**
     * @param value The source endpoint username. This parameter is not required when the `database_type` is `Redis`, or when the `source_endpoint_instance_type` is `Agent` and the `database_type` is `MSSQL`. This parameter is required in other scenarios.
     */
    @JvmName("pylkxeyvfjegugdj")
    public suspend fun sourceEndpointUserName(`value`: Output<String>) {
        this.sourceEndpointUserName = value
    }

    /**
     * @param value The status of the resource. Valid values: `pause`, `running`.
     */
    @JvmName("dklnfmrxuqxmnlgj")
    public suspend fun status(`value`: Output<String>) {
        this.status = value
    }

    /**
     * @param value The storage region.
     */
    @JvmName("cgfjxbowukfugrpj")
    public suspend fun storageRegion(`value`: Output<String>) {
        this.storageRegion = value
    }

    /**
     * @param value Specify purchase duration. When the parameter `period` is `Year`, the `used_time` value is 1 to 9. When the parameter `period` is `Month`, the `used_time` value is 1 to 11.
     */
    @JvmName("nignhlutwdajnjvc")
    public suspend fun usedTime(`value`: Output<Int>) {
        this.usedTime = value
    }

    /**
     * @param value The ID of the backup gateway. This parameter is required when the `source_endpoint_instance_type` is `Agent`.
     */
    @JvmName("wmxwowxbrfrbmtsg")
    public suspend fun backupGatewayId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupGatewayId = mapped
    }

    /**
     * @param value The backup log interval seconds.
     */
    @JvmName("scbugygrfgudmjms")
    public suspend fun backupLogIntervalSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupLogIntervalSeconds = mapped
    }

    /**
     * @param value Backup method. Valid values: `duplication`, `logical`, `physical`.
     */
    @JvmName("yjvkhcyygirehxbj")
    public suspend fun backupMethod(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupMethod = mapped
    }

    /**
     * @param value The backup object.
     */
    @JvmName("quxkfnqmuprlusny")
    public suspend fun backupObjects(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupObjects = mapped
    }

    /**
     * @param value Full backup cycle, Valid values: `Monday`, `Tuesday`, `Wednesday`, `Thursday`, `Friday`, `Saturday`, `Sunday`. supports the selection of multiple fetch values, separated by English commas (,).
     */
    @JvmName("uchrdlltaeithdgd")
    public suspend fun backupPeriod(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupPeriod = mapped
    }

    /**
     * @param value The name of the resource.
     */
    @JvmName("xbyplsnursvyshfm")
    public suspend fun backupPlanName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupPlanName = mapped
    }

    /**
     * @param value The backup rate limit.
     */
    @JvmName("usrsjgiggqnvotco")
    public suspend fun backupRateLimit(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupRateLimit = mapped
    }

    /**
     * @param value The retention time of backup data. Valid values: 0 to 1825. Default value: 730 days.
     */
    @JvmName("ipkkvopdjgoiqgpe")
    public suspend fun backupRetentionPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupRetentionPeriod = mapped
    }

    /**
     * @param value The backup speed limit.
     */
    @JvmName("ngkusuelvbpboixb")
    public suspend fun backupSpeedLimit(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupSpeedLimit = mapped
    }

    /**
     * @param value The start time of full Backup. The format is `<I> HH:mm</I>` Z(UTC time).
     */
    @JvmName("nvjjcqheaedsnvkw")
    public suspend fun backupStartTime(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupStartTime = mapped
    }

    /**
     * @param value Built-in storage type, Valid values: `system`.
     */
    @JvmName("rgontgollmjdoagf")
    public suspend fun backupStorageType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupStorageType = mapped
    }

    /**
     * @param value The backup strategy type. Valid values: `simple`, `manual`.
     */
    @JvmName("belfmxmoyofrwuof")
    public suspend fun backupStrategyType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupStrategyType = mapped
    }

    /**
     * @param value The UID that is backed up across Alibaba cloud accounts.
     */
    @JvmName("lbgcnyyronrpahmm")
    public suspend fun crossAliyunId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.crossAliyunId = mapped
    }

    /**
     * @param value The name of the RAM role that is backed up across Alibaba cloud accounts.
     */
    @JvmName("uqsiamfmdhydecwf")
    public suspend fun crossRoleName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.crossRoleName = mapped
    }

    /**
     * @param value The database region.
     */
    @JvmName("xxrivdgvxikuswir")
    public suspend fun databaseRegion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.databaseRegion = mapped
    }

    /**
     * @param value Database type. Valid values: `DRDS`, `FIle`, `MSSQL`, `MariaDB`, `MongoDB`, `MySQL`, `Oracle`, `PPAS`, `PostgreSQL`, `Redis`.
     */
    @JvmName("eanvffbjjsytjwla")
    public suspend fun databaseType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.databaseType = mapped
    }

    /**
     * @param value The storage time for conversion to archive cold standby is 365 days by default.
     */
    @JvmName("cvfhdosdiduqtwmc")
    public suspend fun duplicationArchivePeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.duplicationArchivePeriod = mapped
    }

    /**
     * @param value The storage time is converted to low-frequency access. The default time is 180 days.
     */
    @JvmName("egpoerxiapuggstm")
    public suspend fun duplicationInfrequentAccessPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.duplicationInfrequentAccessPeriod = mapped
    }

    /**
     * @param value Whether to enable incremental log Backup.
     */
    @JvmName("oaxophhkqjssfkdo")
    public suspend fun enableBackupLog(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableBackupLog = mapped
    }

    /**
     * @param value The instance class. Valid values: `large`, `medium`, `micro`, `small`, `xlarge`.
     */
    @JvmName("bmdkqemynibuoocj")
    public suspend fun instanceClass(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceClass = mapped
    }

    /**
     * @param value The instance type. Valid values: `RDS`, `PolarDB`, `DDS`, `Kvstore`, `Other`.
     */
    @JvmName("nesoibketdktmtao")
    public suspend fun instanceType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceType = mapped
    }

    /**
     * @param value The OSS Bucket name. The system automatically generates a new name by default.
     */
    @JvmName("vbwfhubhjeesigbp")
    public suspend fun ossBucketName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ossBucketName = mapped
    }

    /**
     * @param value The payment type of the resource. Valid values: `PayAsYouGo`, `Subscription`.
     */
    @JvmName("ymnceqeaffvgebsp")
    public suspend fun paymentType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.paymentType = mapped
    }

    /**
     * @param value Specify that the prepaid instance is of the package year or monthly type. Valid values: `Month`, `Year`.
     */
    @JvmName("tnlluqwkbhlogiki")
    public suspend fun period(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.period = mapped
    }

    /**
     * @param value The ID of the resource group.
     */
    @JvmName("nsbwgopfyivetfgk")
    public suspend fun resourceGroupId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resourceGroupId = mapped
    }

    /**
     * @param value The name of the database. This parameter is required when the `database_type` is `PostgreSQL` or `MongoDB`.
     */
    @JvmName("ndthhlixqgdyelcd")
    public suspend fun sourceEndpointDatabaseName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceEndpointDatabaseName = mapped
    }

    /**
     * @param value The ID of the database instance. This parameter is required when the `source_endpoint_instance_type` is `RDS`, `ECS`, `DDS`, or `Express`.
     */
    @JvmName("fnigskxwgbhpfdyd")
    public suspend fun sourceEndpointInstanceId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceEndpointInstanceId = mapped
    }

    /**
     * @param value The location of the database. Valid values: `RDS`, `ECS`, `Express`, `Agent`, `DDS`, `Other`.
     */
    @JvmName("gxslbwbamilkucfh")
    public suspend fun sourceEndpointInstanceType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceEndpointInstanceType = mapped
    }

    /**
     * @param value The source endpoint ip.
     */
    @JvmName("xueqoulxilhhifux")
    public suspend fun sourceEndpointIp(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceEndpointIp = mapped
    }

    /**
     * @param value Oracle SID name. This parameter is required when the `database_type` is `Oracle`.
     */
    @JvmName("qlegcwokiypbdksw")
    public suspend fun sourceEndpointOracleSid(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceEndpointOracleSid = mapped
    }

    /**
     * @param value The source endpoint password.  This parameter is not required when the `database_type` is `Redis`, or when the `source_endpoint_instance_type` is `Agent` and the `database_type` is `MSSQL`. This parameter is required in other scenarios.
     */
    @JvmName("xkccxfhrvbihebaw")
    public suspend fun sourceEndpointPassword(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceEndpointPassword = mapped
    }

    /**
     * @param value The source endpoint port.
     */
    @JvmName("mijjexycrfeuhjpy")
    public suspend fun sourceEndpointPort(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceEndpointPort = mapped
    }

    /**
     * @param value The region of the database. This parameter is required when the `source_endpoint_instance_type` is `RDS`, `ECS`, `DDS`, `Express`, or `Agent`.
     */
    @JvmName("tyroaxtenlvqigbq")
    public suspend fun sourceEndpointRegion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceEndpointRegion = mapped
    }

    /**
     * @param value Oracle SID name. This parameter is required when the `database_type` is `Oracle`.
     */
    @JvmName("pgqgcrbxutpoydxm")
    public suspend fun sourceEndpointSid(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceEndpointSid = mapped
    }

    /**
     * @param value The source endpoint username. This parameter is not required when the `database_type` is `Redis`, or when the `source_endpoint_instance_type` is `Agent` and the `database_type` is `MSSQL`. This parameter is required in other scenarios.
     */
    @JvmName("hfrscouboisoovoi")
    public suspend fun sourceEndpointUserName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceEndpointUserName = mapped
    }

    /**
     * @param value The status of the resource. Valid values: `pause`, `running`.
     */
    @JvmName("miwnffisyhxvybhv")
    public suspend fun status(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.status = mapped
    }

    /**
     * @param value The storage region.
     */
    @JvmName("otoiqvrhxfoqiyox")
    public suspend fun storageRegion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageRegion = mapped
    }

    /**
     * @param value Specify purchase duration. When the parameter `period` is `Year`, the `used_time` value is 1 to 9. When the parameter `period` is `Month`, the `used_time` value is 1 to 11.
     */
    @JvmName("yrnibhwrfgsmocgx")
    public suspend fun usedTime(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.usedTime = mapped
    }

    internal fun build(): BackupPlanArgs = BackupPlanArgs(
        backupGatewayId = backupGatewayId,
        backupLogIntervalSeconds = backupLogIntervalSeconds,
        backupMethod = backupMethod,
        backupObjects = backupObjects,
        backupPeriod = backupPeriod,
        backupPlanName = backupPlanName,
        backupRateLimit = backupRateLimit,
        backupRetentionPeriod = backupRetentionPeriod,
        backupSpeedLimit = backupSpeedLimit,
        backupStartTime = backupStartTime,
        backupStorageType = backupStorageType,
        backupStrategyType = backupStrategyType,
        crossAliyunId = crossAliyunId,
        crossRoleName = crossRoleName,
        databaseRegion = databaseRegion,
        databaseType = databaseType,
        duplicationArchivePeriod = duplicationArchivePeriod,
        duplicationInfrequentAccessPeriod = duplicationInfrequentAccessPeriod,
        enableBackupLog = enableBackupLog,
        instanceClass = instanceClass,
        instanceType = instanceType,
        ossBucketName = ossBucketName,
        paymentType = paymentType,
        period = period,
        resourceGroupId = resourceGroupId,
        sourceEndpointDatabaseName = sourceEndpointDatabaseName,
        sourceEndpointInstanceId = sourceEndpointInstanceId,
        sourceEndpointInstanceType = sourceEndpointInstanceType,
        sourceEndpointIp = sourceEndpointIp,
        sourceEndpointOracleSid = sourceEndpointOracleSid,
        sourceEndpointPassword = sourceEndpointPassword,
        sourceEndpointPort = sourceEndpointPort,
        sourceEndpointRegion = sourceEndpointRegion,
        sourceEndpointSid = sourceEndpointSid,
        sourceEndpointUserName = sourceEndpointUserName,
        status = status,
        storageRegion = storageRegion,
        usedTime = usedTime,
    )
}
