@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.alicloud.fc.kotlin

import com.pulumi.alicloud.fc.V3CustomDomainArgs.builder
import com.pulumi.alicloud.fc.kotlin.inputs.V3CustomDomainAuthConfigArgs
import com.pulumi.alicloud.fc.kotlin.inputs.V3CustomDomainAuthConfigArgsBuilder
import com.pulumi.alicloud.fc.kotlin.inputs.V3CustomDomainCertConfigArgs
import com.pulumi.alicloud.fc.kotlin.inputs.V3CustomDomainCertConfigArgsBuilder
import com.pulumi.alicloud.fc.kotlin.inputs.V3CustomDomainRouteConfigArgs
import com.pulumi.alicloud.fc.kotlin.inputs.V3CustomDomainRouteConfigArgsBuilder
import com.pulumi.alicloud.fc.kotlin.inputs.V3CustomDomainTlsConfigArgs
import com.pulumi.alicloud.fc.kotlin.inputs.V3CustomDomainTlsConfigArgsBuilder
import com.pulumi.alicloud.fc.kotlin.inputs.V3CustomDomainWafConfigArgs
import com.pulumi.alicloud.fc.kotlin.inputs.V3CustomDomainWafConfigArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.jvm.JvmName

/**
 * Provides a FCV3 Custom Domain resource.
 * Custom Domain names allow users to access FC functions through custom domain names, providing convenience for building Web services using function compute.
 * You can bind a custom domain name to Function Compute and set different paths to different functions of different services.
 * For information about FCV3 Custom Domain and how to use it, see [What is Custom Domain](https://www.alibabacloud.com/help/en/functioncompute/developer-reference/api-fc-2023-03-30-getcustomdomain).
 * > **NOTE:** This content is a technical preview, and should not be relied on in a production environment.
 * > **NOTE:** Available since v1.228.0.
 * ## Example Usage
 * Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as alicloud from "@pulumi/alicloud";
 * const config = new pulumi.Config();
 * const name = config.get("name") || "flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net";
 * const functionName1 = config.get("functionName1") || "terraform-custom-domain-t1";
 * const authConfig = config.get("authConfig") || `{
 *     "jwks": {
 *         "keys": [
 *             {
 *                 "p": "8AdUVeldoE4LueFuzEF_C8tvJ7NhlkzS58Gz9KJTPXPr5DADSUVLWJCr5OdFE79q513SneT0UhGo-JfQ1lNMoNv5-YZ1AxIo9fZUEPIe-KyX9ttaglpzCAUE3TeKdm5J-_HZQzBPKbyUwJHAILNgB2-4IBZZwK7LAfbmfi9TmFM",
 *                 "kty": "RSA",
 *                 "q": "x8m5ydXwC8AAp9I-hOnUAx6yQJz1Nx-jXPCfn--XdHpJuNcuwRQsuUCSRQs_h3SoCI3qZZdzswQnPrtHFxgUJtQFuMj-QZpyMnebDb81rmczl2KPVUtaVDVagJEF6U9Ov3PfrLhvHUEv5u7p6s4Z6maBUaByfFlhEVPv4_ao8Us",
 *                 "d": "bjIQAKD2e65gwJ38_Sqq_EmLFuMMey3gjDv1bSCHFH8fyONJTq-utrZfvspz6EegRwW2mSHW9kq87hRwIBW9y7ED5N4KG5gHDjyh57BRE0SKv0Dz1igtKLyp-nl8-aHc1DbONwr1d7tZfFt255TxIN8cPTakXOp2Av_ztql_JotVUGK8eHmXNJFlvq5tc180sKWMHNSNsCUhQgcB1TWb_gwcqxdsIWPsLZI491XKeTGQ98J7z5h6R1cTC97lfJZ0vNtJahd2jHd3WfTUDj5-untMKyZpYYak2Vr8xtFz8H6Q5Rsz8uX_7gtEqYH2CMjPdbXcebrnD1igRSJMYiP0lQ",
 *                 "e": "AQAB",
 *                 "use": "sig",
 *                 "qi": "MTCCRu8AcvvjbHms7V_sDFO7wX0YNyvOJAAbuTmHvQbJ0NDeDta-f-hi8cjkMk7Fpk2hej158E5gDyO62UG99wHZSbmHT34MvIdmhQ5mnbL-5KK9rxde0nayO1ebGepD_GJThPAg9iskzeWpCg5X2etNo2bHoG_ZLQGXj2BQ1VM",
 *                 "dp": "J4_ttKNcTTnP8PlZO81n1VfYoGCOqylKceyZbq76rVxX-yp2wDLtslFWI8qCtjiMtEnglynPo19JzH-pakocjT70us4Qp0rs-W16ebiOpko8WfHZvzaNUzsQjC3FYrPW-fHo74wc4DI3Cm57jmhCYbdmT9OfQ4UL7Oz3HMFMNAU",
 *                 "alg": "RS256",
 *                 "dq": "H4-VgvYB-sk1EU3cRIDv1iJWRHDHKBMeaoM0pD5kLalX1hRgNW4rdoRl1vRk79AU720D11Kqm2APlxBctaA_JrcdxEg0KkbsvV45p11KbKeu9b5DKFVECsN27ZJ7XZUCuqnibtWf7_4pRBD_8PDoFShmS2_ORiiUdflNjzSbEas",
 *                 "n": "u1LWgoomekdOMfB1lEe96OHehd4XRNCbZRm96RqwOYTTc28Sc_U5wKV2umDzolfoI682ct2BNnRRahYgZPhbOCzHYM6i8sRXjz9Ghx3QHw9zrYACtArwQxrTFiejbfzDPGdPrMQg7T8wjtLtkSyDmCzeXpbIdwmxuLyt_ahLfHelr94kEksMDa42V4Fi5bMW4cCLjlEKzBEHGmFdT8UbLPCvpgsM84JK63e5ifdeI9NdadbC8ZMiR--dFCujT7AgRRyMzxgdn2l-nZJ2ZaYzbLUtAW5_U2kfRVkDNa8d1g__2V5zjU6nfLJ1S2MoXMgRgDPeHpEehZVu2kNaSFvDUQ"
 *             }
 *         ]
 *     },
 *     "tokenLookup": "header:auth",
 *     "claimPassBy": "header:name:name"
 * }
 * `;
 * const certificate = config.get("certificate") || `-----BEGIN CERTIFICATE-----
 * MIIDRjCCAq+gAwIBAgIJAJn3ox4K13PoMA0GCSqGSIb3DQEBBQUAMHYxCzAJBgNV
 * BAYTAkNOMQswCQYDVQQIEwJCSjELMAkGA1UEBxMCQkoxDDAKBgNVBAoTA0FMSTEP
 * MA0GA1UECxMGQUxJWVVOMQ0wCwYDVQQDEwR0ZXN0MR8wHQYJKoZIhvcNAQkBFhB0
 * ZXN0QGhvdG1haWwuY29tMB4XDTE0MTEyNDA2MDQyNVoXDTI0MTEyMTA2MDQyNVow
 * djELMAkGA1UEBhMCQ04xCzAJBgNVBAgTAkJKMQswCQYDVQQHEwJCSjEMMAoGA1UE
 * ChMDQUxJMQ8wDQYDVQQLEwZBTElZVU4xDTALBgNVBAMTBHRlc3QxHzAdBgkqhkiG
 * 9w0BCQEWEHRlc3RAaG90bWFpbC5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJ
 * AoGBAM7SS3e9+Nj0HKAsRuIDNSsS3UK6b+62YQb2uuhKrp1HMrOx61WSDR2qkAnB
 * coG00Uz38EE+9DLYNUVQBK7aSgLP5M1Ak4wr4GqGyCgjejzzh3DshUzLCCy2rook
 * KOyRTlPX+Q5l7rE1fcSNzgepcae5i2sE1XXXzLRIDIvQxcspAgMBAAGjgdswgdgw
 * HQYDVR0OBBYEFBdy+OuMsvbkV7R14f0OyoLoh2z4MIGoBgNVHSMEgaAwgZ2AFBdy
 * +OuMsvbkV7R14f0OyoLoh2z4oXqkeDB2MQswCQYDVQQGEwJDTjELMAkGA1UECBMC
 * QkoxCzAJBgNVBAcTAkJKMQwwCgYDVQQKEwNBTEkxDzANBgNVBAsTBkFMSVlVTjEN
 * MAsGA1UEAxMEdGVzdDEfMB0GCSqGSIb3DQEJARYQdGVzdEBob3RtYWlsLmNvbYIJ
 * AJn3ox4K13PoMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQEFBQADgYEAY7KOsnyT
 * cQzfhiiG7ASjiPakw5wXoycHt5GCvLG5htp2TKVzgv9QTliA3gtfv6oV4zRZx7X1
 * Ofi6hVgErtHaXJheuPVeW6eAW8mHBoEfvDAfU3y9waYrtUevSl07643bzKL6v+Qd
 * DUBTxOAvSYfXTtI90EAxEG/bJJyOm5LqoiA=
 * -----END CERTIFICATE-----
 * `;
 * const privateKey = config.get("privateKey") || `-----BEGIN RSA PRIVATE KEY-----
 * MIICXAIBAAKBgQDO0kt3vfjY9BygLEbiAzUrEt1Cum/utmEG9rroSq6dRzKzsetV
 * kg0dqpAJwXKBtNFM9/BBPvQy2DVFUASu2koCz+TNQJOMK+BqhsgoI3o884dw7IVM
 * ywgstq6KJCjskU5T1/kOZe6xNX3Ejc4HqXGnuYtrBNV118y0SAyL0MXLKQIDAQAB
 * AoGAfe3NxbsGKhN42o4bGsKZPQDfeCHMxayGp5bTd10BtQIE/ST4BcJH+ihAS7Bd
 * 6FwQlKzivNd4GP1MckemklCXfsVckdL94e8ZbJl23GdWul3v8V+KndJHqv5zVJmP
 * hwWoKimwIBTb2s0ctVryr2f18N4hhyFw1yGp0VxclGHkjgECQQD9CvllsnOwHpP4
 * MdrDHbdb29QrobKyKW8pPcDd+sth+kP6Y8MnCVuAKXCKj5FeIsgVtfluPOsZjPzz
 * 71QQWS1dAkEA0T0KXO8gaBQwJhIoo/w6hy5JGZnrNSpOPp5xvJuMAafs2eyvmhJm
 * Ev9SN/Pf2VYa1z6FEnBaLOVD6hf6YQIsPQJAX/CZPoW6dzwgvimo1/GcY6eleiWE
 * qygqjWhsh71e/3bz7yuEAnj5yE3t7Zshcp+dXR3xxGo0eSuLfLFxHgGxwQJAAxf8
 * 9DzQ5NkPkTCJi0sqbl8/03IUKTgT6hcbpWdDXa7m8J3wRr3o5nUB+TPQ5nzAbthM
 * zWX931YQeACcwhxvHQJBAN5mTzzJD4w4Ma6YTaNHyXakdYfyAWrOkPIWZxfhMfXe
 * DrlNdiysTI4Dd1dLeErVpjsckAaOW/JDG5PCSwkaMxk=
 * -----END RSA PRIVATE KEY-----
 * `;
 * const _default = new alicloud.fc.V3CustomDomain("default", {
 *     customDomainName: "flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net",
 *     routeConfig: {
 *         routes: [
 *             {
 *                 functionName: functionName1,
 *                 rewriteConfig: {
 *                     regexRules: [
 *                         {
 *                             match: "/api/*",
 *                             replacement: "$1",
 *                         },
 *                         {
 *                             match: "/api1/*",
 *                             replacement: "$1",
 *                         },
 *                         {
 *                             match: "/api2/*",
 *                             replacement: "$1",
 *                         },
 *                     ],
 *                     wildcardRules: [
 *                         {
 *                             match: "^/api1/.+?/(.*)",
 *                             replacement: "/api/v1/$1",
 *                         },
 *                         {
 *                             match: "^/api2/.+?/(.*)",
 *                             replacement: "/api/v2/$1",
 *                         },
 *                         {
 *                             match: "^/api2/.+?/(.*)",
 *                             replacement: "/api/v3/$1",
 *                         },
 *                     ],
 *                     equalRules: [
 *                         {
 *                             match: "/old",
 *                             replacement: "/new",
 *                         },
 *                         {
 *                             replacement: "/new1",
 *                             match: "/old1",
 *                         },
 *                         {
 *                             match: "/old2",
 *                             replacement: "/new2",
 *                         },
 *                     ],
 *                 },
 *                 methods: [
 *                     "GET",
 *                     "POST",
 *                     "DELETE",
 *                     "HEAD",
 *                 ],
 *                 path: "/a",
 *                 qualifier: "LATEST",
 *             },
 *             {
 *                 functionName: functionName1,
 *                 methods: ["GET"],
 *                 path: "/b",
 *                 qualifier: "LATEST",
 *             },
 *             {
 *                 functionName: functionName1,
 *                 methods: ["POST"],
 *                 path: "/c",
 *                 qualifier: "1",
 *             },
 *         ],
 *     },
 *     authConfig: {
 *         authType: "jwt",
 *         authInfo: authConfig,
 *     },
 *     protocol: "HTTP,HTTPS",
 *     certConfig: {
 *         certName: "cert-name",
 *         certificate: certificate,
 *         privateKey: privateKey,
 *     },
 *     tlsConfig: {
 *         cipherSuites: [
 *             "TLS_RSA_WITH_AES_128_CBC_SHA",
 *             "TLS_RSA_WITH_AES_256_CBC_SHA",
 *             "TLS_RSA_WITH_AES_128_GCM_SHA256",
 *             "TLS_RSA_WITH_AES_256_GCM_SHA384",
 *         ],
 *         maxVersion: "TLSv1.3",
 *         minVersion: "TLSv1.0",
 *     },
 *     wafConfig: {
 *         enableWaf: false,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_alicloud as alicloud
 * config = pulumi.Config()
 * name = config.get("name")
 * if name is None:
 *     name = "flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net"
 * function_name1 = config.get("functionName1")
 * if function_name1 is None:
 *     function_name1 = "terraform-custom-domain-t1"
 * auth_config = config.get("authConfig")
 * if auth_config is None:
 *     auth_config = """{
 *     "jwks": {
 *         "keys": [
 *             {
 *                 "p": "8AdUVeldoE4LueFuzEF_C8tvJ7NhlkzS58Gz9KJTPXPr5DADSUVLWJCr5OdFE79q513SneT0UhGo-JfQ1lNMoNv5-YZ1AxIo9fZUEPIe-KyX9ttaglpzCAUE3TeKdm5J-_HZQzBPKbyUwJHAILNgB2-4IBZZwK7LAfbmfi9TmFM",
 *                 "kty": "RSA",
 *                 "q": "x8m5ydXwC8AAp9I-hOnUAx6yQJz1Nx-jXPCfn--XdHpJuNcuwRQsuUCSRQs_h3SoCI3qZZdzswQnPrtHFxgUJtQFuMj-QZpyMnebDb81rmczl2KPVUtaVDVagJEF6U9Ov3PfrLhvHUEv5u7p6s4Z6maBUaByfFlhEVPv4_ao8Us",
 *                 "d": "bjIQAKD2e65gwJ38_Sqq_EmLFuMMey3gjDv1bSCHFH8fyONJTq-utrZfvspz6EegRwW2mSHW9kq87hRwIBW9y7ED5N4KG5gHDjyh57BRE0SKv0Dz1igtKLyp-nl8-aHc1DbONwr1d7tZfFt255TxIN8cPTakXOp2Av_ztql_JotVUGK8eHmXNJFlvq5tc180sKWMHNSNsCUhQgcB1TWb_gwcqxdsIWPsLZI491XKeTGQ98J7z5h6R1cTC97lfJZ0vNtJahd2jHd3WfTUDj5-untMKyZpYYak2Vr8xtFz8H6Q5Rsz8uX_7gtEqYH2CMjPdbXcebrnD1igRSJMYiP0lQ",
 *                 "e": "AQAB",
 *                 "use": "sig",
 *                 "qi": "MTCCRu8AcvvjbHms7V_sDFO7wX0YNyvOJAAbuTmHvQbJ0NDeDta-f-hi8cjkMk7Fpk2hej158E5gDyO62UG99wHZSbmHT34MvIdmhQ5mnbL-5KK9rxde0nayO1ebGepD_GJThPAg9iskzeWpCg5X2etNo2bHoG_ZLQGXj2BQ1VM",
 *                 "dp": "J4_ttKNcTTnP8PlZO81n1VfYoGCOqylKceyZbq76rVxX-yp2wDLtslFWI8qCtjiMtEnglynPo19JzH-pakocjT70us4Qp0rs-W16ebiOpko8WfHZvzaNUzsQjC3FYrPW-fHo74wc4DI3Cm57jmhCYbdmT9OfQ4UL7Oz3HMFMNAU",
 *                 "alg": "RS256",
 *                 "dq": "H4-VgvYB-sk1EU3cRIDv1iJWRHDHKBMeaoM0pD5kLalX1hRgNW4rdoRl1vRk79AU720D11Kqm2APlxBctaA_JrcdxEg0KkbsvV45p11KbKeu9b5DKFVECsN27ZJ7XZUCuqnibtWf7_4pRBD_8PDoFShmS2_ORiiUdflNjzSbEas",
 *                 "n": "u1LWgoomekdOMfB1lEe96OHehd4XRNCbZRm96RqwOYTTc28Sc_U5wKV2umDzolfoI682ct2BNnRRahYgZPhbOCzHYM6i8sRXjz9Ghx3QHw9zrYACtArwQxrTFiejbfzDPGdPrMQg7T8wjtLtkSyDmCzeXpbIdwmxuLyt_ahLfHelr94kEksMDa42V4Fi5bMW4cCLjlEKzBEHGmFdT8UbLPCvpgsM84JK63e5ifdeI9NdadbC8ZMiR--dFCujT7AgRRyMzxgdn2l-nZJ2ZaYzbLUtAW5_U2kfRVkDNa8d1g__2V5zjU6nfLJ1S2MoXMgRgDPeHpEehZVu2kNaSFvDUQ"
 *             }
 *         ]
 *     },
 *     "tokenLookup": "header:auth",
 *     "claimPassBy": "header:name:name"
 * }
 * """
 * certificate = config.get("certificate")
 * if certificate is None:
 *     certificate = """-----BEGIN CERTIFICATE-----
 * MIIDRjCCAq+gAwIBAgIJAJn3ox4K13PoMA0GCSqGSIb3DQEBBQUAMHYxCzAJBgNV
 * BAYTAkNOMQswCQYDVQQIEwJCSjELMAkGA1UEBxMCQkoxDDAKBgNVBAoTA0FMSTEP
 * MA0GA1UECxMGQUxJWVVOMQ0wCwYDVQQDEwR0ZXN0MR8wHQYJKoZIhvcNAQkBFhB0
 * ZXN0QGhvdG1haWwuY29tMB4XDTE0MTEyNDA2MDQyNVoXDTI0MTEyMTA2MDQyNVow
 * djELMAkGA1UEBhMCQ04xCzAJBgNVBAgTAkJKMQswCQYDVQQHEwJCSjEMMAoGA1UE
 * ChMDQUxJMQ8wDQYDVQQLEwZBTElZVU4xDTALBgNVBAMTBHRlc3QxHzAdBgkqhkiG
 * 9w0BCQEWEHRlc3RAaG90bWFpbC5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJ
 * AoGBAM7SS3e9+Nj0HKAsRuIDNSsS3UK6b+62YQb2uuhKrp1HMrOx61WSDR2qkAnB
 * coG00Uz38EE+9DLYNUVQBK7aSgLP5M1Ak4wr4GqGyCgjejzzh3DshUzLCCy2rook
 * KOyRTlPX+Q5l7rE1fcSNzgepcae5i2sE1XXXzLRIDIvQxcspAgMBAAGjgdswgdgw
 * HQYDVR0OBBYEFBdy+OuMsvbkV7R14f0OyoLoh2z4MIGoBgNVHSMEgaAwgZ2AFBdy
 * +OuMsvbkV7R14f0OyoLoh2z4oXqkeDB2MQswCQYDVQQGEwJDTjELMAkGA1UECBMC
 * QkoxCzAJBgNVBAcTAkJKMQwwCgYDVQQKEwNBTEkxDzANBgNVBAsTBkFMSVlVTjEN
 * MAsGA1UEAxMEdGVzdDEfMB0GCSqGSIb3DQEJARYQdGVzdEBob3RtYWlsLmNvbYIJ
 * AJn3ox4K13PoMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQEFBQADgYEAY7KOsnyT
 * cQzfhiiG7ASjiPakw5wXoycHt5GCvLG5htp2TKVzgv9QTliA3gtfv6oV4zRZx7X1
 * Ofi6hVgErtHaXJheuPVeW6eAW8mHBoEfvDAfU3y9waYrtUevSl07643bzKL6v+Qd
 * DUBTxOAvSYfXTtI90EAxEG/bJJyOm5LqoiA=
 * -----END CERTIFICATE-----
 * """
 * private_key = config.get("privateKey")
 * if private_key is None:
 *     private_key = """-----BEGIN RSA PRIVATE KEY-----
 * MIICXAIBAAKBgQDO0kt3vfjY9BygLEbiAzUrEt1Cum/utmEG9rroSq6dRzKzsetV
 * kg0dqpAJwXKBtNFM9/BBPvQy2DVFUASu2koCz+TNQJOMK+BqhsgoI3o884dw7IVM
 * ywgstq6KJCjskU5T1/kOZe6xNX3Ejc4HqXGnuYtrBNV118y0SAyL0MXLKQIDAQAB
 * AoGAfe3NxbsGKhN42o4bGsKZPQDfeCHMxayGp5bTd10BtQIE/ST4BcJH+ihAS7Bd
 * 6FwQlKzivNd4GP1MckemklCXfsVckdL94e8ZbJl23GdWul3v8V+KndJHqv5zVJmP
 * hwWoKimwIBTb2s0ctVryr2f18N4hhyFw1yGp0VxclGHkjgECQQD9CvllsnOwHpP4
 * MdrDHbdb29QrobKyKW8pPcDd+sth+kP6Y8MnCVuAKXCKj5FeIsgVtfluPOsZjPzz
 * 71QQWS1dAkEA0T0KXO8gaBQwJhIoo/w6hy5JGZnrNSpOPp5xvJuMAafs2eyvmhJm
 * Ev9SN/Pf2VYa1z6FEnBaLOVD6hf6YQIsPQJAX/CZPoW6dzwgvimo1/GcY6eleiWE
 * qygqjWhsh71e/3bz7yuEAnj5yE3t7Zshcp+dXR3xxGo0eSuLfLFxHgGxwQJAAxf8
 * 9DzQ5NkPkTCJi0sqbl8/03IUKTgT6hcbpWdDXa7m8J3wRr3o5nUB+TPQ5nzAbthM
 * zWX931YQeACcwhxvHQJBAN5mTzzJD4w4Ma6YTaNHyXakdYfyAWrOkPIWZxfhMfXe
 * DrlNdiysTI4Dd1dLeErVpjsckAaOW/JDG5PCSwkaMxk=
 * -----END RSA PRIVATE KEY-----
 * """
 * default = alicloud.fc.V3CustomDomain("default",
 *     custom_domain_name="flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net",
 *     route_config={
 *         "routes": [
 *             {
 *                 "function_name": function_name1,
 *                 "rewrite_config": {
 *                     "regex_rules": [
 *                         {
 *                             "match": "/api/*",
 *                             "replacement": "$1",
 *                         },
 *                         {
 *                             "match": "/api1/*",
 *                             "replacement": "$1",
 *                         },
 *                         {
 *                             "match": "/api2/*",
 *                             "replacement": "$1",
 *                         },
 *                     ],
 *                     "wildcard_rules": [
 *                         {
 *                             "match": "^/api1/.+?/(.*)",
 *                             "replacement": "/api/v1/$1",
 *                         },
 *                         {
 *                             "match": "^/api2/.+?/(.*)",
 *                             "replacement": "/api/v2/$1",
 *                         },
 *                         {
 *                             "match": "^/api2/.+?/(.*)",
 *                             "replacement": "/api/v3/$1",
 *                         },
 *                     ],
 *                     "equal_rules": [
 *                         {
 *                             "match": "/old",
 *                             "replacement": "/new",
 *                         },
 *                         {
 *                             "replacement": "/new1",
 *                             "match": "/old1",
 *                         },
 *                         {
 *                             "match": "/old2",
 *                             "replacement": "/new2",
 *                         },
 *                     ],
 *                 },
 *                 "methods": [
 *                     "GET",
 *                     "POST",
 *                     "DELETE",
 *                     "HEAD",
 *                 ],
 *                 "path": "/a",
 *                 "qualifier": "LATEST",
 *             },
 *             {
 *                 "function_name": function_name1,
 *                 "methods": ["GET"],
 *                 "path": "/b",
 *                 "qualifier": "LATEST",
 *             },
 *             {
 *                 "function_name": function_name1,
 *                 "methods": ["POST"],
 *                 "path": "/c",
 *                 "qualifier": "1",
 *             },
 *         ],
 *     },
 *     auth_config={
 *         "auth_type": "jwt",
 *         "auth_info": auth_config,
 *     },
 *     protocol="HTTP,HTTPS",
 *     cert_config={
 *         "cert_name": "cert-name",
 *         "certificate": certificate,
 *         "private_key": private_key,
 *     },
 *     tls_config={
 *         "cipher_suites": [
 *             "TLS_RSA_WITH_AES_128_CBC_SHA",
 *             "TLS_RSA_WITH_AES_256_CBC_SHA",
 *             "TLS_RSA_WITH_AES_128_GCM_SHA256",
 *             "TLS_RSA_WITH_AES_256_GCM_SHA384",
 *         ],
 *         "max_version": "TLSv1.3",
 *         "min_version": "TLSv1.0",
 *     },
 *     waf_config={
 *         "enable_waf": False,
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AliCloud = Pulumi.AliCloud;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var name = config.Get("name") ?? "flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net";
 *     var functionName1 = config.Get("functionName1") ?? "terraform-custom-domain-t1";
 *     var authConfig = config.Get("authConfig") ?? @"{
 *     ""jwks"": {
 *         ""keys"": [
 *             {
 *                 ""p"": ""8AdUVeldoE4LueFuzEF_C8tvJ7NhlkzS58Gz9KJTPXPr5DADSUVLWJCr5OdFE79q513SneT0UhGo-JfQ1lNMoNv5-YZ1AxIo9fZUEPIe-KyX9ttaglpzCAUE3TeKdm5J-_HZQzBPKbyUwJHAILNgB2-4IBZZwK7LAfbmfi9TmFM"",
 *                 ""kty"": ""RSA"",
 *                 ""q"": ""x8m5ydXwC8AAp9I-hOnUAx6yQJz1Nx-jXPCfn--XdHpJuNcuwRQsuUCSRQs_h3SoCI3qZZdzswQnPrtHFxgUJtQFuMj-QZpyMnebDb81rmczl2KPVUtaVDVagJEF6U9Ov3PfrLhvHUEv5u7p6s4Z6maBUaByfFlhEVPv4_ao8Us"",
 *                 ""d"": ""bjIQAKD2e65gwJ38_Sqq_EmLFuMMey3gjDv1bSCHFH8fyONJTq-utrZfvspz6EegRwW2mSHW9kq87hRwIBW9y7ED5N4KG5gHDjyh57BRE0SKv0Dz1igtKLyp-nl8-aHc1DbONwr1d7tZfFt255TxIN8cPTakXOp2Av_ztql_JotVUGK8eHmXNJFlvq5tc180sKWMHNSNsCUhQgcB1TWb_gwcqxdsIWPsLZI491XKeTGQ98J7z5h6R1cTC97lfJZ0vNtJahd2jHd3WfTUDj5-untMKyZpYYak2Vr8xtFz8H6Q5Rsz8uX_7gtEqYH2CMjPdbXcebrnD1igRSJMYiP0lQ"",
 *                 ""e"": ""AQAB"",
 *                 ""use"": ""sig"",
 *                 ""qi"": ""MTCCRu8AcvvjbHms7V_sDFO7wX0YNyvOJAAbuTmHvQbJ0NDeDta-f-hi8cjkMk7Fpk2hej158E5gDyO62UG99wHZSbmHT34MvIdmhQ5mnbL-5KK9rxde0nayO1ebGepD_GJThPAg9iskzeWpCg5X2etNo2bHoG_ZLQGXj2BQ1VM"",
 *                 ""dp"": ""J4_ttKNcTTnP8PlZO81n1VfYoGCOqylKceyZbq76rVxX-yp2wDLtslFWI8qCtjiMtEnglynPo19JzH-pakocjT70us4Qp0rs-W16ebiOpko8WfHZvzaNUzsQjC3FYrPW-fHo74wc4DI3Cm57jmhCYbdmT9OfQ4UL7Oz3HMFMNAU"",
 *                 ""alg"": ""RS256"",
 *                 ""dq"": ""H4-VgvYB-sk1EU3cRIDv1iJWRHDHKBMeaoM0pD5kLalX1hRgNW4rdoRl1vRk79AU720D11Kqm2APlxBctaA_JrcdxEg0KkbsvV45p11KbKeu9b5DKFVECsN27ZJ7XZUCuqnibtWf7_4pRBD_8PDoFShmS2_ORiiUdflNjzSbEas"",
 *                 ""n"": ""u1LWgoomekdOMfB1lEe96OHehd4XRNCbZRm96RqwOYTTc28Sc_U5wKV2umDzolfoI682ct2BNnRRahYgZPhbOCzHYM6i8sRXjz9Ghx3QHw9zrYACtArwQxrTFiejbfzDPGdPrMQg7T8wjtLtkSyDmCzeXpbIdwmxuLyt_ahLfHelr94kEksMDa42V4Fi5bMW4cCLjlEKzBEHGmFdT8UbLPCvpgsM84JK63e5ifdeI9NdadbC8ZMiR--dFCujT7AgRRyMzxgdn2l-nZJ2ZaYzbLUtAW5_U2kfRVkDNa8d1g__2V5zjU6nfLJ1S2MoXMgRgDPeHpEehZVu2kNaSFvDUQ""
 *             }
 *         ]
 *     },
 *     ""tokenLookup"": ""header:auth"",
 *     ""claimPassBy"": ""header:name:name""
 * }
 * ";
 *     var certificate = config.Get("certificate") ?? @"-----BEGIN CERTIFICATE-----
 * MIIDRjCCAq+gAwIBAgIJAJn3ox4K13PoMA0GCSqGSIb3DQEBBQUAMHYxCzAJBgNV
 * BAYTAkNOMQswCQYDVQQIEwJCSjELMAkGA1UEBxMCQkoxDDAKBgNVBAoTA0FMSTEP
 * MA0GA1UECxMGQUxJWVVOMQ0wCwYDVQQDEwR0ZXN0MR8wHQYJKoZIhvcNAQkBFhB0
 * ZXN0QGhvdG1haWwuY29tMB4XDTE0MTEyNDA2MDQyNVoXDTI0MTEyMTA2MDQyNVow
 * djELMAkGA1UEBhMCQ04xCzAJBgNVBAgTAkJKMQswCQYDVQQHEwJCSjEMMAoGA1UE
 * ChMDQUxJMQ8wDQYDVQQLEwZBTElZVU4xDTALBgNVBAMTBHRlc3QxHzAdBgkqhkiG
 * 9w0BCQEWEHRlc3RAaG90bWFpbC5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJ
 * AoGBAM7SS3e9+Nj0HKAsRuIDNSsS3UK6b+62YQb2uuhKrp1HMrOx61WSDR2qkAnB
 * coG00Uz38EE+9DLYNUVQBK7aSgLP5M1Ak4wr4GqGyCgjejzzh3DshUzLCCy2rook
 * KOyRTlPX+Q5l7rE1fcSNzgepcae5i2sE1XXXzLRIDIvQxcspAgMBAAGjgdswgdgw
 * HQYDVR0OBBYEFBdy+OuMsvbkV7R14f0OyoLoh2z4MIGoBgNVHSMEgaAwgZ2AFBdy
 * +OuMsvbkV7R14f0OyoLoh2z4oXqkeDB2MQswCQYDVQQGEwJDTjELMAkGA1UECBMC
 * QkoxCzAJBgNVBAcTAkJKMQwwCgYDVQQKEwNBTEkxDzANBgNVBAsTBkFMSVlVTjEN
 * MAsGA1UEAxMEdGVzdDEfMB0GCSqGSIb3DQEJARYQdGVzdEBob3RtYWlsLmNvbYIJ
 * AJn3ox4K13PoMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQEFBQADgYEAY7KOsnyT
 * cQzfhiiG7ASjiPakw5wXoycHt5GCvLG5htp2TKVzgv9QTliA3gtfv6oV4zRZx7X1
 * Ofi6hVgErtHaXJheuPVeW6eAW8mHBoEfvDAfU3y9waYrtUevSl07643bzKL6v+Qd
 * DUBTxOAvSYfXTtI90EAxEG/bJJyOm5LqoiA=
 * -----END CERTIFICATE-----
 * ";
 *     var privateKey = config.Get("privateKey") ?? @"-----BEGIN RSA PRIVATE KEY-----
 * MIICXAIBAAKBgQDO0kt3vfjY9BygLEbiAzUrEt1Cum/utmEG9rroSq6dRzKzsetV
 * kg0dqpAJwXKBtNFM9/BBPvQy2DVFUASu2koCz+TNQJOMK+BqhsgoI3o884dw7IVM
 * ywgstq6KJCjskU5T1/kOZe6xNX3Ejc4HqXGnuYtrBNV118y0SAyL0MXLKQIDAQAB
 * AoGAfe3NxbsGKhN42o4bGsKZPQDfeCHMxayGp5bTd10BtQIE/ST4BcJH+ihAS7Bd
 * 6FwQlKzivNd4GP1MckemklCXfsVckdL94e8ZbJl23GdWul3v8V+KndJHqv5zVJmP
 * hwWoKimwIBTb2s0ctVryr2f18N4hhyFw1yGp0VxclGHkjgECQQD9CvllsnOwHpP4
 * MdrDHbdb29QrobKyKW8pPcDd+sth+kP6Y8MnCVuAKXCKj5FeIsgVtfluPOsZjPzz
 * 71QQWS1dAkEA0T0KXO8gaBQwJhIoo/w6hy5JGZnrNSpOPp5xvJuMAafs2eyvmhJm
 * Ev9SN/Pf2VYa1z6FEnBaLOVD6hf6YQIsPQJAX/CZPoW6dzwgvimo1/GcY6eleiWE
 * qygqjWhsh71e/3bz7yuEAnj5yE3t7Zshcp+dXR3xxGo0eSuLfLFxHgGxwQJAAxf8
 * 9DzQ5NkPkTCJi0sqbl8/03IUKTgT6hcbpWdDXa7m8J3wRr3o5nUB+TPQ5nzAbthM
 * zWX931YQeACcwhxvHQJBAN5mTzzJD4w4Ma6YTaNHyXakdYfyAWrOkPIWZxfhMfXe
 * DrlNdiysTI4Dd1dLeErVpjsckAaOW/JDG5PCSwkaMxk=
 * -----END RSA PRIVATE KEY-----
 * ";
 *     var @default = new AliCloud.FC.V3CustomDomain("default", new()
 *     {
 *         CustomDomainName = "flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net",
 *         RouteConfig = new AliCloud.FC.Inputs.V3CustomDomainRouteConfigArgs
 *         {
 *             Routes = new[]
 *             {
 *                 new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteArgs
 *                 {
 *                     FunctionName = functionName1,
 *                     RewriteConfig = new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteRewriteConfigArgs
 *                     {
 *                         RegexRules = new[]
 *                         {
 *                             new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteRewriteConfigRegexRuleArgs
 *                             {
 *                                 Match = "/api/*",
 *                                 Replacement = "$1",
 *                             },
 *                             new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteRewriteConfigRegexRuleArgs
 *                             {
 *                                 Match = "/api1/*",
 *                                 Replacement = "$1",
 *                             },
 *                             new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteRewriteConfigRegexRuleArgs
 *                             {
 *                                 Match = "/api2/*",
 *                                 Replacement = "$1",
 *                             },
 *                         },
 *                         WildcardRules = new[]
 *                         {
 *                             new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteRewriteConfigWildcardRuleArgs
 *                             {
 *                                 Match = "^/api1/.+?/(.*)",
 *                                 Replacement = "/api/v1/$1",
 *                             },
 *                             new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteRewriteConfigWildcardRuleArgs
 *                             {
 *                                 Match = "^/api2/.+?/(.*)",
 *                                 Replacement = "/api/v2/$1",
 *                             },
 *                             new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteRewriteConfigWildcardRuleArgs
 *                             {
 *                                 Match = "^/api2/.+?/(.*)",
 *                                 Replacement = "/api/v3/$1",
 *                             },
 *                         },
 *                         EqualRules = new[]
 *                         {
 *                             new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteRewriteConfigEqualRuleArgs
 *                             {
 *                                 Match = "/old",
 *                                 Replacement = "/new",
 *                             },
 *                             new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteRewriteConfigEqualRuleArgs
 *                             {
 *                                 Replacement = "/new1",
 *                                 Match = "/old1",
 *                             },
 *                             new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteRewriteConfigEqualRuleArgs
 *                             {
 *                                 Match = "/old2",
 *                                 Replacement = "/new2",
 *                             },
 *                         },
 *                     },
 *                     Methods = new[]
 *                     {
 *                         "GET",
 *                         "POST",
 *                         "DELETE",
 *                         "HEAD",
 *                     },
 *                     Path = "/a",
 *                     Qualifier = "LATEST",
 *                 },
 *                 new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteArgs
 *                 {
 *                     FunctionName = functionName1,
 *                     Methods = new[]
 *                     {
 *                         "GET",
 *                     },
 *                     Path = "/b",
 *                     Qualifier = "LATEST",
 *                 },
 *                 new AliCloud.FC.Inputs.V3CustomDomainRouteConfigRouteArgs
 *                 {
 *                     FunctionName = functionName1,
 *                     Methods = new[]
 *                     {
 *                         "POST",
 *                     },
 *                     Path = "/c",
 *                     Qualifier = "1",
 *                 },
 *             },
 *         },
 *         AuthConfig = new AliCloud.FC.Inputs.V3CustomDomainAuthConfigArgs
 *         {
 *             AuthType = "jwt",
 *             AuthInfo = authConfig,
 *         },
 *         Protocol = "HTTP,HTTPS",
 *         CertConfig = new AliCloud.FC.Inputs.V3CustomDomainCertConfigArgs
 *         {
 *             CertName = "cert-name",
 *             Certificate = certificate,
 *             PrivateKey = privateKey,
 *         },
 *         TlsConfig = new AliCloud.FC.Inputs.V3CustomDomainTlsConfigArgs
 *         {
 *             CipherSuites = new[]
 *             {
 *                 "TLS_RSA_WITH_AES_128_CBC_SHA",
 *                 "TLS_RSA_WITH_AES_256_CBC_SHA",
 *                 "TLS_RSA_WITH_AES_128_GCM_SHA256",
 *                 "TLS_RSA_WITH_AES_256_GCM_SHA384",
 *             },
 *             MaxVersion = "TLSv1.3",
 *             MinVersion = "TLSv1.0",
 *         },
 *         WafConfig = new AliCloud.FC.Inputs.V3CustomDomainWafConfigArgs
 *         {
 *             EnableWaf = false,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/fc"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		cfg := config.New(ctx, "")
 * 		name := "flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net"
 * 		if param := cfg.Get("name"); param != "" {
 * 			name = param
 * 		}
 * 		functionName1 := "terraform-custom-domain-t1"
 * 		if param := cfg.Get("functionName1"); param != "" {
 * 			functionName1 = param
 * 		}
 * 		authConfig := `{
 *     "jwks": {
 *         "keys": [
 *             {
 *                 "p": "8AdUVeldoE4LueFuzEF_C8tvJ7NhlkzS58Gz9KJTPXPr5DADSUVLWJCr5OdFE79q513SneT0UhGo-JfQ1lNMoNv5-YZ1AxIo9fZUEPIe-KyX9ttaglpzCAUE3TeKdm5J-_HZQzBPKbyUwJHAILNgB2-4IBZZwK7LAfbmfi9TmFM",
 *                 "kty": "RSA",
 *                 "q": "x8m5ydXwC8AAp9I-hOnUAx6yQJz1Nx-jXPCfn--XdHpJuNcuwRQsuUCSRQs_h3SoCI3qZZdzswQnPrtHFxgUJtQFuMj-QZpyMnebDb81rmczl2KPVUtaVDVagJEF6U9Ov3PfrLhvHUEv5u7p6s4Z6maBUaByfFlhEVPv4_ao8Us",
 *                 "d": "bjIQAKD2e65gwJ38_Sqq_EmLFuMMey3gjDv1bSCHFH8fyONJTq-utrZfvspz6EegRwW2mSHW9kq87hRwIBW9y7ED5N4KG5gHDjyh57BRE0SKv0Dz1igtKLyp-nl8-aHc1DbONwr1d7tZfFt255TxIN8cPTakXOp2Av_ztql_JotVUGK8eHmXNJFlvq5tc180sKWMHNSNsCUhQgcB1TWb_gwcqxdsIWPsLZI491XKeTGQ98J7z5h6R1cTC97lfJZ0vNtJahd2jHd3WfTUDj5-untMKyZpYYak2Vr8xtFz8H6Q5Rsz8uX_7gtEqYH2CMjPdbXcebrnD1igRSJMYiP0lQ",
 *                 "e": "AQAB",
 *                 "use": "sig",
 *                 "qi": "MTCCRu8AcvvjbHms7V_sDFO7wX0YNyvOJAAbuTmHvQbJ0NDeDta-f-hi8cjkMk7Fpk2hej158E5gDyO62UG99wHZSbmHT34MvIdmhQ5mnbL-5KK9rxde0nayO1ebGepD_GJThPAg9iskzeWpCg5X2etNo2bHoG_ZLQGXj2BQ1VM",
 *                 "dp": "J4_ttKNcTTnP8PlZO81n1VfYoGCOqylKceyZbq76rVxX-yp2wDLtslFWI8qCtjiMtEnglynPo19JzH-pakocjT70us4Qp0rs-W16ebiOpko8WfHZvzaNUzsQjC3FYrPW-fHo74wc4DI3Cm57jmhCYbdmT9OfQ4UL7Oz3HMFMNAU",
 *                 "alg": "RS256",
 *                 "dq": "H4-VgvYB-sk1EU3cRIDv1iJWRHDHKBMeaoM0pD5kLalX1hRgNW4rdoRl1vRk79AU720D11Kqm2APlxBctaA_JrcdxEg0KkbsvV45p11KbKeu9b5DKFVECsN27ZJ7XZUCuqnibtWf7_4pRBD_8PDoFShmS2_ORiiUdflNjzSbEas",
 *                 "n": "u1LWgoomekdOMfB1lEe96OHehd4XRNCbZRm96RqwOYTTc28Sc_U5wKV2umDzolfoI682ct2BNnRRahYgZPhbOCzHYM6i8sRXjz9Ghx3QHw9zrYACtArwQxrTFiejbfzDPGdPrMQg7T8wjtLtkSyDmCzeXpbIdwmxuLyt_ahLfHelr94kEksMDa42V4Fi5bMW4cCLjlEKzBEHGmFdT8UbLPCvpgsM84JK63e5ifdeI9NdadbC8ZMiR--dFCujT7AgRRyMzxgdn2l-nZJ2ZaYzbLUtAW5_U2kfRVkDNa8d1g__2V5zjU6nfLJ1S2MoXMgRgDPeHpEehZVu2kNaSFvDUQ"
 *             }
 *         ]
 *     },
 *     "tokenLookup": "header:auth",
 *     "claimPassBy": "header:name:name"
 * }
 * `
 * 		if param := cfg.Get("authConfig"); param != "" {
 * 			authConfig = param
 * 		}
 * 		certificate := `-----BEGIN CERTIFICATE-----
 * MIIDRjCCAq+gAwIBAgIJAJn3ox4K13PoMA0GCSqGSIb3DQEBBQUAMHYxCzAJBgNV
 * BAYTAkNOMQswCQYDVQQIEwJCSjELMAkGA1UEBxMCQkoxDDAKBgNVBAoTA0FMSTEP
 * MA0GA1UECxMGQUxJWVVOMQ0wCwYDVQQDEwR0ZXN0MR8wHQYJKoZIhvcNAQkBFhB0
 * ZXN0QGhvdG1haWwuY29tMB4XDTE0MTEyNDA2MDQyNVoXDTI0MTEyMTA2MDQyNVow
 * djELMAkGA1UEBhMCQ04xCzAJBgNVBAgTAkJKMQswCQYDVQQHEwJCSjEMMAoGA1UE
 * ChMDQUxJMQ8wDQYDVQQLEwZBTElZVU4xDTALBgNVBAMTBHRlc3QxHzAdBgkqhkiG
 * 9w0BCQEWEHRlc3RAaG90bWFpbC5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJ
 * AoGBAM7SS3e9+Nj0HKAsRuIDNSsS3UK6b+62YQb2uuhKrp1HMrOx61WSDR2qkAnB
 * coG00Uz38EE+9DLYNUVQBK7aSgLP5M1Ak4wr4GqGyCgjejzzh3DshUzLCCy2rook
 * KOyRTlPX+Q5l7rE1fcSNzgepcae5i2sE1XXXzLRIDIvQxcspAgMBAAGjgdswgdgw
 * HQYDVR0OBBYEFBdy+OuMsvbkV7R14f0OyoLoh2z4MIGoBgNVHSMEgaAwgZ2AFBdy
 * +OuMsvbkV7R14f0OyoLoh2z4oXqkeDB2MQswCQYDVQQGEwJDTjELMAkGA1UECBMC
 * QkoxCzAJBgNVBAcTAkJKMQwwCgYDVQQKEwNBTEkxDzANBgNVBAsTBkFMSVlVTjEN
 * MAsGA1UEAxMEdGVzdDEfMB0GCSqGSIb3DQEJARYQdGVzdEBob3RtYWlsLmNvbYIJ
 * AJn3ox4K13PoMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQEFBQADgYEAY7KOsnyT
 * cQzfhiiG7ASjiPakw5wXoycHt5GCvLG5htp2TKVzgv9QTliA3gtfv6oV4zRZx7X1
 * Ofi6hVgErtHaXJheuPVeW6eAW8mHBoEfvDAfU3y9waYrtUevSl07643bzKL6v+Qd
 * DUBTxOAvSYfXTtI90EAxEG/bJJyOm5LqoiA=
 * -----END CERTIFICATE-----
 * `
 * 		if param := cfg.Get("certificate"); param != "" {
 * 			certificate = param
 * 		}
 * 		privateKey := `-----BEGIN RSA PRIVATE KEY-----
 * MIICXAIBAAKBgQDO0kt3vfjY9BygLEbiAzUrEt1Cum/utmEG9rroSq6dRzKzsetV
 * kg0dqpAJwXKBtNFM9/BBPvQy2DVFUASu2koCz+TNQJOMK+BqhsgoI3o884dw7IVM
 * ywgstq6KJCjskU5T1/kOZe6xNX3Ejc4HqXGnuYtrBNV118y0SAyL0MXLKQIDAQAB
 * AoGAfe3NxbsGKhN42o4bGsKZPQDfeCHMxayGp5bTd10BtQIE/ST4BcJH+ihAS7Bd
 * 6FwQlKzivNd4GP1MckemklCXfsVckdL94e8ZbJl23GdWul3v8V+KndJHqv5zVJmP
 * hwWoKimwIBTb2s0ctVryr2f18N4hhyFw1yGp0VxclGHkjgECQQD9CvllsnOwHpP4
 * MdrDHbdb29QrobKyKW8pPcDd+sth+kP6Y8MnCVuAKXCKj5FeIsgVtfluPOsZjPzz
 * 71QQWS1dAkEA0T0KXO8gaBQwJhIoo/w6hy5JGZnrNSpOPp5xvJuMAafs2eyvmhJm
 * Ev9SN/Pf2VYa1z6FEnBaLOVD6hf6YQIsPQJAX/CZPoW6dzwgvimo1/GcY6eleiWE
 * qygqjWhsh71e/3bz7yuEAnj5yE3t7Zshcp+dXR3xxGo0eSuLfLFxHgGxwQJAAxf8
 * 9DzQ5NkPkTCJi0sqbl8/03IUKTgT6hcbpWdDXa7m8J3wRr3o5nUB+TPQ5nzAbthM
 * zWX931YQeACcwhxvHQJBAN5mTzzJD4w4Ma6YTaNHyXakdYfyAWrOkPIWZxfhMfXe
 * DrlNdiysTI4Dd1dLeErVpjsckAaOW/JDG5PCSwkaMxk=
 * -----END RSA PRIVATE KEY-----
 * `
 * 		if param := cfg.Get("privateKey"); param != "" {
 * 			privateKey = param
 * 		}
 * 		_, err := fc.NewV3CustomDomain(ctx, "default", &fc.V3CustomDomainArgs{
 * 			CustomDomainName: pulumi.String("flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net"),
 * 			RouteConfig: &fc.V3CustomDomainRouteConfigArgs{
 * 				Routes: fc.V3CustomDomainRouteConfigRouteArray{
 * 					&fc.V3CustomDomainRouteConfigRouteArgs{
 * 						FunctionName: pulumi.String(functionName1),
 * 						RewriteConfig: &fc.V3CustomDomainRouteConfigRouteRewriteConfigArgs{
 * 							RegexRules: fc.V3CustomDomainRouteConfigRouteRewriteConfigRegexRuleArray{
 * 								&fc.V3CustomDomainRouteConfigRouteRewriteConfigRegexRuleArgs{
 * 									Match:       pulumi.String("/api/*"),
 * 									Replacement: pulumi.String("$1"),
 * 								},
 * 								&fc.V3CustomDomainRouteConfigRouteRewriteConfigRegexRuleArgs{
 * 									Match:       pulumi.String("/api1/*"),
 * 									Replacement: pulumi.String("$1"),
 * 								},
 * 								&fc.V3CustomDomainRouteConfigRouteRewriteConfigRegexRuleArgs{
 * 									Match:       pulumi.String("/api2/*"),
 * 									Replacement: pulumi.String("$1"),
 * 								},
 * 							},
 * 							WildcardRules: fc.V3CustomDomainRouteConfigRouteRewriteConfigWildcardRuleArray{
 * 								&fc.V3CustomDomainRouteConfigRouteRewriteConfigWildcardRuleArgs{
 * 									Match:       pulumi.String("^/api1/.+?/(.*)"),
 * 									Replacement: pulumi.String("/api/v1/$1"),
 * 								},
 * 								&fc.V3CustomDomainRouteConfigRouteRewriteConfigWildcardRuleArgs{
 * 									Match:       pulumi.String("^/api2/.+?/(.*)"),
 * 									Replacement: pulumi.String("/api/v2/$1"),
 * 								},
 * 								&fc.V3CustomDomainRouteConfigRouteRewriteConfigWildcardRuleArgs{
 * 									Match:       pulumi.String("^/api2/.+?/(.*)"),
 * 									Replacement: pulumi.String("/api/v3/$1"),
 * 								},
 * 							},
 * 							EqualRules: fc.V3CustomDomainRouteConfigRouteRewriteConfigEqualRuleArray{
 * 								&fc.V3CustomDomainRouteConfigRouteRewriteConfigEqualRuleArgs{
 * 									Match:       pulumi.String("/old"),
 * 									Replacement: pulumi.String("/new"),
 * 								},
 * 								&fc.V3CustomDomainRouteConfigRouteRewriteConfigEqualRuleArgs{
 * 									Replacement: pulumi.String("/new1"),
 * 									Match:       pulumi.String("/old1"),
 * 								},
 * 								&fc.V3CustomDomainRouteConfigRouteRewriteConfigEqualRuleArgs{
 * 									Match:       pulumi.String("/old2"),
 * 									Replacement: pulumi.String("/new2"),
 * 								},
 * 							},
 * 						},
 * 						Methods: pulumi.StringArray{
 * 							pulumi.String("GET"),
 * 							pulumi.String("POST"),
 * 							pulumi.String("DELETE"),
 * 							pulumi.String("HEAD"),
 * 						},
 * 						Path:      pulumi.String("/a"),
 * 						Qualifier: pulumi.String("LATEST"),
 * 					},
 * 					&fc.V3CustomDomainRouteConfigRouteArgs{
 * 						FunctionName: pulumi.String(functionName1),
 * 						Methods: pulumi.StringArray{
 * 							pulumi.String("GET"),
 * 						},
 * 						Path:      pulumi.String("/b"),
 * 						Qualifier: pulumi.String("LATEST"),
 * 					},
 * 					&fc.V3CustomDomainRouteConfigRouteArgs{
 * 						FunctionName: pulumi.String(functionName1),
 * 						Methods: pulumi.StringArray{
 * 							pulumi.String("POST"),
 * 						},
 * 						Path:      pulumi.String("/c"),
 * 						Qualifier: pulumi.String("1"),
 * 					},
 * 				},
 * 			},
 * 			AuthConfig: &fc.V3CustomDomainAuthConfigArgs{
 * 				AuthType: pulumi.String("jwt"),
 * 				AuthInfo: pulumi.String(authConfig),
 * 			},
 * 			Protocol: pulumi.String("HTTP,HTTPS"),
 * 			CertConfig: &fc.V3CustomDomainCertConfigArgs{
 * 				CertName:    pulumi.String("cert-name"),
 * 				Certificate: pulumi.String(certificate),
 * 				PrivateKey:  pulumi.String(privateKey),
 * 			},
 * 			TlsConfig: &fc.V3CustomDomainTlsConfigArgs{
 * 				CipherSuites: pulumi.StringArray{
 * 					pulumi.String("TLS_RSA_WITH_AES_128_CBC_SHA"),
 * 					pulumi.String("TLS_RSA_WITH_AES_256_CBC_SHA"),
 * 					pulumi.String("TLS_RSA_WITH_AES_128_GCM_SHA256"),
 * 					pulumi.String("TLS_RSA_WITH_AES_256_GCM_SHA384"),
 * 				},
 * 				MaxVersion: pulumi.String("TLSv1.3"),
 * 				MinVersion: pulumi.String("TLSv1.0"),
 * 			},
 * 			WafConfig: &fc.V3CustomDomainWafConfigArgs{
 * 				EnableWaf: pulumi.Bool(false),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.alicloud.fc.V3CustomDomain;
 * import com.pulumi.alicloud.fc.V3CustomDomainArgs;
 * import com.pulumi.alicloud.fc.inputs.V3CustomDomainRouteConfigArgs;
 * import com.pulumi.alicloud.fc.inputs.V3CustomDomainAuthConfigArgs;
 * import com.pulumi.alicloud.fc.inputs.V3CustomDomainCertConfigArgs;
 * import com.pulumi.alicloud.fc.inputs.V3CustomDomainTlsConfigArgs;
 * import com.pulumi.alicloud.fc.inputs.V3CustomDomainWafConfigArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var name = config.get("name").orElse("flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net");
 *         final var functionName1 = config.get("functionName1").orElse("terraform-custom-domain-t1");
 *         final var authConfig = config.get("authConfig").orElse("""
 * {
 *     "jwks": {
 *         "keys": [
 *             {
 *                 "p": "8AdUVeldoE4LueFuzEF_C8tvJ7NhlkzS58Gz9KJTPXPr5DADSUVLWJCr5OdFE79q513SneT0UhGo-JfQ1lNMoNv5-YZ1AxIo9fZUEPIe-KyX9ttaglpzCAUE3TeKdm5J-_HZQzBPKbyUwJHAILNgB2-4IBZZwK7LAfbmfi9TmFM",
 *                 "kty": "RSA",
 *                 "q": "x8m5ydXwC8AAp9I-hOnUAx6yQJz1Nx-jXPCfn--XdHpJuNcuwRQsuUCSRQs_h3SoCI3qZZdzswQnPrtHFxgUJtQFuMj-QZpyMnebDb81rmczl2KPVUtaVDVagJEF6U9Ov3PfrLhvHUEv5u7p6s4Z6maBUaByfFlhEVPv4_ao8Us",
 *                 "d": "bjIQAKD2e65gwJ38_Sqq_EmLFuMMey3gjDv1bSCHFH8fyONJTq-utrZfvspz6EegRwW2mSHW9kq87hRwIBW9y7ED5N4KG5gHDjyh57BRE0SKv0Dz1igtKLyp-nl8-aHc1DbONwr1d7tZfFt255TxIN8cPTakXOp2Av_ztql_JotVUGK8eHmXNJFlvq5tc180sKWMHNSNsCUhQgcB1TWb_gwcqxdsIWPsLZI491XKeTGQ98J7z5h6R1cTC97lfJZ0vNtJahd2jHd3WfTUDj5-untMKyZpYYak2Vr8xtFz8H6Q5Rsz8uX_7gtEqYH2CMjPdbXcebrnD1igRSJMYiP0lQ",
 *                 "e": "AQAB",
 *                 "use": "sig",
 *                 "qi": "MTCCRu8AcvvjbHms7V_sDFO7wX0YNyvOJAAbuTmHvQbJ0NDeDta-f-hi8cjkMk7Fpk2hej158E5gDyO62UG99wHZSbmHT34MvIdmhQ5mnbL-5KK9rxde0nayO1ebGepD_GJThPAg9iskzeWpCg5X2etNo2bHoG_ZLQGXj2BQ1VM",
 *                 "dp": "J4_ttKNcTTnP8PlZO81n1VfYoGCOqylKceyZbq76rVxX-yp2wDLtslFWI8qCtjiMtEnglynPo19JzH-pakocjT70us4Qp0rs-W16ebiOpko8WfHZvzaNUzsQjC3FYrPW-fHo74wc4DI3Cm57jmhCYbdmT9OfQ4UL7Oz3HMFMNAU",
 *                 "alg": "RS256",
 *                 "dq": "H4-VgvYB-sk1EU3cRIDv1iJWRHDHKBMeaoM0pD5kLalX1hRgNW4rdoRl1vRk79AU720D11Kqm2APlxBctaA_JrcdxEg0KkbsvV45p11KbKeu9b5DKFVECsN27ZJ7XZUCuqnibtWf7_4pRBD_8PDoFShmS2_ORiiUdflNjzSbEas",
 *                 "n": "u1LWgoomekdOMfB1lEe96OHehd4XRNCbZRm96RqwOYTTc28Sc_U5wKV2umDzolfoI682ct2BNnRRahYgZPhbOCzHYM6i8sRXjz9Ghx3QHw9zrYACtArwQxrTFiejbfzDPGdPrMQg7T8wjtLtkSyDmCzeXpbIdwmxuLyt_ahLfHelr94kEksMDa42V4Fi5bMW4cCLjlEKzBEHGmFdT8UbLPCvpgsM84JK63e5ifdeI9NdadbC8ZMiR--dFCujT7AgRRyMzxgdn2l-nZJ2ZaYzbLUtAW5_U2kfRVkDNa8d1g__2V5zjU6nfLJ1S2MoXMgRgDPeHpEehZVu2kNaSFvDUQ"
 *             }
 *         ]
 *     },
 *     "tokenLookup": "header:auth",
 *     "claimPassBy": "header:name:name"
 * }
 *         """);
 *         final var certificate = config.get("certificate").orElse("""
 * -----BEGIN CERTIFICATE-----
 * MIIDRjCCAq+gAwIBAgIJAJn3ox4K13PoMA0GCSqGSIb3DQEBBQUAMHYxCzAJBgNV
 * BAYTAkNOMQswCQYDVQQIEwJCSjELMAkGA1UEBxMCQkoxDDAKBgNVBAoTA0FMSTEP
 * MA0GA1UECxMGQUxJWVVOMQ0wCwYDVQQDEwR0ZXN0MR8wHQYJKoZIhvcNAQkBFhB0
 * ZXN0QGhvdG1haWwuY29tMB4XDTE0MTEyNDA2MDQyNVoXDTI0MTEyMTA2MDQyNVow
 * djELMAkGA1UEBhMCQ04xCzAJBgNVBAgTAkJKMQswCQYDVQQHEwJCSjEMMAoGA1UE
 * ChMDQUxJMQ8wDQYDVQQLEwZBTElZVU4xDTALBgNVBAMTBHRlc3QxHzAdBgkqhkiG
 * 9w0BCQEWEHRlc3RAaG90bWFpbC5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJ
 * AoGBAM7SS3e9+Nj0HKAsRuIDNSsS3UK6b+62YQb2uuhKrp1HMrOx61WSDR2qkAnB
 * coG00Uz38EE+9DLYNUVQBK7aSgLP5M1Ak4wr4GqGyCgjejzzh3DshUzLCCy2rook
 * KOyRTlPX+Q5l7rE1fcSNzgepcae5i2sE1XXXzLRIDIvQxcspAgMBAAGjgdswgdgw
 * HQYDVR0OBBYEFBdy+OuMsvbkV7R14f0OyoLoh2z4MIGoBgNVHSMEgaAwgZ2AFBdy
 * +OuMsvbkV7R14f0OyoLoh2z4oXqkeDB2MQswCQYDVQQGEwJDTjELMAkGA1UECBMC
 * QkoxCzAJBgNVBAcTAkJKMQwwCgYDVQQKEwNBTEkxDzANBgNVBAsTBkFMSVlVTjEN
 * MAsGA1UEAxMEdGVzdDEfMB0GCSqGSIb3DQEJARYQdGVzdEBob3RtYWlsLmNvbYIJ
 * AJn3ox4K13PoMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQEFBQADgYEAY7KOsnyT
 * cQzfhiiG7ASjiPakw5wXoycHt5GCvLG5htp2TKVzgv9QTliA3gtfv6oV4zRZx7X1
 * Ofi6hVgErtHaXJheuPVeW6eAW8mHBoEfvDAfU3y9waYrtUevSl07643bzKL6v+Qd
 * DUBTxOAvSYfXTtI90EAxEG/bJJyOm5LqoiA=
 * -----END CERTIFICATE-----
 *         """);
 *         final var privateKey = config.get("privateKey").orElse("""
 * -----BEGIN RSA PRIVATE KEY-----
 * MIICXAIBAAKBgQDO0kt3vfjY9BygLEbiAzUrEt1Cum/utmEG9rroSq6dRzKzsetV
 * kg0dqpAJwXKBtNFM9/BBPvQy2DVFUASu2koCz+TNQJOMK+BqhsgoI3o884dw7IVM
 * ywgstq6KJCjskU5T1/kOZe6xNX3Ejc4HqXGnuYtrBNV118y0SAyL0MXLKQIDAQAB
 * AoGAfe3NxbsGKhN42o4bGsKZPQDfeCHMxayGp5bTd10BtQIE/ST4BcJH+ihAS7Bd
 * 6FwQlKzivNd4GP1MckemklCXfsVckdL94e8ZbJl23GdWul3v8V+KndJHqv5zVJmP
 * hwWoKimwIBTb2s0ctVryr2f18N4hhyFw1yGp0VxclGHkjgECQQD9CvllsnOwHpP4
 * MdrDHbdb29QrobKyKW8pPcDd+sth+kP6Y8MnCVuAKXCKj5FeIsgVtfluPOsZjPzz
 * 71QQWS1dAkEA0T0KXO8gaBQwJhIoo/w6hy5JGZnrNSpOPp5xvJuMAafs2eyvmhJm
 * Ev9SN/Pf2VYa1z6FEnBaLOVD6hf6YQIsPQJAX/CZPoW6dzwgvimo1/GcY6eleiWE
 * qygqjWhsh71e/3bz7yuEAnj5yE3t7Zshcp+dXR3xxGo0eSuLfLFxHgGxwQJAAxf8
 * 9DzQ5NkPkTCJi0sqbl8/03IUKTgT6hcbpWdDXa7m8J3wRr3o5nUB+TPQ5nzAbthM
 * zWX931YQeACcwhxvHQJBAN5mTzzJD4w4Ma6YTaNHyXakdYfyAWrOkPIWZxfhMfXe
 * DrlNdiysTI4Dd1dLeErVpjsckAaOW/JDG5PCSwkaMxk=
 * -----END RSA PRIVATE KEY-----
 *         """);
 *         var default_ = new V3CustomDomain("default", V3CustomDomainArgs.builder()
 *             .customDomainName("flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net")
 *             .routeConfig(V3CustomDomainRouteConfigArgs.builder()
 *                 .routes(
 *                     V3CustomDomainRouteConfigRouteArgs.builder()
 *                         .functionName(functionName1)
 *                         .rewriteConfig(V3CustomDomainRouteConfigRouteRewriteConfigArgs.builder()
 *                             .regexRules(
 *                                 V3CustomDomainRouteConfigRouteRewriteConfigRegexRuleArgs.builder()
 *                                     .match("/api/*")
 *                                     .replacement("$1")
 *                                     .build(),
 *                                 V3CustomDomainRouteConfigRouteRewriteConfigRegexRuleArgs.builder()
 *                                     .match("/api1/*")
 *                                     .replacement("$1")
 *                                     .build(),
 *                                 V3CustomDomainRouteConfigRouteRewriteConfigRegexRuleArgs.builder()
 *                                     .match("/api2/*")
 *                                     .replacement("$1")
 *                                     .build())
 *                             .wildcardRules(
 *                                 V3CustomDomainRouteConfigRouteRewriteConfigWildcardRuleArgs.builder()
 *                                     .match("^/api1/.+?/(.*)")
 *                                     .replacement("/api/v1/$1")
 *                                     .build(),
 *                                 V3CustomDomainRouteConfigRouteRewriteConfigWildcardRuleArgs.builder()
 *                                     .match("^/api2/.+?/(.*)")
 *                                     .replacement("/api/v2/$1")
 *                                     .build(),
 *                                 V3CustomDomainRouteConfigRouteRewriteConfigWildcardRuleArgs.builder()
 *                                     .match("^/api2/.+?/(.*)")
 *                                     .replacement("/api/v3/$1")
 *                                     .build())
 *                             .equalRules(
 *                                 V3CustomDomainRouteConfigRouteRewriteConfigEqualRuleArgs.builder()
 *                                     .match("/old")
 *                                     .replacement("/new")
 *                                     .build(),
 *                                 V3CustomDomainRouteConfigRouteRewriteConfigEqualRuleArgs.builder()
 *                                     .replacement("/new1")
 *                                     .match("/old1")
 *                                     .build(),
 *                                 V3CustomDomainRouteConfigRouteRewriteConfigEqualRuleArgs.builder()
 *                                     .match("/old2")
 *                                     .replacement("/new2")
 *                                     .build())
 *                             .build())
 *                         .methods(
 *                             "GET",
 *                             "POST",
 *                             "DELETE",
 *                             "HEAD")
 *                         .path("/a")
 *                         .qualifier("LATEST")
 *                         .build(),
 *                     V3CustomDomainRouteConfigRouteArgs.builder()
 *                         .functionName(functionName1)
 *                         .methods("GET")
 *                         .path("/b")
 *                         .qualifier("LATEST")
 *                         .build(),
 *                     V3CustomDomainRouteConfigRouteArgs.builder()
 *                         .functionName(functionName1)
 *                         .methods("POST")
 *                         .path("/c")
 *                         .qualifier("1")
 *                         .build())
 *                 .build())
 *             .authConfig(V3CustomDomainAuthConfigArgs.builder()
 *                 .authType("jwt")
 *                 .authInfo(authConfig)
 *                 .build())
 *             .protocol("HTTP,HTTPS")
 *             .certConfig(V3CustomDomainCertConfigArgs.builder()
 *                 .certName("cert-name")
 *                 .certificate(certificate)
 *                 .privateKey(privateKey)
 *                 .build())
 *             .tlsConfig(V3CustomDomainTlsConfigArgs.builder()
 *                 .cipherSuites(
 *                     "TLS_RSA_WITH_AES_128_CBC_SHA",
 *                     "TLS_RSA_WITH_AES_256_CBC_SHA",
 *                     "TLS_RSA_WITH_AES_128_GCM_SHA256",
 *                     "TLS_RSA_WITH_AES_256_GCM_SHA384")
 *                 .maxVersion("TLSv1.3")
 *                 .minVersion("TLSv1.0")
 *                 .build())
 *             .wafConfig(V3CustomDomainWafConfigArgs.builder()
 *                 .enableWaf("false")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * configuration:
 *   name:
 *     type: string
 *     default: flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net
 *   functionName1:
 *     type: string
 *     default: terraform-custom-domain-t1
 *   authConfig:
 *     type: string
 *     default: |
 *       {
 *           "jwks": {
 *               "keys": [
 *                   {
 *                       "p": "8AdUVeldoE4LueFuzEF_C8tvJ7NhlkzS58Gz9KJTPXPr5DADSUVLWJCr5OdFE79q513SneT0UhGo-JfQ1lNMoNv5-YZ1AxIo9fZUEPIe-KyX9ttaglpzCAUE3TeKdm5J-_HZQzBPKbyUwJHAILNgB2-4IBZZwK7LAfbmfi9TmFM",
 *                       "kty": "RSA",
 *                       "q": "x8m5ydXwC8AAp9I-hOnUAx6yQJz1Nx-jXPCfn--XdHpJuNcuwRQsuUCSRQs_h3SoCI3qZZdzswQnPrtHFxgUJtQFuMj-QZpyMnebDb81rmczl2KPVUtaVDVagJEF6U9Ov3PfrLhvHUEv5u7p6s4Z6maBUaByfFlhEVPv4_ao8Us",
 *                       "d": "bjIQAKD2e65gwJ38_Sqq_EmLFuMMey3gjDv1bSCHFH8fyONJTq-utrZfvspz6EegRwW2mSHW9kq87hRwIBW9y7ED5N4KG5gHDjyh57BRE0SKv0Dz1igtKLyp-nl8-aHc1DbONwr1d7tZfFt255TxIN8cPTakXOp2Av_ztql_JotVUGK8eHmXNJFlvq5tc180sKWMHNSNsCUhQgcB1TWb_gwcqxdsIWPsLZI491XKeTGQ98J7z5h6R1cTC97lfJZ0vNtJahd2jHd3WfTUDj5-untMKyZpYYak2Vr8xtFz8H6Q5Rsz8uX_7gtEqYH2CMjPdbXcebrnD1igRSJMYiP0lQ",
 *                       "e": "AQAB",
 *                       "use": "sig",
 *                       "qi": "MTCCRu8AcvvjbHms7V_sDFO7wX0YNyvOJAAbuTmHvQbJ0NDeDta-f-hi8cjkMk7Fpk2hej158E5gDyO62UG99wHZSbmHT34MvIdmhQ5mnbL-5KK9rxde0nayO1ebGepD_GJThPAg9iskzeWpCg5X2etNo2bHoG_ZLQGXj2BQ1VM",
 *                       "dp": "J4_ttKNcTTnP8PlZO81n1VfYoGCOqylKceyZbq76rVxX-yp2wDLtslFWI8qCtjiMtEnglynPo19JzH-pakocjT70us4Qp0rs-W16ebiOpko8WfHZvzaNUzsQjC3FYrPW-fHo74wc4DI3Cm57jmhCYbdmT9OfQ4UL7Oz3HMFMNAU",
 *                       "alg": "RS256",
 *                       "dq": "H4-VgvYB-sk1EU3cRIDv1iJWRHDHKBMeaoM0pD5kLalX1hRgNW4rdoRl1vRk79AU720D11Kqm2APlxBctaA_JrcdxEg0KkbsvV45p11KbKeu9b5DKFVECsN27ZJ7XZUCuqnibtWf7_4pRBD_8PDoFShmS2_ORiiUdflNjzSbEas",
 *                       "n": "u1LWgoomekdOMfB1lEe96OHehd4XRNCbZRm96RqwOYTTc28Sc_U5wKV2umDzolfoI682ct2BNnRRahYgZPhbOCzHYM6i8sRXjz9Ghx3QHw9zrYACtArwQxrTFiejbfzDPGdPrMQg7T8wjtLtkSyDmCzeXpbIdwmxuLyt_ahLfHelr94kEksMDa42V4Fi5bMW4cCLjlEKzBEHGmFdT8UbLPCvpgsM84JK63e5ifdeI9NdadbC8ZMiR--dFCujT7AgRRyMzxgdn2l-nZJ2ZaYzbLUtAW5_U2kfRVkDNa8d1g__2V5zjU6nfLJ1S2MoXMgRgDPeHpEehZVu2kNaSFvDUQ"
 *                   }
 *               ]
 *           },
 *           "tokenLookup": "header:auth",
 *           "claimPassBy": "header:name:name"
 *       }
 *   certificate:
 *     type: string
 *     default: |
 *       -----BEGIN CERTIFICATE-----
 *       MIIDRjCCAq+gAwIBAgIJAJn3ox4K13PoMA0GCSqGSIb3DQEBBQUAMHYxCzAJBgNV
 *       BAYTAkNOMQswCQYDVQQIEwJCSjELMAkGA1UEBxMCQkoxDDAKBgNVBAoTA0FMSTEP
 *       MA0GA1UECxMGQUxJWVVOMQ0wCwYDVQQDEwR0ZXN0MR8wHQYJKoZIhvcNAQkBFhB0
 *       ZXN0QGhvdG1haWwuY29tMB4XDTE0MTEyNDA2MDQyNVoXDTI0MTEyMTA2MDQyNVow
 *       djELMAkGA1UEBhMCQ04xCzAJBgNVBAgTAkJKMQswCQYDVQQHEwJCSjEMMAoGA1UE
 *       ChMDQUxJMQ8wDQYDVQQLEwZBTElZVU4xDTALBgNVBAMTBHRlc3QxHzAdBgkqhkiG
 *       9w0BCQEWEHRlc3RAaG90bWFpbC5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJ
 *       AoGBAM7SS3e9+Nj0HKAsRuIDNSsS3UK6b+62YQb2uuhKrp1HMrOx61WSDR2qkAnB
 *       coG00Uz38EE+9DLYNUVQBK7aSgLP5M1Ak4wr4GqGyCgjejzzh3DshUzLCCy2rook
 *       KOyRTlPX+Q5l7rE1fcSNzgepcae5i2sE1XXXzLRIDIvQxcspAgMBAAGjgdswgdgw
 *       HQYDVR0OBBYEFBdy+OuMsvbkV7R14f0OyoLoh2z4MIGoBgNVHSMEgaAwgZ2AFBdy
 *       +OuMsvbkV7R14f0OyoLoh2z4oXqkeDB2MQswCQYDVQQGEwJDTjELMAkGA1UECBMC
 *       QkoxCzAJBgNVBAcTAkJKMQwwCgYDVQQKEwNBTEkxDzANBgNVBAsTBkFMSVlVTjEN
 *       MAsGA1UEAxMEdGVzdDEfMB0GCSqGSIb3DQEJARYQdGVzdEBob3RtYWlsLmNvbYIJ
 *       AJn3ox4K13PoMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQEFBQADgYEAY7KOsnyT
 *       cQzfhiiG7ASjiPakw5wXoycHt5GCvLG5htp2TKVzgv9QTliA3gtfv6oV4zRZx7X1
 *       Ofi6hVgErtHaXJheuPVeW6eAW8mHBoEfvDAfU3y9waYrtUevSl07643bzKL6v+Qd
 *       DUBTxOAvSYfXTtI90EAxEG/bJJyOm5LqoiA=
 *       -----END CERTIFICATE-----
 *   privateKey:
 *     type: string
 *     default: |
 *       -----BEGIN RSA PRIVATE KEY-----
 *       MIICXAIBAAKBgQDO0kt3vfjY9BygLEbiAzUrEt1Cum/utmEG9rroSq6dRzKzsetV
 *       kg0dqpAJwXKBtNFM9/BBPvQy2DVFUASu2koCz+TNQJOMK+BqhsgoI3o884dw7IVM
 *       ywgstq6KJCjskU5T1/kOZe6xNX3Ejc4HqXGnuYtrBNV118y0SAyL0MXLKQIDAQAB
 *       AoGAfe3NxbsGKhN42o4bGsKZPQDfeCHMxayGp5bTd10BtQIE/ST4BcJH+ihAS7Bd
 *       6FwQlKzivNd4GP1MckemklCXfsVckdL94e8ZbJl23GdWul3v8V+KndJHqv5zVJmP
 *       hwWoKimwIBTb2s0ctVryr2f18N4hhyFw1yGp0VxclGHkjgECQQD9CvllsnOwHpP4
 *       MdrDHbdb29QrobKyKW8pPcDd+sth+kP6Y8MnCVuAKXCKj5FeIsgVtfluPOsZjPzz
 *       71QQWS1dAkEA0T0KXO8gaBQwJhIoo/w6hy5JGZnrNSpOPp5xvJuMAafs2eyvmhJm
 *       Ev9SN/Pf2VYa1z6FEnBaLOVD6hf6YQIsPQJAX/CZPoW6dzwgvimo1/GcY6eleiWE
 *       qygqjWhsh71e/3bz7yuEAnj5yE3t7Zshcp+dXR3xxGo0eSuLfLFxHgGxwQJAAxf8
 *       9DzQ5NkPkTCJi0sqbl8/03IUKTgT6hcbpWdDXa7m8J3wRr3o5nUB+TPQ5nzAbthM
 *       zWX931YQeACcwhxvHQJBAN5mTzzJD4w4Ma6YTaNHyXakdYfyAWrOkPIWZxfhMfXe
 *       DrlNdiysTI4Dd1dLeErVpjsckAaOW/JDG5PCSwkaMxk=
 *       -----END RSA PRIVATE KEY-----
 * resources:
 *   default:
 *     type: alicloud:fc:V3CustomDomain
 *     properties:
 *       customDomainName: flask-6ew9.fcv3.1511928242963727.cn-shanghai.fc.devsapp.net
 *       routeConfig:
 *         routes:
 *           - functionName: ${functionName1}
 *             rewriteConfig:
 *               regexRules:
 *                 - match: /api/*
 *                   replacement: $1
 *                 - match: /api1/*
 *                   replacement: $1
 *                 - match: /api2/*
 *                   replacement: $1
 *               wildcardRules:
 *                 - match: ^/api1/.+?/(.*)
 *                   replacement: /api/v1/$1
 *                 - match: ^/api2/.+?/(.*)
 *                   replacement: /api/v2/$1
 *                 - match: ^/api2/.+?/(.*)
 *                   replacement: /api/v3/$1
 *               equalRules:
 *                 - match: /old
 *                   replacement: /new
 *                 - replacement: /new1
 *                   match: /old1
 *                 - match: /old2
 *                   replacement: /new2
 *             methods:
 *               - GET
 *               - POST
 *               - DELETE
 *               - HEAD
 *             path: /a
 *             qualifier: LATEST
 *           - functionName: ${functionName1}
 *             methods:
 *               - GET
 *             path: /b
 *             qualifier: LATEST
 *           - functionName: ${functionName1}
 *             methods:
 *               - POST
 *             path: /c
 *             qualifier: '1'
 *       authConfig:
 *         authType: jwt
 *         authInfo: ${authConfig}
 *       protocol: HTTP,HTTPS
 *       certConfig:
 *         certName: cert-name
 *         certificate: ${certificate}
 *         privateKey: ${privateKey}
 *       tlsConfig:
 *         cipherSuites:
 *           - TLS_RSA_WITH_AES_128_CBC_SHA
 *           - TLS_RSA_WITH_AES_256_CBC_SHA
 *           - TLS_RSA_WITH_AES_128_GCM_SHA256
 *           - TLS_RSA_WITH_AES_256_GCM_SHA384
 *         maxVersion: TLSv1.3
 *         minVersion: TLSv1.0
 *       wafConfig:
 *         enableWaf: 'false'
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * FCV3 Custom Domain can be imported using the id, e.g.
 * ```sh
 * $ pulumi import alicloud:fc/v3CustomDomain:V3CustomDomain example <id>
 * ```
 * @property authConfig Permission authentication configuration See `auth_config` below.
 * @property certConfig HTTPS certificate information See `cert_config` below.
 * @property customDomainName The name of the resource
 * @property protocol The protocol type supported by the domain name. HTTP: only HTTP protocol is supported. HTTPS: only HTTPS is supported. HTTP,HTTPS: Supports HTTP and HTTPS protocols.
 * @property routeConfig Route matching rule configuration See `route_config` below.
 * @property tlsConfig TLS configuration information See `tls_config` below.
 * @property wafConfig Web application firewall configuration information See `waf_config` below.
 * */*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/
 */
public data class V3CustomDomainArgs(
    public val authConfig: Output<V3CustomDomainAuthConfigArgs>? = null,
    public val certConfig: Output<V3CustomDomainCertConfigArgs>? = null,
    public val customDomainName: Output<String>? = null,
    public val protocol: Output<String>? = null,
    public val routeConfig: Output<V3CustomDomainRouteConfigArgs>? = null,
    public val tlsConfig: Output<V3CustomDomainTlsConfigArgs>? = null,
    public val wafConfig: Output<V3CustomDomainWafConfigArgs>? = null,
) : ConvertibleToJava<com.pulumi.alicloud.fc.V3CustomDomainArgs> {
    override fun toJava(): com.pulumi.alicloud.fc.V3CustomDomainArgs =
        com.pulumi.alicloud.fc.V3CustomDomainArgs.builder()
            .authConfig(authConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .certConfig(certConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .customDomainName(customDomainName?.applyValue({ args0 -> args0 }))
            .protocol(protocol?.applyValue({ args0 -> args0 }))
            .routeConfig(routeConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .tlsConfig(tlsConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .wafConfig(wafConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) })).build()
}

/**
 * Builder for [V3CustomDomainArgs].
 */
@PulumiTagMarker
public class V3CustomDomainArgsBuilder internal constructor() {
    private var authConfig: Output<V3CustomDomainAuthConfigArgs>? = null

    private var certConfig: Output<V3CustomDomainCertConfigArgs>? = null

    private var customDomainName: Output<String>? = null

    private var protocol: Output<String>? = null

    private var routeConfig: Output<V3CustomDomainRouteConfigArgs>? = null

    private var tlsConfig: Output<V3CustomDomainTlsConfigArgs>? = null

    private var wafConfig: Output<V3CustomDomainWafConfigArgs>? = null

    /**
     * @param value Permission authentication configuration See `auth_config` below.
     */
    @JvmName("srxaprkaahdpjyfr")
    public suspend fun authConfig(`value`: Output<V3CustomDomainAuthConfigArgs>) {
        this.authConfig = value
    }

    /**
     * @param value HTTPS certificate information See `cert_config` below.
     */
    @JvmName("axewxrnreljucbxm")
    public suspend fun certConfig(`value`: Output<V3CustomDomainCertConfigArgs>) {
        this.certConfig = value
    }

    /**
     * @param value The name of the resource
     */
    @JvmName("rpwantkkttdncmxk")
    public suspend fun customDomainName(`value`: Output<String>) {
        this.customDomainName = value
    }

    /**
     * @param value The protocol type supported by the domain name. HTTP: only HTTP protocol is supported. HTTPS: only HTTPS is supported. HTTP,HTTPS: Supports HTTP and HTTPS protocols.
     */
    @JvmName("cqlkirsudrfontxf")
    public suspend fun protocol(`value`: Output<String>) {
        this.protocol = value
    }

    /**
     * @param value Route matching rule configuration See `route_config` below.
     */
    @JvmName("rrlqlxvhmeoltqrb")
    public suspend fun routeConfig(`value`: Output<V3CustomDomainRouteConfigArgs>) {
        this.routeConfig = value
    }

    /**
     * @param value TLS configuration information See `tls_config` below.
     */
    @JvmName("yflushekcaxdeqxo")
    public suspend fun tlsConfig(`value`: Output<V3CustomDomainTlsConfigArgs>) {
        this.tlsConfig = value
    }

    /**
     * @param value Web application firewall configuration information See `waf_config` below.
     */
    @JvmName("wgmyruqoxiwrxuhd")
    public suspend fun wafConfig(`value`: Output<V3CustomDomainWafConfigArgs>) {
        this.wafConfig = value
    }

    /**
     * @param value Permission authentication configuration See `auth_config` below.
     */
    @JvmName("seeeskcascvmsaxn")
    public suspend fun authConfig(`value`: V3CustomDomainAuthConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.authConfig = mapped
    }

    /**
     * @param argument Permission authentication configuration See `auth_config` below.
     */
    @JvmName("ablplucvyacxlrms")
    public suspend fun authConfig(argument: suspend V3CustomDomainAuthConfigArgsBuilder.() -> Unit) {
        val toBeMapped = V3CustomDomainAuthConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.authConfig = mapped
    }

    /**
     * @param value HTTPS certificate information See `cert_config` below.
     */
    @JvmName("gftudimafqmogwyu")
    public suspend fun certConfig(`value`: V3CustomDomainCertConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.certConfig = mapped
    }

    /**
     * @param argument HTTPS certificate information See `cert_config` below.
     */
    @JvmName("waxcrydupiorgxia")
    public suspend fun certConfig(argument: suspend V3CustomDomainCertConfigArgsBuilder.() -> Unit) {
        val toBeMapped = V3CustomDomainCertConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.certConfig = mapped
    }

    /**
     * @param value The name of the resource
     */
    @JvmName("mmcbyyerdaltqntu")
    public suspend fun customDomainName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customDomainName = mapped
    }

    /**
     * @param value The protocol type supported by the domain name. HTTP: only HTTP protocol is supported. HTTPS: only HTTPS is supported. HTTP,HTTPS: Supports HTTP and HTTPS protocols.
     */
    @JvmName("ikpfxxcxbuotaoom")
    public suspend fun protocol(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.protocol = mapped
    }

    /**
     * @param value Route matching rule configuration See `route_config` below.
     */
    @JvmName("sfwuhpjmogcfgyav")
    public suspend fun routeConfig(`value`: V3CustomDomainRouteConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.routeConfig = mapped
    }

    /**
     * @param argument Route matching rule configuration See `route_config` below.
     */
    @JvmName("jyvwmnehwrclpsxh")
    public suspend fun routeConfig(argument: suspend V3CustomDomainRouteConfigArgsBuilder.() -> Unit) {
        val toBeMapped = V3CustomDomainRouteConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.routeConfig = mapped
    }

    /**
     * @param value TLS configuration information See `tls_config` below.
     */
    @JvmName("frsmhfhfndjagali")
    public suspend fun tlsConfig(`value`: V3CustomDomainTlsConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tlsConfig = mapped
    }

    /**
     * @param argument TLS configuration information See `tls_config` below.
     */
    @JvmName("spcxcodiiclokwfx")
    public suspend fun tlsConfig(argument: suspend V3CustomDomainTlsConfigArgsBuilder.() -> Unit) {
        val toBeMapped = V3CustomDomainTlsConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.tlsConfig = mapped
    }

    /**
     * @param value Web application firewall configuration information See `waf_config` below.
     */
    @JvmName("wekltsbtmuwtqqsv")
    public suspend fun wafConfig(`value`: V3CustomDomainWafConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.wafConfig = mapped
    }

    /**
     * @param argument Web application firewall configuration information See `waf_config` below.
     */
    @JvmName("unafpqbdbiobsogl")
    public suspend fun wafConfig(argument: suspend V3CustomDomainWafConfigArgsBuilder.() -> Unit) {
        val toBeMapped = V3CustomDomainWafConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.wafConfig = mapped
    }

    internal fun build(): V3CustomDomainArgs = V3CustomDomainArgs(
        authConfig = authConfig,
        certConfig = certConfig,
        customDomainName = customDomainName,
        protocol = protocol,
        routeConfig = routeConfig,
        tlsConfig = tlsConfig,
        wafConfig = wafConfig,
    )
}
