@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.alicloud.nas.kotlin

import com.pulumi.alicloud.nas.RecycleBinArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Provides a Network Attached Storage (NAS) Recycle Bin resource.
 * For information about Network Attached Storage (NAS) Recycle Bin and how to use it, see [What is Recycle Bin](https://www.alibabacloud.com/help/en/doc-detail/264185.html).
 * > **NOTE:** Available in v1.155.0+.
 * ## Example Usage
 * Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as alicloud from "@pulumi/alicloud";
 * const example = alicloud.nas.getZones({
 *     fileSystemType: "standard",
 * });
 * const exampleFileSystem = new alicloud.nas.FileSystem("example", {
 *     protocolType: "NFS",
 *     storageType: "Performance",
 *     description: "terraform-example",
 *     encryptType: 1,
 *     zoneId: example.then(example => example.zones?.[0]?.zoneId),
 * });
 * const exampleRecycleBin = new alicloud.nas.RecycleBin("example", {
 *     fileSystemId: exampleFileSystem.id,
 *     reservedDays: 3,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_alicloud as alicloud
 * example = alicloud.nas.get_zones(file_system_type="standard")
 * example_file_system = alicloud.nas.FileSystem("example",
 *     protocol_type="NFS",
 *     storage_type="Performance",
 *     description="terraform-example",
 *     encrypt_type=1,
 *     zone_id=example.zones[0].zone_id)
 * example_recycle_bin = alicloud.nas.RecycleBin("example",
 *     file_system_id=example_file_system.id,
 *     reserved_days=3)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AliCloud = Pulumi.AliCloud;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = AliCloud.Nas.GetZones.Invoke(new()
 *     {
 *         FileSystemType = "standard",
 *     });
 *     var exampleFileSystem = new AliCloud.Nas.FileSystem("example", new()
 *     {
 *         ProtocolType = "NFS",
 *         StorageType = "Performance",
 *         Description = "terraform-example",
 *         EncryptType = 1,
 *         ZoneId = example.Apply(getZonesResult => getZonesResult.Zones[0]?.ZoneId),
 *     });
 *     var exampleRecycleBin = new AliCloud.Nas.RecycleBin("example", new()
 *     {
 *         FileSystemId = exampleFileSystem.Id,
 *         ReservedDays = 3,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/nas"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := nas.GetZones(ctx, &nas.GetZonesArgs{
 * 			FileSystemType: pulumi.StringRef("standard"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleFileSystem, err := nas.NewFileSystem(ctx, "example", &nas.FileSystemArgs{
 * 			ProtocolType: pulumi.String("NFS"),
 * 			StorageType:  pulumi.String("Performance"),
 * 			Description:  pulumi.String("terraform-example"),
 * 			EncryptType:  pulumi.Int(1),
 * 			ZoneId:       pulumi.String(example.Zones[0].ZoneId),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = nas.NewRecycleBin(ctx, "example", &nas.RecycleBinArgs{
 * 			FileSystemId: exampleFileSystem.ID(),
 * 			ReservedDays: pulumi.Int(3),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.alicloud.nas.NasFunctions;
 * import com.pulumi.alicloud.nas.inputs.GetZonesArgs;
 * import com.pulumi.alicloud.nas.FileSystem;
 * import com.pulumi.alicloud.nas.FileSystemArgs;
 * import com.pulumi.alicloud.nas.RecycleBin;
 * import com.pulumi.alicloud.nas.RecycleBinArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var example = NasFunctions.getZones(GetZonesArgs.builder()
 *             .fileSystemType("standard")
 *             .build());
 *         var exampleFileSystem = new FileSystem("exampleFileSystem", FileSystemArgs.builder()
 *             .protocolType("NFS")
 *             .storageType("Performance")
 *             .description("terraform-example")
 *             .encryptType("1")
 *             .zoneId(example.applyValue(getZonesResult -> getZonesResult.zones()[0].zoneId()))
 *             .build());
 *         var exampleRecycleBin = new RecycleBin("exampleRecycleBin", RecycleBinArgs.builder()
 *             .fileSystemId(exampleFileSystem.id())
 *             .reservedDays(3)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleFileSystem:
 *     type: alicloud:nas:FileSystem
 *     name: example
 *     properties:
 *       protocolType: NFS
 *       storageType: Performance
 *       description: terraform-example
 *       encryptType: '1'
 *       zoneId: ${example.zones[0].zoneId}
 *   exampleRecycleBin:
 *     type: alicloud:nas:RecycleBin
 *     name: example
 *     properties:
 *       fileSystemId: ${exampleFileSystem.id}
 *       reservedDays: 3
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: alicloud:nas:getZones
 *       Arguments:
 *         fileSystemType: standard
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Network Attached Storage (NAS) Recycle Bin can be imported using the id, e.g.
 * ```sh
 * $ pulumi import alicloud:nas/recycleBin:RecycleBin example <file_system_id>
 * ```
 * @property fileSystemId The ID of the file system for which you want to enable the recycle bin feature.
 * @property reservedDays The period for which the files in the recycle bin are retained. Unit: days. Valid values: `1` to `180`.
 */
public data class RecycleBinArgs(
    public val fileSystemId: Output<String>? = null,
    public val reservedDays: Output<Int>? = null,
) : ConvertibleToJava<com.pulumi.alicloud.nas.RecycleBinArgs> {
    override fun toJava(): com.pulumi.alicloud.nas.RecycleBinArgs =
        com.pulumi.alicloud.nas.RecycleBinArgs.builder()
            .fileSystemId(fileSystemId?.applyValue({ args0 -> args0 }))
            .reservedDays(reservedDays?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [RecycleBinArgs].
 */
@PulumiTagMarker
public class RecycleBinArgsBuilder internal constructor() {
    private var fileSystemId: Output<String>? = null

    private var reservedDays: Output<Int>? = null

    /**
     * @param value The ID of the file system for which you want to enable the recycle bin feature.
     */
    @JvmName("ojjhyvpfiuohceyy")
    public suspend fun fileSystemId(`value`: Output<String>) {
        this.fileSystemId = value
    }

    /**
     * @param value The period for which the files in the recycle bin are retained. Unit: days. Valid values: `1` to `180`.
     */
    @JvmName("wbhhfpbcmfykgttx")
    public suspend fun reservedDays(`value`: Output<Int>) {
        this.reservedDays = value
    }

    /**
     * @param value The ID of the file system for which you want to enable the recycle bin feature.
     */
    @JvmName("dewdwxjifdqfiuhu")
    public suspend fun fileSystemId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.fileSystemId = mapped
    }

    /**
     * @param value The period for which the files in the recycle bin are retained. Unit: days. Valid values: `1` to `180`.
     */
    @JvmName("afmyebvafwgharqi")
    public suspend fun reservedDays(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.reservedDays = mapped
    }

    internal fun build(): RecycleBinArgs = RecycleBinArgs(
        fileSystemId = fileSystemId,
        reservedDays = reservedDays,
    )
}
