@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.alicloud.rds.kotlin

import com.pulumi.alicloud.rds.RdsUpgradeDbInstanceArgs.builder
import com.pulumi.alicloud.rds.kotlin.inputs.RdsUpgradeDbInstanceParameterArgs
import com.pulumi.alicloud.rds.kotlin.inputs.RdsUpgradeDbInstanceParameterArgsBuilder
import com.pulumi.alicloud.rds.kotlin.inputs.RdsUpgradeDbInstancePgHbaConfArgs
import com.pulumi.alicloud.rds.kotlin.inputs.RdsUpgradeDbInstancePgHbaConfArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * Provides a RDS Upgrade DB Instance resource.
 * For information about RDS Upgrade DB Instance and how to use it, see [What is ApsaraDB for RDS](https://www.alibabacloud.com/help/en/doc-detail/26092.htm).
 * > **NOTE:** Available since v1.153.0+.
 * ## Example Usage
 * ### Create a RDS PostgreSQL upgrade instance
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as alicloud from "@pulumi/alicloud";
 * const example = alicloud.rds.getZones({
 *     engine: "PostgreSQL",
 *     engineVersion: "13.0",
 *     instanceChargeType: "PostPaid",
 *     category: "HighAvailability",
 *     dbInstanceStorageType: "cloud_essd",
 * });
 * const exampleGetInstanceClasses = example.then(example => alicloud.rds.getInstanceClasses({
 *     zoneId: example.zones?.[0]?.id,
 *     engine: "PostgreSQL",
 *     engineVersion: "13.0",
 *     category: "HighAvailability",
 *     dbInstanceStorageType: "cloud_essd",
 *     instanceChargeType: "PostPaid",
 * }));
 * const exampleGetCrossRegions = alicloud.rds.getCrossRegions({});
 * const exampleNetwork = new alicloud.vpc.Network("example", {
 *     vpcName: "terraform-example",
 *     cidrBlock: "172.16.0.0/16",
 * });
 * const exampleSwitch = new alicloud.vpc.Switch("example", {
 *     vpcId: exampleNetwork.id,
 *     cidrBlock: "172.16.0.0/24",
 *     zoneId: example.then(example => example.zones?.[0]?.id),
 *     vswitchName: "terraform-example",
 * });
 * const exampleInstance = new alicloud.rds.Instance("example", {
 *     engine: "PostgreSQL",
 *     engineVersion: "13.0",
 *     dbInstanceStorageType: "cloud_essd",
 *     instanceType: exampleGetInstanceClasses.then(exampleGetInstanceClasses => exampleGetInstanceClasses.instanceClasses?.[0]?.instanceClass),
 *     instanceStorage: exampleGetInstanceClasses.then(exampleGetInstanceClasses => exampleGetInstanceClasses.instanceClasses?.[0]?.storageRange?.min),
 *     instanceChargeType: "Postpaid",
 *     instanceName: "terraform-example",
 *     vswitchId: exampleSwitch.id,
 *     monitoringPeriod: 60,
 * });
 * const exampleRdsUpgradeDbInstance = new alicloud.rds.RdsUpgradeDbInstance("example", {
 *     sourceDbInstanceId: exampleInstance.id,
 *     targetMajorVersion: "14.0",
 *     dbInstanceClass: exampleInstance.instanceType,
 *     dbInstanceStorage: exampleInstance.instanceStorage,
 *     dbInstanceStorageType: exampleInstance.dbInstanceStorageType,
 *     instanceNetworkType: "VPC",
 *     collectStatMode: "After",
 *     switchOver: "false",
 *     paymentType: "PayAsYouGo",
 *     dbInstanceDescription: "terraform-example",
 *     vswitchId: exampleSwitch.id,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_alicloud as alicloud
 * example = alicloud.rds.get_zones(engine="PostgreSQL",
 *     engine_version="13.0",
 *     instance_charge_type="PostPaid",
 *     category="HighAvailability",
 *     db_instance_storage_type="cloud_essd")
 * example_get_instance_classes = alicloud.rds.get_instance_classes(zone_id=example.zones[0].id,
 *     engine="PostgreSQL",
 *     engine_version="13.0",
 *     category="HighAvailability",
 *     db_instance_storage_type="cloud_essd",
 *     instance_charge_type="PostPaid")
 * example_get_cross_regions = alicloud.rds.get_cross_regions()
 * example_network = alicloud.vpc.Network("example",
 *     vpc_name="terraform-example",
 *     cidr_block="172.16.0.0/16")
 * example_switch = alicloud.vpc.Switch("example",
 *     vpc_id=example_network.id,
 *     cidr_block="172.16.0.0/24",
 *     zone_id=example.zones[0].id,
 *     vswitch_name="terraform-example")
 * example_instance = alicloud.rds.Instance("example",
 *     engine="PostgreSQL",
 *     engine_version="13.0",
 *     db_instance_storage_type="cloud_essd",
 *     instance_type=example_get_instance_classes.instance_classes[0].instance_class,
 *     instance_storage=example_get_instance_classes.instance_classes[0].storage_range.min,
 *     instance_charge_type="Postpaid",
 *     instance_name="terraform-example",
 *     vswitch_id=example_switch.id,
 *     monitoring_period=60)
 * example_rds_upgrade_db_instance = alicloud.rds.RdsUpgradeDbInstance("example",
 *     source_db_instance_id=example_instance.id,
 *     target_major_version="14.0",
 *     db_instance_class=example_instance.instance_type,
 *     db_instance_storage=example_instance.instance_storage,
 *     db_instance_storage_type=example_instance.db_instance_storage_type,
 *     instance_network_type="VPC",
 *     collect_stat_mode="After",
 *     switch_over="false",
 *     payment_type="PayAsYouGo",
 *     db_instance_description="terraform-example",
 *     vswitch_id=example_switch.id)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AliCloud = Pulumi.AliCloud;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = AliCloud.Rds.GetZones.Invoke(new()
 *     {
 *         Engine = "PostgreSQL",
 *         EngineVersion = "13.0",
 *         InstanceChargeType = "PostPaid",
 *         Category = "HighAvailability",
 *         DbInstanceStorageType = "cloud_essd",
 *     });
 *     var exampleGetInstanceClasses = AliCloud.Rds.GetInstanceClasses.Invoke(new()
 *     {
 *         ZoneId = example.Apply(getZonesResult => getZonesResult.Zones[0]?.Id),
 *         Engine = "PostgreSQL",
 *         EngineVersion = "13.0",
 *         Category = "HighAvailability",
 *         DbInstanceStorageType = "cloud_essd",
 *         InstanceChargeType = "PostPaid",
 *     });
 *     var exampleGetCrossRegions = AliCloud.Rds.GetCrossRegions.Invoke();
 *     var exampleNetwork = new AliCloud.Vpc.Network("example", new()
 *     {
 *         VpcName = "terraform-example",
 *         CidrBlock = "172.16.0.0/16",
 *     });
 *     var exampleSwitch = new AliCloud.Vpc.Switch("example", new()
 *     {
 *         VpcId = exampleNetwork.Id,
 *         CidrBlock = "172.16.0.0/24",
 *         ZoneId = example.Apply(getZonesResult => getZonesResult.Zones[0]?.Id),
 *         VswitchName = "terraform-example",
 *     });
 *     var exampleInstance = new AliCloud.Rds.Instance("example", new()
 *     {
 *         Engine = "PostgreSQL",
 *         EngineVersion = "13.0",
 *         DbInstanceStorageType = "cloud_essd",
 *         InstanceType = exampleGetInstanceClasses.Apply(getInstanceClassesResult => getInstanceClassesResult.InstanceClasses[0]?.InstanceClass),
 *         InstanceStorage = exampleGetInstanceClasses.Apply(getInstanceClassesResult => getInstanceClassesResult.InstanceClasses[0]?.StorageRange?.Min),
 *         InstanceChargeType = "Postpaid",
 *         InstanceName = "terraform-example",
 *         VswitchId = exampleSwitch.Id,
 *         MonitoringPeriod = 60,
 *     });
 *     var exampleRdsUpgradeDbInstance = new AliCloud.Rds.RdsUpgradeDbInstance("example", new()
 *     {
 *         SourceDbInstanceId = exampleInstance.Id,
 *         TargetMajorVersion = "14.0",
 *         DbInstanceClass = exampleInstance.InstanceType,
 *         DbInstanceStorage = exampleInstance.InstanceStorage,
 *         DbInstanceStorageType = exampleInstance.DbInstanceStorageType,
 *         InstanceNetworkType = "VPC",
 *         CollectStatMode = "After",
 *         SwitchOver = "false",
 *         PaymentType = "PayAsYouGo",
 *         DbInstanceDescription = "terraform-example",
 *         VswitchId = exampleSwitch.Id,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/rds"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/vpc"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := rds.GetZones(ctx, &rds.GetZonesArgs{
 * 			Engine:                pulumi.StringRef("PostgreSQL"),
 * 			EngineVersion:         pulumi.StringRef("13.0"),
 * 			InstanceChargeType:    pulumi.StringRef("PostPaid"),
 * 			Category:              pulumi.StringRef("HighAvailability"),
 * 			DbInstanceStorageType: pulumi.StringRef("cloud_essd"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleGetInstanceClasses, err := rds.GetInstanceClasses(ctx, &rds.GetInstanceClassesArgs{
 * 			ZoneId:                pulumi.StringRef(example.Zones[0].Id),
 * 			Engine:                pulumi.StringRef("PostgreSQL"),
 * 			EngineVersion:         pulumi.StringRef("13.0"),
 * 			Category:              pulumi.StringRef("HighAvailability"),
 * 			DbInstanceStorageType: pulumi.StringRef("cloud_essd"),
 * 			InstanceChargeType:    pulumi.StringRef("PostPaid"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.GetCrossRegions(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleNetwork, err := vpc.NewNetwork(ctx, "example", &vpc.NetworkArgs{
 * 			VpcName:   pulumi.String("terraform-example"),
 * 			CidrBlock: pulumi.String("172.16.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleSwitch, err := vpc.NewSwitch(ctx, "example", &vpc.SwitchArgs{
 * 			VpcId:       exampleNetwork.ID(),
 * 			CidrBlock:   pulumi.String("172.16.0.0/24"),
 * 			ZoneId:      pulumi.String(example.Zones[0].Id),
 * 			VswitchName: pulumi.String("terraform-example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleInstance, err := rds.NewInstance(ctx, "example", &rds.InstanceArgs{
 * 			Engine:                pulumi.String("PostgreSQL"),
 * 			EngineVersion:         pulumi.String("13.0"),
 * 			DbInstanceStorageType: pulumi.String("cloud_essd"),
 * 			InstanceType:          pulumi.String(exampleGetInstanceClasses.InstanceClasses[0].InstanceClass),
 * 			InstanceStorage:       pulumi.String(exampleGetInstanceClasses.InstanceClasses[0].StorageRange.Min),
 * 			InstanceChargeType:    pulumi.String("Postpaid"),
 * 			InstanceName:          pulumi.String("terraform-example"),
 * 			VswitchId:             exampleSwitch.ID(),
 * 			MonitoringPeriod:      pulumi.Int(60),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewRdsUpgradeDbInstance(ctx, "example", &rds.RdsUpgradeDbInstanceArgs{
 * 			SourceDbInstanceId:    exampleInstance.ID(),
 * 			TargetMajorVersion:    pulumi.String("14.0"),
 * 			DbInstanceClass:       exampleInstance.InstanceType,
 * 			DbInstanceStorage:     exampleInstance.InstanceStorage,
 * 			DbInstanceStorageType: exampleInstance.DbInstanceStorageType,
 * 			InstanceNetworkType:   pulumi.String("VPC"),
 * 			CollectStatMode:       pulumi.String("After"),
 * 			SwitchOver:            pulumi.String("false"),
 * 			PaymentType:           pulumi.String("PayAsYouGo"),
 * 			DbInstanceDescription: pulumi.String("terraform-example"),
 * 			VswitchId:             exampleSwitch.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.alicloud.rds.RdsFunctions;
 * import com.pulumi.alicloud.rds.inputs.GetZonesArgs;
 * import com.pulumi.alicloud.rds.inputs.GetInstanceClassesArgs;
 * import com.pulumi.alicloud.rds.inputs.GetCrossRegionsArgs;
 * import com.pulumi.alicloud.vpc.Network;
 * import com.pulumi.alicloud.vpc.NetworkArgs;
 * import com.pulumi.alicloud.vpc.Switch;
 * import com.pulumi.alicloud.vpc.SwitchArgs;
 * import com.pulumi.alicloud.rds.Instance;
 * import com.pulumi.alicloud.rds.InstanceArgs;
 * import com.pulumi.alicloud.rds.RdsUpgradeDbInstance;
 * import com.pulumi.alicloud.rds.RdsUpgradeDbInstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var example = RdsFunctions.getZones(GetZonesArgs.builder()
 *             .engine("PostgreSQL")
 *             .engineVersion("13.0")
 *             .instanceChargeType("PostPaid")
 *             .category("HighAvailability")
 *             .dbInstanceStorageType("cloud_essd")
 *             .build());
 *         final var exampleGetInstanceClasses = RdsFunctions.getInstanceClasses(GetInstanceClassesArgs.builder()
 *             .zoneId(example.applyValue(getZonesResult -> getZonesResult.zones()[0].id()))
 *             .engine("PostgreSQL")
 *             .engineVersion("13.0")
 *             .category("HighAvailability")
 *             .dbInstanceStorageType("cloud_essd")
 *             .instanceChargeType("PostPaid")
 *             .build());
 *         final var exampleGetCrossRegions = RdsFunctions.getCrossRegions();
 *         var exampleNetwork = new Network("exampleNetwork", NetworkArgs.builder()
 *             .vpcName("terraform-example")
 *             .cidrBlock("172.16.0.0/16")
 *             .build());
 *         var exampleSwitch = new Switch("exampleSwitch", SwitchArgs.builder()
 *             .vpcId(exampleNetwork.id())
 *             .cidrBlock("172.16.0.0/24")
 *             .zoneId(example.applyValue(getZonesResult -> getZonesResult.zones()[0].id()))
 *             .vswitchName("terraform-example")
 *             .build());
 *         var exampleInstance = new Instance("exampleInstance", InstanceArgs.builder()
 *             .engine("PostgreSQL")
 *             .engineVersion("13.0")
 *             .dbInstanceStorageType("cloud_essd")
 *             .instanceType(exampleGetInstanceClasses.applyValue(getInstanceClassesResult -> getInstanceClassesResult.instanceClasses()[0].instanceClass()))
 *             .instanceStorage(exampleGetInstanceClasses.applyValue(getInstanceClassesResult -> getInstanceClassesResult.instanceClasses()[0].storageRange().min()))
 *             .instanceChargeType("Postpaid")
 *             .instanceName("terraform-example")
 *             .vswitchId(exampleSwitch.id())
 *             .monitoringPeriod("60")
 *             .build());
 *         var exampleRdsUpgradeDbInstance = new RdsUpgradeDbInstance("exampleRdsUpgradeDbInstance", RdsUpgradeDbInstanceArgs.builder()
 *             .sourceDbInstanceId(exampleInstance.id())
 *             .targetMajorVersion("14.0")
 *             .dbInstanceClass(exampleInstance.instanceType())
 *             .dbInstanceStorage(exampleInstance.instanceStorage())
 *             .dbInstanceStorageType(exampleInstance.dbInstanceStorageType())
 *             .instanceNetworkType("VPC")
 *             .collectStatMode("After")
 *             .switchOver("false")
 *             .paymentType("PayAsYouGo")
 *             .dbInstanceDescription("terraform-example")
 *             .vswitchId(exampleSwitch.id())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleNetwork:
 *     type: alicloud:vpc:Network
 *     name: example
 *     properties:
 *       vpcName: terraform-example
 *       cidrBlock: 172.16.0.0/16
 *   exampleSwitch:
 *     type: alicloud:vpc:Switch
 *     name: example
 *     properties:
 *       vpcId: ${exampleNetwork.id}
 *       cidrBlock: 172.16.0.0/24
 *       zoneId: ${example.zones[0].id}
 *       vswitchName: terraform-example
 *   exampleInstance:
 *     type: alicloud:rds:Instance
 *     name: example
 *     properties:
 *       engine: PostgreSQL
 *       engineVersion: '13.0'
 *       dbInstanceStorageType: cloud_essd
 *       instanceType: ${exampleGetInstanceClasses.instanceClasses[0].instanceClass}
 *       instanceStorage: ${exampleGetInstanceClasses.instanceClasses[0].storageRange.min}
 *       instanceChargeType: Postpaid
 *       instanceName: terraform-example
 *       vswitchId: ${exampleSwitch.id}
 *       monitoringPeriod: '60'
 *   exampleRdsUpgradeDbInstance:
 *     type: alicloud:rds:RdsUpgradeDbInstance
 *     name: example
 *     properties:
 *       sourceDbInstanceId: ${exampleInstance.id}
 *       targetMajorVersion: '14.0'
 *       dbInstanceClass: ${exampleInstance.instanceType}
 *       dbInstanceStorage: ${exampleInstance.instanceStorage}
 *       dbInstanceStorageType: ${exampleInstance.dbInstanceStorageType}
 *       instanceNetworkType: VPC
 *       collectStatMode: After
 *       switchOver: 'false'
 *       paymentType: PayAsYouGo
 *       dbInstanceDescription: terraform-example
 *       vswitchId: ${exampleSwitch.id}
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: alicloud:rds:getZones
 *       Arguments:
 *         engine: PostgreSQL
 *         engineVersion: '13.0'
 *         instanceChargeType: PostPaid
 *         category: HighAvailability
 *         dbInstanceStorageType: cloud_essd
 *   exampleGetInstanceClasses:
 *     fn::invoke:
 *       Function: alicloud:rds:getInstanceClasses
 *       Arguments:
 *         zoneId: ${example.zones[0].id}
 *         engine: PostgreSQL
 *         engineVersion: '13.0'
 *         category: HighAvailability
 *         dbInstanceStorageType: cloud_essd
 *         instanceChargeType: PostPaid
 *   exampleGetCrossRegions:
 *     fn::invoke:
 *       Function: alicloud:rds:getCrossRegions
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * RDS Upgrade DB Instance can be imported using the id, e.g.
 * ```sh
 * $ pulumi import alicloud:rds/rdsUpgradeDbInstance:RdsUpgradeDbInstance example <id>
 * ```
 * @property acl This parameter is only supported by the RDS PostgreSQL cloud disk version. This parameter indicates the authentication method. It is allowed only when the public key of the client certificate authority is enabled. Valid values: `cert` and `perfer` and `verify-ca` and `verify-full (supported by RDS PostgreSQL above 12)`.
 * @property autoUpgradeMinorVersion How to upgrade the minor version of the instance. Valid values:
 * * **Auto**: automatically upgrade the minor version.
 * * **Manual**: It is not automatically upgraded. It is only mandatory when the current version is offline.
 * @property caType This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the certificate type. When the value of ssl_action is Open, the default value of this parameter is aliyun. Value range:
 * * **aliyun**: using cloud certificates.
 * * **custom**: use a custom certificate. Valid values: `aliyun`, `custom`.
 * @property certificate The file that contains the certificate used for TDE.
 * @property clientCaCert This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the public key of the client certification authority. If the value of client_ca_enabled is 1, this parameter must be configured.
 * @property clientCaEnabled The client ca enabled.
 * @property clientCertRevocationList This parameter is only supported by the RDS PostgreSQL cloud disk version, which indicates that the client revokes the certificate file. If the value of client_crl_enabled is 1, this parameter must be configured.
 * @property clientCrlEnabled The client crl enabled.
 * @property collectStatMode The time at which ApsaraDB RDS collects the statistics of the new instance.
 * * **Before**: ApsaraDB RDS collects the statistics of the new instance before the switchover to ensure service stability. If the original instance contains a large amount of data, the upgrade may require a long period of time.
 * * **After**: ApsaraDB RDS collects the statistics of the new instance after the switchover to accelerate the upgrade. If you access tables for which no statistics are generated, the query plans that you specify may be inaccurately executed. In addition, your database service may be unavailable during peak hours.
 * > **NOTE** If you set the SwitchOver parameter to false, the value Before specifies that ApsaraDB RDS collects the statistics of the new instance before the new instance starts to process read and write requests, and the value After specifies that ApsaraDB RDS collects the statistics of the new instance after the new instance starts to process read and write requests.
 * @property connectionStringPrefix The connection string prefix.
 * @property dbInstanceClass The instance type of the new instance. For information, see [Primary ApsaraDB RDS instance types](https://www.alibabacloud.com/help/en/rds/product-overview/primary-apsaradb-rds-instance-types).
 * @property dbInstanceDescription The db instance description.
 * @property dbInstanceStorage The storage capacity of the new instance. Unit: GB. The storage capacity increases in increments of 5 GB. For more information, see [Primary ApsaraDB RDS instance types](https://www.alibabacloud.com/help/en/rds/product-overview/primary-apsaradb-rds-instance-types).
 * > **NOTE:** The default value of this parameter is the storage capacity of the original instance.
 * @property dbInstanceStorageType The type of storage media that is used for the new instance. Valid values:
 * * **local_ssd**: local SSDs.
 * * **cloud_ssd**: standard SSDs.
 * * **cloud_essd**: enhanced SSDs (ESSDs) of performance level 1 (PL1).
 * * **cloud_essd2**: ESSDs of PL2.
 * * **cloud_essd3**: ESSDs of PL3.
 * @property dbName The name of the database for which you want to enable TDE. Up to 50 names can be entered in a single request. If you specify multiple names, separate these names with commas (,).
 * > **NOTE:** This parameter is available and must be specified only when the instance runs SQL Server 2019 SE or an Enterprise Edition of SQL Server.
 * @property dedicatedHostGroupId The ID of the dedicated cluster to which the new instance belongs. This parameter takes effect only when you create the new instance in a dedicated cluster.
 * @property deletionProtection The switch of delete protection. Valid values:
 * - true: delete protect.
 * - false: no delete protect.
 * > **NOTE:** `deletion_protection` is valid only when attribute `payment_type` is set to `PayAsYouGo`, supported engine type: **MySQL**, **PostgreSQL**, **MariaDB**, **MSSQL**.
 * @property direction The direction. Valid values: `Auto`, `Down`, `TempUpgrade`, `Up`.
 * @property effectiveTime The effective time.
 * @property encryptionKey The ID of the private key.
 * > **NOTE:** This parameter is available only when the instance runs MySQL.
 * @property engine Database type. Value options: MySQL, SQLServer, PostgreSQL.
 * @property engineVersion Database version. Value:
 * * MySQL: **5.5/5.6/5.7/8.0**.
 * * SQL Server: **2008r2/08r2_ent_ha/2012/2012_ent_ha/2012_std_ha/2012_web/2014_std_ha/2016_ent_ha/2016_std_ha/2016_web/2017_std_ha/2017_ent/2019_std_ha/2019_ent**.
 * * PostgreSQL: **9.4/10.0/11.0/12.0/13.0**.
 * * MariaDB: **10.3**.
 * @property forceRestart Set it to true to make some parameter efficient when modifying them. Default to false.
 * @property haMode The high availability mode. Valid values:
 * * **RPO**: Data persistence is preferred. The instance preferentially ensures data reliability to minimize data loss. Use this mode if you have higher requirements on data consistency.
 * * **RTO**: Instance availability is preferred. The instance restores services as soon as possible to ensure availability. Use this mode if you have higher requirements on service availability.
 * @property instanceNetworkType The network type of the instance. Valid values:
 * * **Classic**: Classic Network.
 * * **VPC**: VPC.
 * @property maintainTime The maintainable time period of the instance. Format: <I> HH:mm</I> Z-<I> HH:mm</I> Z(UTC time).
 * @property parameters Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
 * @property password The password of the certificate.
 * > **NOTE:** This parameter is available only when the instance runs SQL Server 2019 SE or an Enterprise Edition of SQL Server.
 * @property paymentType The billing method of the new instance. Valid values: `PayAsYouGo` and `Subscription`.
 * @property pgHbaConfs The configuration of [AD domain](https://www.alibabacloud.com/help/en/doc-detail/349288.htm) . See `pg_hba_conf` below.
 * @property port The port.
 * @property privateIpAddress The intranet IP address of the new instance must be within the specified vSwitch IP address range. By default, the system automatically allocates by using **VPCId** and **VSwitchId**.
 * @property privateKey The file that contains the private key used for TDE.
 * @property releasedKeepPolicy The released keep policy.
 * @property replicationAcl This parameter is only supported by the RDS PostgreSQL cloud disk version, indicating the authentication method of the replication permission. It is only allowed when the public key of the client certificate authority is enabled. Valid values: `cert` and `perfer` and `verify-ca` and `verify-full (supported by RDS PostgreSQL above 12)`.
 * @property resourceGroupId The resource group id.
 * @property roleArn The Alibaba Cloud Resource Name (ARN) of a RAM role. A RAM role is a virtual RAM identity that you can create within your Alibaba Cloud account. For more information, see [RAM role overview](https://www.alibabacloud.com/help/en/ram/user-guide/ram-role-overview).
 * > **NOTE:** This parameter is available only when the instance runs MySQL.
 * @property securityIps The IP address whitelist of the instance. Separate multiple IP addresses with commas (,) and cannot be repeated. The following two formats are supported:
 * * IP address form, for example: 10.23.12.24.
 * * CIDR format, for example, 10.23.12.0/24 (no Inter-Domain Routing, 24 indicates the length of the prefix in the address, ranging from 1 to 32).
 * > **NOTE:** each instance can add up to 1000 IP addresses or IP segments, that is, the total number of IP addresses or IP segments in all IP whitelist groups cannot exceed 1000. When there are more IP addresses, it is recommended to merge them into IP segments, for example, 10.23.12.0/24.
 * @property serverCert This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the content of the server certificate. If the CAType value is custom, this parameter must be configured.
 * @property serverKey This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the private key of the server certificate. If the value of CAType is custom, this parameter must be configured.
 * @property sourceBiz The source biz.
 * @property sourceDbInstanceId The source db instance id.
 * @property sslEnabled Enable or disable SSL. Valid values: `0` and `1`.
 * @property switchOver Specifies whether ApsaraDB RDS automatically switches your workloads over to the new instance after data is migrated to the new instance. Valid values:
 * * **true**: ApsaraDB RDS automatically switches workloads over to the new instance. If you set this parameter to true, you must take note of the following information:
 * * After the switchover is complete, you cannot roll your workloads back to the original instance. Proceed with caution.
 * * During the switchover, the original instance processes only read requests. You must perform the switchover during off-peak hours.
 * * If read-only instances are attached to the original instance, you can set this parameter only to false. In this case, the read-only instances that are attached to the original instance cannot be cloned. After the upgrade is complete, you must create read-only instances for the new instance.
 * * **false**: ApsaraDB RDS does not automatically switch your workloads over to the new instance. Before you perform an upgrade, we recommend that you set this parameter to false to test whether the new major engine version is compatible with your workloads. If you set this parameter to false, you must take note of the following information:
 * * The data migration does not interrupt your workloads on the original instance.
 * * After data is migrated to the new instance, you must update the endpoint configuration on your application. This update requires you to replace the endpoint of the original instance with the endpoint of the new instance. For more information about how to view the endpoint of an instance, see [View and change the internal and public endpoints and port numbers of an ApsaraDB RDS for PostgreSQL instance](https://www.alibabacloud.com/help/doc-detail/96788.htm).
 * @property switchTime The time at which you want to apply the specification changes. The time follows the ISO 8601 standard in the yyyy-MM-ddTHH:mm:ssZ format. The time must be in UTC.
 * @property switchTimeMode The time at which ApsaraDB RDS switches your workloads over to the new instance. This parameter is used together with the SwitchOver parameter and takes effect only when you set the SwitchOver parameter to true. Valid values:
 * * **Immediate**: After data is migrated to the new instance, ApsaraDB RDS immediately switches your workloads over to the new instance.
 * * **MaintainTime**: After data is migrated to the new instance, ApsaraDB RDS switches your workloads over to the new instance during the maintenance window that you specify. You can call the [ModifyDBInstanceMaintainTime](https://www.alibabacloud.com/help/doc-detail/26249.htm) operation to change the maintenance window of an instance.
 * @property syncMode [The data replication mode](https://www.alibabacloud.com/help/doc-detail/96055.htm). Valid values:
 * * **Sync**: strong synchronization.
 * * **Semi-sync**: Semi-synchronous.
 * * **Async**: asynchronous.
 * > **NOTE:** SQL Server 2017 cluster version is currently not supported.
 * @property targetMajorVersion The major engine version of the new instance. The value of this parameter must be the major engine version on which an upgrade check is performed.
 * > **NOTE** You can call the [UpgradeDBInstanceMajorVersionPrecheck](https://www.alibabacloud.com/help/doc-detail/330050.htm) operation to perform an upgrade check on a major engine version.
 * @property tcpConnectionType The availability check method of the instance. Valid values:
 * - **SHORT**: Alibaba Cloud uses short-lived connections to check the availability of the instance.
 * - **LONG**: Alibaba Cloud uses persistent connections to check the availability of the instance.
 * @property tdeStatus Specifies whether to enable TDE. Valid values: `Enabled` and `Disabled`.
 * @property vpcId The ID of the VPC to which the new instance belongs.
 * > **NOTE:** Make sure that the VPC resides in the specified region.
 * @property vswitchId The ID of the vSwitch associated with the specified VPC.
 * > **NOTE:** Make sure that the vSwitch belongs to the specified VPC and region.
 * @property zoneId The ID of the zone to which the new instance belongs. You can call the [DescribeRegions](https://www.alibabacloud.com/help/en/rds/developer-reference/api-rds-2014-08-15-describeregions) operation to query the most recent region list.
 * > **NOTE:** The default value of this parameter is the ID of the zone to which the original instance belongs.
 * @property zoneIdSlave1 The ID of the zone to which the secondary instance of the new instance belongs. You can specify this parameter only when the original instance runs RDS High-availability Edition. You can select a zone that belongs to the region where the original instance resides. You can call the [DescribeRegions](https://www.alibabacloud.com/help/en/rds/developer-reference/api-rds-2014-08-15-describeregions) operation to query zone IDs.
 */
public data class RdsUpgradeDbInstanceArgs(
    public val acl: Output<String>? = null,
    public val autoUpgradeMinorVersion: Output<String>? = null,
    public val caType: Output<String>? = null,
    public val certificate: Output<String>? = null,
    public val clientCaCert: Output<String>? = null,
    public val clientCaEnabled: Output<Int>? = null,
    public val clientCertRevocationList: Output<String>? = null,
    public val clientCrlEnabled: Output<Int>? = null,
    public val collectStatMode: Output<String>? = null,
    public val connectionStringPrefix: Output<String>? = null,
    public val dbInstanceClass: Output<String>? = null,
    public val dbInstanceDescription: Output<String>? = null,
    public val dbInstanceStorage: Output<Int>? = null,
    public val dbInstanceStorageType: Output<String>? = null,
    public val dbName: Output<String>? = null,
    public val dedicatedHostGroupId: Output<String>? = null,
    public val deletionProtection: Output<Boolean>? = null,
    public val direction: Output<String>? = null,
    public val effectiveTime: Output<String>? = null,
    public val encryptionKey: Output<String>? = null,
    public val engine: Output<String>? = null,
    public val engineVersion: Output<String>? = null,
    public val forceRestart: Output<Boolean>? = null,
    public val haMode: Output<String>? = null,
    public val instanceNetworkType: Output<String>? = null,
    public val maintainTime: Output<String>? = null,
    public val parameters: Output<List<RdsUpgradeDbInstanceParameterArgs>>? = null,
    public val password: Output<String>? = null,
    public val paymentType: Output<String>? = null,
    public val pgHbaConfs: Output<List<RdsUpgradeDbInstancePgHbaConfArgs>>? = null,
    public val port: Output<String>? = null,
    public val privateIpAddress: Output<String>? = null,
    public val privateKey: Output<String>? = null,
    public val releasedKeepPolicy: Output<String>? = null,
    public val replicationAcl: Output<String>? = null,
    public val resourceGroupId: Output<String>? = null,
    public val roleArn: Output<String>? = null,
    public val securityIps: Output<List<String>>? = null,
    public val serverCert: Output<String>? = null,
    public val serverKey: Output<String>? = null,
    public val sourceBiz: Output<String>? = null,
    public val sourceDbInstanceId: Output<String>? = null,
    public val sslEnabled: Output<Int>? = null,
    public val switchOver: Output<String>? = null,
    public val switchTime: Output<String>? = null,
    public val switchTimeMode: Output<String>? = null,
    public val syncMode: Output<String>? = null,
    public val targetMajorVersion: Output<String>? = null,
    public val tcpConnectionType: Output<String>? = null,
    public val tdeStatus: Output<String>? = null,
    public val vpcId: Output<String>? = null,
    public val vswitchId: Output<String>? = null,
    public val zoneId: Output<String>? = null,
    public val zoneIdSlave1: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.alicloud.rds.RdsUpgradeDbInstanceArgs> {
    override fun toJava(): com.pulumi.alicloud.rds.RdsUpgradeDbInstanceArgs =
        com.pulumi.alicloud.rds.RdsUpgradeDbInstanceArgs.builder()
            .acl(acl?.applyValue({ args0 -> args0 }))
            .autoUpgradeMinorVersion(autoUpgradeMinorVersion?.applyValue({ args0 -> args0 }))
            .caType(caType?.applyValue({ args0 -> args0 }))
            .certificate(certificate?.applyValue({ args0 -> args0 }))
            .clientCaCert(clientCaCert?.applyValue({ args0 -> args0 }))
            .clientCaEnabled(clientCaEnabled?.applyValue({ args0 -> args0 }))
            .clientCertRevocationList(clientCertRevocationList?.applyValue({ args0 -> args0 }))
            .clientCrlEnabled(clientCrlEnabled?.applyValue({ args0 -> args0 }))
            .collectStatMode(collectStatMode?.applyValue({ args0 -> args0 }))
            .connectionStringPrefix(connectionStringPrefix?.applyValue({ args0 -> args0 }))
            .dbInstanceClass(dbInstanceClass?.applyValue({ args0 -> args0 }))
            .dbInstanceDescription(dbInstanceDescription?.applyValue({ args0 -> args0 }))
            .dbInstanceStorage(dbInstanceStorage?.applyValue({ args0 -> args0 }))
            .dbInstanceStorageType(dbInstanceStorageType?.applyValue({ args0 -> args0 }))
            .dbName(dbName?.applyValue({ args0 -> args0 }))
            .dedicatedHostGroupId(dedicatedHostGroupId?.applyValue({ args0 -> args0 }))
            .deletionProtection(deletionProtection?.applyValue({ args0 -> args0 }))
            .direction(direction?.applyValue({ args0 -> args0 }))
            .effectiveTime(effectiveTime?.applyValue({ args0 -> args0 }))
            .encryptionKey(encryptionKey?.applyValue({ args0 -> args0 }))
            .engine(engine?.applyValue({ args0 -> args0 }))
            .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
            .forceRestart(forceRestart?.applyValue({ args0 -> args0 }))
            .haMode(haMode?.applyValue({ args0 -> args0 }))
            .instanceNetworkType(instanceNetworkType?.applyValue({ args0 -> args0 }))
            .maintainTime(maintainTime?.applyValue({ args0 -> args0 }))
            .parameters(
                parameters?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .password(password?.applyValue({ args0 -> args0 }))
            .paymentType(paymentType?.applyValue({ args0 -> args0 }))
            .pgHbaConfs(
                pgHbaConfs?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .port(port?.applyValue({ args0 -> args0 }))
            .privateIpAddress(privateIpAddress?.applyValue({ args0 -> args0 }))
            .privateKey(privateKey?.applyValue({ args0 -> args0 }))
            .releasedKeepPolicy(releasedKeepPolicy?.applyValue({ args0 -> args0 }))
            .replicationAcl(replicationAcl?.applyValue({ args0 -> args0 }))
            .resourceGroupId(resourceGroupId?.applyValue({ args0 -> args0 }))
            .roleArn(roleArn?.applyValue({ args0 -> args0 }))
            .securityIps(securityIps?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .serverCert(serverCert?.applyValue({ args0 -> args0 }))
            .serverKey(serverKey?.applyValue({ args0 -> args0 }))
            .sourceBiz(sourceBiz?.applyValue({ args0 -> args0 }))
            .sourceDbInstanceId(sourceDbInstanceId?.applyValue({ args0 -> args0 }))
            .sslEnabled(sslEnabled?.applyValue({ args0 -> args0 }))
            .switchOver(switchOver?.applyValue({ args0 -> args0 }))
            .switchTime(switchTime?.applyValue({ args0 -> args0 }))
            .switchTimeMode(switchTimeMode?.applyValue({ args0 -> args0 }))
            .syncMode(syncMode?.applyValue({ args0 -> args0 }))
            .targetMajorVersion(targetMajorVersion?.applyValue({ args0 -> args0 }))
            .tcpConnectionType(tcpConnectionType?.applyValue({ args0 -> args0 }))
            .tdeStatus(tdeStatus?.applyValue({ args0 -> args0 }))
            .vpcId(vpcId?.applyValue({ args0 -> args0 }))
            .vswitchId(vswitchId?.applyValue({ args0 -> args0 }))
            .zoneId(zoneId?.applyValue({ args0 -> args0 }))
            .zoneIdSlave1(zoneIdSlave1?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [RdsUpgradeDbInstanceArgs].
 */
@PulumiTagMarker
public class RdsUpgradeDbInstanceArgsBuilder internal constructor() {
    private var acl: Output<String>? = null

    private var autoUpgradeMinorVersion: Output<String>? = null

    private var caType: Output<String>? = null

    private var certificate: Output<String>? = null

    private var clientCaCert: Output<String>? = null

    private var clientCaEnabled: Output<Int>? = null

    private var clientCertRevocationList: Output<String>? = null

    private var clientCrlEnabled: Output<Int>? = null

    private var collectStatMode: Output<String>? = null

    private var connectionStringPrefix: Output<String>? = null

    private var dbInstanceClass: Output<String>? = null

    private var dbInstanceDescription: Output<String>? = null

    private var dbInstanceStorage: Output<Int>? = null

    private var dbInstanceStorageType: Output<String>? = null

    private var dbName: Output<String>? = null

    private var dedicatedHostGroupId: Output<String>? = null

    private var deletionProtection: Output<Boolean>? = null

    private var direction: Output<String>? = null

    private var effectiveTime: Output<String>? = null

    private var encryptionKey: Output<String>? = null

    private var engine: Output<String>? = null

    private var engineVersion: Output<String>? = null

    private var forceRestart: Output<Boolean>? = null

    private var haMode: Output<String>? = null

    private var instanceNetworkType: Output<String>? = null

    private var maintainTime: Output<String>? = null

    private var parameters: Output<List<RdsUpgradeDbInstanceParameterArgs>>? = null

    private var password: Output<String>? = null

    private var paymentType: Output<String>? = null

    private var pgHbaConfs: Output<List<RdsUpgradeDbInstancePgHbaConfArgs>>? = null

    private var port: Output<String>? = null

    private var privateIpAddress: Output<String>? = null

    private var privateKey: Output<String>? = null

    private var releasedKeepPolicy: Output<String>? = null

    private var replicationAcl: Output<String>? = null

    private var resourceGroupId: Output<String>? = null

    private var roleArn: Output<String>? = null

    private var securityIps: Output<List<String>>? = null

    private var serverCert: Output<String>? = null

    private var serverKey: Output<String>? = null

    private var sourceBiz: Output<String>? = null

    private var sourceDbInstanceId: Output<String>? = null

    private var sslEnabled: Output<Int>? = null

    private var switchOver: Output<String>? = null

    private var switchTime: Output<String>? = null

    private var switchTimeMode: Output<String>? = null

    private var syncMode: Output<String>? = null

    private var targetMajorVersion: Output<String>? = null

    private var tcpConnectionType: Output<String>? = null

    private var tdeStatus: Output<String>? = null

    private var vpcId: Output<String>? = null

    private var vswitchId: Output<String>? = null

    private var zoneId: Output<String>? = null

    private var zoneIdSlave1: Output<String>? = null

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version. This parameter indicates the authentication method. It is allowed only when the public key of the client certificate authority is enabled. Valid values: `cert` and `perfer` and `verify-ca` and `verify-full (supported by RDS PostgreSQL above 12)`.
     */
    @JvmName("phewfeybxwyojfyc")
    public suspend fun acl(`value`: Output<String>) {
        this.acl = value
    }

    /**
     * @param value How to upgrade the minor version of the instance. Valid values:
     * * **Auto**: automatically upgrade the minor version.
     * * **Manual**: It is not automatically upgraded. It is only mandatory when the current version is offline.
     */
    @JvmName("pwypvumxpsmtpddq")
    public suspend fun autoUpgradeMinorVersion(`value`: Output<String>) {
        this.autoUpgradeMinorVersion = value
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the certificate type. When the value of ssl_action is Open, the default value of this parameter is aliyun. Value range:
     * * **aliyun**: using cloud certificates.
     * * **custom**: use a custom certificate. Valid values: `aliyun`, `custom`.
     */
    @JvmName("qpucofnsyodwfiwa")
    public suspend fun caType(`value`: Output<String>) {
        this.caType = value
    }

    /**
     * @param value The file that contains the certificate used for TDE.
     */
    @JvmName("ewmvjnjhtbspljci")
    public suspend fun certificate(`value`: Output<String>) {
        this.certificate = value
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the public key of the client certification authority. If the value of client_ca_enabled is 1, this parameter must be configured.
     */
    @JvmName("rikibneoxlmtijid")
    public suspend fun clientCaCert(`value`: Output<String>) {
        this.clientCaCert = value
    }

    /**
     * @param value The client ca enabled.
     */
    @JvmName("yhggmucmuulstbub")
    public suspend fun clientCaEnabled(`value`: Output<Int>) {
        this.clientCaEnabled = value
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version, which indicates that the client revokes the certificate file. If the value of client_crl_enabled is 1, this parameter must be configured.
     */
    @JvmName("ykroqoxsnygovnen")
    public suspend fun clientCertRevocationList(`value`: Output<String>) {
        this.clientCertRevocationList = value
    }

    /**
     * @param value The client crl enabled.
     */
    @JvmName("lvrhxuruiollrjef")
    public suspend fun clientCrlEnabled(`value`: Output<Int>) {
        this.clientCrlEnabled = value
    }

    /**
     * @param value The time at which ApsaraDB RDS collects the statistics of the new instance.
     * * **Before**: ApsaraDB RDS collects the statistics of the new instance before the switchover to ensure service stability. If the original instance contains a large amount of data, the upgrade may require a long period of time.
     * * **After**: ApsaraDB RDS collects the statistics of the new instance after the switchover to accelerate the upgrade. If you access tables for which no statistics are generated, the query plans that you specify may be inaccurately executed. In addition, your database service may be unavailable during peak hours.
     * > **NOTE** If you set the SwitchOver parameter to false, the value Before specifies that ApsaraDB RDS collects the statistics of the new instance before the new instance starts to process read and write requests, and the value After specifies that ApsaraDB RDS collects the statistics of the new instance after the new instance starts to process read and write requests.
     */
    @JvmName("jmrdjsybpdhokppn")
    public suspend fun collectStatMode(`value`: Output<String>) {
        this.collectStatMode = value
    }

    /**
     * @param value The connection string prefix.
     */
    @JvmName("dlovbmlaayncrvhr")
    public suspend fun connectionStringPrefix(`value`: Output<String>) {
        this.connectionStringPrefix = value
    }

    /**
     * @param value The instance type of the new instance. For information, see [Primary ApsaraDB RDS instance types](https://www.alibabacloud.com/help/en/rds/product-overview/primary-apsaradb-rds-instance-types).
     */
    @JvmName("wmmfrtnvjnotqnox")
    public suspend fun dbInstanceClass(`value`: Output<String>) {
        this.dbInstanceClass = value
    }

    /**
     * @param value The db instance description.
     */
    @JvmName("raureewotfktjpum")
    public suspend fun dbInstanceDescription(`value`: Output<String>) {
        this.dbInstanceDescription = value
    }

    /**
     * @param value The storage capacity of the new instance. Unit: GB. The storage capacity increases in increments of 5 GB. For more information, see [Primary ApsaraDB RDS instance types](https://www.alibabacloud.com/help/en/rds/product-overview/primary-apsaradb-rds-instance-types).
     * > **NOTE:** The default value of this parameter is the storage capacity of the original instance.
     */
    @JvmName("htetgenobffjnsyp")
    public suspend fun dbInstanceStorage(`value`: Output<Int>) {
        this.dbInstanceStorage = value
    }

    /**
     * @param value The type of storage media that is used for the new instance. Valid values:
     * * **local_ssd**: local SSDs.
     * * **cloud_ssd**: standard SSDs.
     * * **cloud_essd**: enhanced SSDs (ESSDs) of performance level 1 (PL1).
     * * **cloud_essd2**: ESSDs of PL2.
     * * **cloud_essd3**: ESSDs of PL3.
     */
    @JvmName("wweixekljvjexppi")
    public suspend fun dbInstanceStorageType(`value`: Output<String>) {
        this.dbInstanceStorageType = value
    }

    /**
     * @param value The name of the database for which you want to enable TDE. Up to 50 names can be entered in a single request. If you specify multiple names, separate these names with commas (,).
     * > **NOTE:** This parameter is available and must be specified only when the instance runs SQL Server 2019 SE or an Enterprise Edition of SQL Server.
     */
    @JvmName("unmbkuyywafsdwop")
    public suspend fun dbName(`value`: Output<String>) {
        this.dbName = value
    }

    /**
     * @param value The ID of the dedicated cluster to which the new instance belongs. This parameter takes effect only when you create the new instance in a dedicated cluster.
     */
    @JvmName("oihpoegycfoimhkf")
    public suspend fun dedicatedHostGroupId(`value`: Output<String>) {
        this.dedicatedHostGroupId = value
    }

    /**
     * @param value The switch of delete protection. Valid values:
     * - true: delete protect.
     * - false: no delete protect.
     * > **NOTE:** `deletion_protection` is valid only when attribute `payment_type` is set to `PayAsYouGo`, supported engine type: **MySQL**, **PostgreSQL**, **MariaDB**, **MSSQL**.
     */
    @JvmName("pcmarfuedmbcaxkh")
    public suspend fun deletionProtection(`value`: Output<Boolean>) {
        this.deletionProtection = value
    }

    /**
     * @param value The direction. Valid values: `Auto`, `Down`, `TempUpgrade`, `Up`.
     */
    @JvmName("ywdjnrnxmnykngid")
    public suspend fun direction(`value`: Output<String>) {
        this.direction = value
    }

    /**
     * @param value The effective time.
     */
    @JvmName("alktgkaaiibptuus")
    public suspend fun effectiveTime(`value`: Output<String>) {
        this.effectiveTime = value
    }

    /**
     * @param value The ID of the private key.
     * > **NOTE:** This parameter is available only when the instance runs MySQL.
     */
    @JvmName("vxkswvchhgriqeed")
    public suspend fun encryptionKey(`value`: Output<String>) {
        this.encryptionKey = value
    }

    /**
     * @param value Database type. Value options: MySQL, SQLServer, PostgreSQL.
     */
    @JvmName("sxcexxuuatiwrptb")
    public suspend fun engine(`value`: Output<String>) {
        this.engine = value
    }

    /**
     * @param value Database version. Value:
     * * MySQL: **5.5/5.6/5.7/8.0**.
     * * SQL Server: **2008r2/08r2_ent_ha/2012/2012_ent_ha/2012_std_ha/2012_web/2014_std_ha/2016_ent_ha/2016_std_ha/2016_web/2017_std_ha/2017_ent/2019_std_ha/2019_ent**.
     * * PostgreSQL: **9.4/10.0/11.0/12.0/13.0**.
     * * MariaDB: **10.3**.
     */
    @JvmName("siudnwyuqacxjdhs")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value Set it to true to make some parameter efficient when modifying them. Default to false.
     */
    @JvmName("yushmbpfdsvmfyug")
    public suspend fun forceRestart(`value`: Output<Boolean>) {
        this.forceRestart = value
    }

    /**
     * @param value The high availability mode. Valid values:
     * * **RPO**: Data persistence is preferred. The instance preferentially ensures data reliability to minimize data loss. Use this mode if you have higher requirements on data consistency.
     * * **RTO**: Instance availability is preferred. The instance restores services as soon as possible to ensure availability. Use this mode if you have higher requirements on service availability.
     */
    @JvmName("jarvekxswumfrtgs")
    public suspend fun haMode(`value`: Output<String>) {
        this.haMode = value
    }

    /**
     * @param value The network type of the instance. Valid values:
     * * **Classic**: Classic Network.
     * * **VPC**: VPC.
     */
    @JvmName("ortrlnkqfhxaerhi")
    public suspend fun instanceNetworkType(`value`: Output<String>) {
        this.instanceNetworkType = value
    }

    /**
     * @param value The maintainable time period of the instance. Format: <I> HH:mm</I> Z-<I> HH:mm</I> Z(UTC time).
     */
    @JvmName("wswtwmoquqpoueio")
    public suspend fun maintainTime(`value`: Output<String>) {
        this.maintainTime = value
    }

    /**
     * @param value Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("pdyeaeyheqnfdrnh")
    public suspend fun parameters(`value`: Output<List<RdsUpgradeDbInstanceParameterArgs>>) {
        this.parameters = value
    }

    @JvmName("ebevgrqwbhcdgffv")
    public suspend fun parameters(vararg values: Output<RdsUpgradeDbInstanceParameterArgs>) {
        this.parameters = Output.all(values.asList())
    }

    /**
     * @param values Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("auetmreksgeaesmd")
    public suspend fun parameters(values: List<Output<RdsUpgradeDbInstanceParameterArgs>>) {
        this.parameters = Output.all(values)
    }

    /**
     * @param value The password of the certificate.
     * > **NOTE:** This parameter is available only when the instance runs SQL Server 2019 SE or an Enterprise Edition of SQL Server.
     */
    @JvmName("cebxxvuvfbbnnhcf")
    public suspend fun password(`value`: Output<String>) {
        this.password = value
    }

    /**
     * @param value The billing method of the new instance. Valid values: `PayAsYouGo` and `Subscription`.
     */
    @JvmName("roatmxqfpcoxlsoy")
    public suspend fun paymentType(`value`: Output<String>) {
        this.paymentType = value
    }

    /**
     * @param value The configuration of [AD domain](https://www.alibabacloud.com/help/en/doc-detail/349288.htm) . See `pg_hba_conf` below.
     */
    @JvmName("bwpqmpfeftmylwbb")
    public suspend fun pgHbaConfs(`value`: Output<List<RdsUpgradeDbInstancePgHbaConfArgs>>) {
        this.pgHbaConfs = value
    }

    @JvmName("xyjokxmuhtdieqet")
    public suspend fun pgHbaConfs(vararg values: Output<RdsUpgradeDbInstancePgHbaConfArgs>) {
        this.pgHbaConfs = Output.all(values.asList())
    }

    /**
     * @param values The configuration of [AD domain](https://www.alibabacloud.com/help/en/doc-detail/349288.htm) . See `pg_hba_conf` below.
     */
    @JvmName("qxykonxwybvysise")
    public suspend fun pgHbaConfs(values: List<Output<RdsUpgradeDbInstancePgHbaConfArgs>>) {
        this.pgHbaConfs = Output.all(values)
    }

    /**
     * @param value The port.
     */
    @JvmName("pilvclvtjebquewt")
    public suspend fun port(`value`: Output<String>) {
        this.port = value
    }

    /**
     * @param value The intranet IP address of the new instance must be within the specified vSwitch IP address range. By default, the system automatically allocates by using **VPCId** and **VSwitchId**.
     */
    @JvmName("aspkeedghwrcicfb")
    public suspend fun privateIpAddress(`value`: Output<String>) {
        this.privateIpAddress = value
    }

    /**
     * @param value The file that contains the private key used for TDE.
     */
    @JvmName("ybgqtvllrmwyxbas")
    public suspend fun privateKey(`value`: Output<String>) {
        this.privateKey = value
    }

    /**
     * @param value The released keep policy.
     */
    @JvmName("qvqcvgovbrkrhagu")
    public suspend fun releasedKeepPolicy(`value`: Output<String>) {
        this.releasedKeepPolicy = value
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version, indicating the authentication method of the replication permission. It is only allowed when the public key of the client certificate authority is enabled. Valid values: `cert` and `perfer` and `verify-ca` and `verify-full (supported by RDS PostgreSQL above 12)`.
     */
    @JvmName("gnvscdkycxrpinyx")
    public suspend fun replicationAcl(`value`: Output<String>) {
        this.replicationAcl = value
    }

    /**
     * @param value The resource group id.
     */
    @JvmName("pnbpwnxfuwnccyqn")
    public suspend fun resourceGroupId(`value`: Output<String>) {
        this.resourceGroupId = value
    }

    /**
     * @param value The Alibaba Cloud Resource Name (ARN) of a RAM role. A RAM role is a virtual RAM identity that you can create within your Alibaba Cloud account. For more information, see [RAM role overview](https://www.alibabacloud.com/help/en/ram/user-guide/ram-role-overview).
     * > **NOTE:** This parameter is available only when the instance runs MySQL.
     */
    @JvmName("lwbhegawkcdgmwkp")
    public suspend fun roleArn(`value`: Output<String>) {
        this.roleArn = value
    }

    /**
     * @param value The IP address whitelist of the instance. Separate multiple IP addresses with commas (,) and cannot be repeated. The following two formats are supported:
     * * IP address form, for example: 10.23.12.24.
     * * CIDR format, for example, 10.23.12.0/24 (no Inter-Domain Routing, 24 indicates the length of the prefix in the address, ranging from 1 to 32).
     * > **NOTE:** each instance can add up to 1000 IP addresses or IP segments, that is, the total number of IP addresses or IP segments in all IP whitelist groups cannot exceed 1000. When there are more IP addresses, it is recommended to merge them into IP segments, for example, 10.23.12.0/24.
     */
    @JvmName("bhygimbqnkkkuppo")
    public suspend fun securityIps(`value`: Output<List<String>>) {
        this.securityIps = value
    }

    @JvmName("risriocfxumhyxhd")
    public suspend fun securityIps(vararg values: Output<String>) {
        this.securityIps = Output.all(values.asList())
    }

    /**
     * @param values The IP address whitelist of the instance. Separate multiple IP addresses with commas (,) and cannot be repeated. The following two formats are supported:
     * * IP address form, for example: 10.23.12.24.
     * * CIDR format, for example, 10.23.12.0/24 (no Inter-Domain Routing, 24 indicates the length of the prefix in the address, ranging from 1 to 32).
     * > **NOTE:** each instance can add up to 1000 IP addresses or IP segments, that is, the total number of IP addresses or IP segments in all IP whitelist groups cannot exceed 1000. When there are more IP addresses, it is recommended to merge them into IP segments, for example, 10.23.12.0/24.
     */
    @JvmName("oxqnhdgcmelesoks")
    public suspend fun securityIps(values: List<Output<String>>) {
        this.securityIps = Output.all(values)
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the content of the server certificate. If the CAType value is custom, this parameter must be configured.
     */
    @JvmName("gwfritvchmhudcoa")
    public suspend fun serverCert(`value`: Output<String>) {
        this.serverCert = value
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the private key of the server certificate. If the value of CAType is custom, this parameter must be configured.
     */
    @JvmName("colsjouasehselrg")
    public suspend fun serverKey(`value`: Output<String>) {
        this.serverKey = value
    }

    /**
     * @param value The source biz.
     */
    @JvmName("fadxfpwcolpqfkju")
    public suspend fun sourceBiz(`value`: Output<String>) {
        this.sourceBiz = value
    }

    /**
     * @param value The source db instance id.
     */
    @JvmName("egokkdahmksrcbvh")
    public suspend fun sourceDbInstanceId(`value`: Output<String>) {
        this.sourceDbInstanceId = value
    }

    /**
     * @param value Enable or disable SSL. Valid values: `0` and `1`.
     */
    @JvmName("rdemyqickicsxfkh")
    public suspend fun sslEnabled(`value`: Output<Int>) {
        this.sslEnabled = value
    }

    /**
     * @param value Specifies whether ApsaraDB RDS automatically switches your workloads over to the new instance after data is migrated to the new instance. Valid values:
     * * **true**: ApsaraDB RDS automatically switches workloads over to the new instance. If you set this parameter to true, you must take note of the following information:
     * * After the switchover is complete, you cannot roll your workloads back to the original instance. Proceed with caution.
     * * During the switchover, the original instance processes only read requests. You must perform the switchover during off-peak hours.
     * * If read-only instances are attached to the original instance, you can set this parameter only to false. In this case, the read-only instances that are attached to the original instance cannot be cloned. After the upgrade is complete, you must create read-only instances for the new instance.
     * * **false**: ApsaraDB RDS does not automatically switch your workloads over to the new instance. Before you perform an upgrade, we recommend that you set this parameter to false to test whether the new major engine version is compatible with your workloads. If you set this parameter to false, you must take note of the following information:
     * * The data migration does not interrupt your workloads on the original instance.
     * * After data is migrated to the new instance, you must update the endpoint configuration on your application. This update requires you to replace the endpoint of the original instance with the endpoint of the new instance. For more information about how to view the endpoint of an instance, see [View and change the internal and public endpoints and port numbers of an ApsaraDB RDS for PostgreSQL instance](https://www.alibabacloud.com/help/doc-detail/96788.htm).
     */
    @JvmName("jejpcdtpxpgsaclk")
    public suspend fun switchOver(`value`: Output<String>) {
        this.switchOver = value
    }

    /**
     * @param value The time at which you want to apply the specification changes. The time follows the ISO 8601 standard in the yyyy-MM-ddTHH:mm:ssZ format. The time must be in UTC.
     */
    @JvmName("jqgoolumolmplgty")
    public suspend fun switchTime(`value`: Output<String>) {
        this.switchTime = value
    }

    /**
     * @param value The time at which ApsaraDB RDS switches your workloads over to the new instance. This parameter is used together with the SwitchOver parameter and takes effect only when you set the SwitchOver parameter to true. Valid values:
     * * **Immediate**: After data is migrated to the new instance, ApsaraDB RDS immediately switches your workloads over to the new instance.
     * * **MaintainTime**: After data is migrated to the new instance, ApsaraDB RDS switches your workloads over to the new instance during the maintenance window that you specify. You can call the [ModifyDBInstanceMaintainTime](https://www.alibabacloud.com/help/doc-detail/26249.htm) operation to change the maintenance window of an instance.
     */
    @JvmName("exmvnlsadvevxwmd")
    public suspend fun switchTimeMode(`value`: Output<String>) {
        this.switchTimeMode = value
    }

    /**
     * @param value [The data replication mode](https://www.alibabacloud.com/help/doc-detail/96055.htm). Valid values:
     * * **Sync**: strong synchronization.
     * * **Semi-sync**: Semi-synchronous.
     * * **Async**: asynchronous.
     * > **NOTE:** SQL Server 2017 cluster version is currently not supported.
     */
    @JvmName("bchmhpiwskmnaxkn")
    public suspend fun syncMode(`value`: Output<String>) {
        this.syncMode = value
    }

    /**
     * @param value The major engine version of the new instance. The value of this parameter must be the major engine version on which an upgrade check is performed.
     * > **NOTE** You can call the [UpgradeDBInstanceMajorVersionPrecheck](https://www.alibabacloud.com/help/doc-detail/330050.htm) operation to perform an upgrade check on a major engine version.
     */
    @JvmName("sjppdgyfcxtjdejn")
    public suspend fun targetMajorVersion(`value`: Output<String>) {
        this.targetMajorVersion = value
    }

    /**
     * @param value The availability check method of the instance. Valid values:
     * - **SHORT**: Alibaba Cloud uses short-lived connections to check the availability of the instance.
     * - **LONG**: Alibaba Cloud uses persistent connections to check the availability of the instance.
     */
    @JvmName("eeofxedydmbnibne")
    public suspend fun tcpConnectionType(`value`: Output<String>) {
        this.tcpConnectionType = value
    }

    /**
     * @param value Specifies whether to enable TDE. Valid values: `Enabled` and `Disabled`.
     */
    @JvmName("ajokymrpmmbmthjs")
    public suspend fun tdeStatus(`value`: Output<String>) {
        this.tdeStatus = value
    }

    /**
     * @param value The ID of the VPC to which the new instance belongs.
     * > **NOTE:** Make sure that the VPC resides in the specified region.
     */
    @JvmName("ivticqpauwjrdbqa")
    public suspend fun vpcId(`value`: Output<String>) {
        this.vpcId = value
    }

    /**
     * @param value The ID of the vSwitch associated with the specified VPC.
     * > **NOTE:** Make sure that the vSwitch belongs to the specified VPC and region.
     */
    @JvmName("fhaxqtkjiburnljw")
    public suspend fun vswitchId(`value`: Output<String>) {
        this.vswitchId = value
    }

    /**
     * @param value The ID of the zone to which the new instance belongs. You can call the [DescribeRegions](https://www.alibabacloud.com/help/en/rds/developer-reference/api-rds-2014-08-15-describeregions) operation to query the most recent region list.
     * > **NOTE:** The default value of this parameter is the ID of the zone to which the original instance belongs.
     */
    @JvmName("gveifdhkcpubpreg")
    public suspend fun zoneId(`value`: Output<String>) {
        this.zoneId = value
    }

    /**
     * @param value The ID of the zone to which the secondary instance of the new instance belongs. You can specify this parameter only when the original instance runs RDS High-availability Edition. You can select a zone that belongs to the region where the original instance resides. You can call the [DescribeRegions](https://www.alibabacloud.com/help/en/rds/developer-reference/api-rds-2014-08-15-describeregions) operation to query zone IDs.
     */
    @JvmName("wvmxsmdwlmkyevgk")
    public suspend fun zoneIdSlave1(`value`: Output<String>) {
        this.zoneIdSlave1 = value
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version. This parameter indicates the authentication method. It is allowed only when the public key of the client certificate authority is enabled. Valid values: `cert` and `perfer` and `verify-ca` and `verify-full (supported by RDS PostgreSQL above 12)`.
     */
    @JvmName("mersieqtqyypgccb")
    public suspend fun acl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.acl = mapped
    }

    /**
     * @param value How to upgrade the minor version of the instance. Valid values:
     * * **Auto**: automatically upgrade the minor version.
     * * **Manual**: It is not automatically upgraded. It is only mandatory when the current version is offline.
     */
    @JvmName("vgjboxujprygjksm")
    public suspend fun autoUpgradeMinorVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoUpgradeMinorVersion = mapped
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the certificate type. When the value of ssl_action is Open, the default value of this parameter is aliyun. Value range:
     * * **aliyun**: using cloud certificates.
     * * **custom**: use a custom certificate. Valid values: `aliyun`, `custom`.
     */
    @JvmName("cuejjvgeqhblhoaf")
    public suspend fun caType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.caType = mapped
    }

    /**
     * @param value The file that contains the certificate used for TDE.
     */
    @JvmName("ueyvqaplosaxkupb")
    public suspend fun certificate(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.certificate = mapped
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the public key of the client certification authority. If the value of client_ca_enabled is 1, this parameter must be configured.
     */
    @JvmName("eqmptolrwmowbkhf")
    public suspend fun clientCaCert(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clientCaCert = mapped
    }

    /**
     * @param value The client ca enabled.
     */
    @JvmName("whrwehpxxbxcsxcp")
    public suspend fun clientCaEnabled(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clientCaEnabled = mapped
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version, which indicates that the client revokes the certificate file. If the value of client_crl_enabled is 1, this parameter must be configured.
     */
    @JvmName("ajdfbmhajghstdfb")
    public suspend fun clientCertRevocationList(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clientCertRevocationList = mapped
    }

    /**
     * @param value The client crl enabled.
     */
    @JvmName("kfuflqjamhwykwtc")
    public suspend fun clientCrlEnabled(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clientCrlEnabled = mapped
    }

    /**
     * @param value The time at which ApsaraDB RDS collects the statistics of the new instance.
     * * **Before**: ApsaraDB RDS collects the statistics of the new instance before the switchover to ensure service stability. If the original instance contains a large amount of data, the upgrade may require a long period of time.
     * * **After**: ApsaraDB RDS collects the statistics of the new instance after the switchover to accelerate the upgrade. If you access tables for which no statistics are generated, the query plans that you specify may be inaccurately executed. In addition, your database service may be unavailable during peak hours.
     * > **NOTE** If you set the SwitchOver parameter to false, the value Before specifies that ApsaraDB RDS collects the statistics of the new instance before the new instance starts to process read and write requests, and the value After specifies that ApsaraDB RDS collects the statistics of the new instance after the new instance starts to process read and write requests.
     */
    @JvmName("pdedmmnxrdcirsri")
    public suspend fun collectStatMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.collectStatMode = mapped
    }

    /**
     * @param value The connection string prefix.
     */
    @JvmName("hqyghnrnqhcarywu")
    public suspend fun connectionStringPrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.connectionStringPrefix = mapped
    }

    /**
     * @param value The instance type of the new instance. For information, see [Primary ApsaraDB RDS instance types](https://www.alibabacloud.com/help/en/rds/product-overview/primary-apsaradb-rds-instance-types).
     */
    @JvmName("xjuvcffyfsppkmve")
    public suspend fun dbInstanceClass(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbInstanceClass = mapped
    }

    /**
     * @param value The db instance description.
     */
    @JvmName("gvkgdxpvptlamaql")
    public suspend fun dbInstanceDescription(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbInstanceDescription = mapped
    }

    /**
     * @param value The storage capacity of the new instance. Unit: GB. The storage capacity increases in increments of 5 GB. For more information, see [Primary ApsaraDB RDS instance types](https://www.alibabacloud.com/help/en/rds/product-overview/primary-apsaradb-rds-instance-types).
     * > **NOTE:** The default value of this parameter is the storage capacity of the original instance.
     */
    @JvmName("orhmwmygdvibiqps")
    public suspend fun dbInstanceStorage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbInstanceStorage = mapped
    }

    /**
     * @param value The type of storage media that is used for the new instance. Valid values:
     * * **local_ssd**: local SSDs.
     * * **cloud_ssd**: standard SSDs.
     * * **cloud_essd**: enhanced SSDs (ESSDs) of performance level 1 (PL1).
     * * **cloud_essd2**: ESSDs of PL2.
     * * **cloud_essd3**: ESSDs of PL3.
     */
    @JvmName("xvypmmulrmayanmf")
    public suspend fun dbInstanceStorageType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbInstanceStorageType = mapped
    }

    /**
     * @param value The name of the database for which you want to enable TDE. Up to 50 names can be entered in a single request. If you specify multiple names, separate these names with commas (,).
     * > **NOTE:** This parameter is available and must be specified only when the instance runs SQL Server 2019 SE or an Enterprise Edition of SQL Server.
     */
    @JvmName("luqrliwcugaqebfo")
    public suspend fun dbName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbName = mapped
    }

    /**
     * @param value The ID of the dedicated cluster to which the new instance belongs. This parameter takes effect only when you create the new instance in a dedicated cluster.
     */
    @JvmName("xbejyolngsmttsoq")
    public suspend fun dedicatedHostGroupId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dedicatedHostGroupId = mapped
    }

    /**
     * @param value The switch of delete protection. Valid values:
     * - true: delete protect.
     * - false: no delete protect.
     * > **NOTE:** `deletion_protection` is valid only when attribute `payment_type` is set to `PayAsYouGo`, supported engine type: **MySQL**, **PostgreSQL**, **MariaDB**, **MSSQL**.
     */
    @JvmName("lfsifrjhetjhcuhy")
    public suspend fun deletionProtection(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deletionProtection = mapped
    }

    /**
     * @param value The direction. Valid values: `Auto`, `Down`, `TempUpgrade`, `Up`.
     */
    @JvmName("rvctvhxqdrgtrbni")
    public suspend fun direction(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.direction = mapped
    }

    /**
     * @param value The effective time.
     */
    @JvmName("xcvxgtwvwyivpjpy")
    public suspend fun effectiveTime(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.effectiveTime = mapped
    }

    /**
     * @param value The ID of the private key.
     * > **NOTE:** This parameter is available only when the instance runs MySQL.
     */
    @JvmName("gtyjsydtijyirwvf")
    public suspend fun encryptionKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.encryptionKey = mapped
    }

    /**
     * @param value Database type. Value options: MySQL, SQLServer, PostgreSQL.
     */
    @JvmName("hptclwllfbchgrox")
    public suspend fun engine(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value Database version. Value:
     * * MySQL: **5.5/5.6/5.7/8.0**.
     * * SQL Server: **2008r2/08r2_ent_ha/2012/2012_ent_ha/2012_std_ha/2012_web/2014_std_ha/2016_ent_ha/2016_std_ha/2016_web/2017_std_ha/2017_ent/2019_std_ha/2019_ent**.
     * * PostgreSQL: **9.4/10.0/11.0/12.0/13.0**.
     * * MariaDB: **10.3**.
     */
    @JvmName("lfnvbdmlruvqffqf")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value Set it to true to make some parameter efficient when modifying them. Default to false.
     */
    @JvmName("eomiovltlxcwfppr")
    public suspend fun forceRestart(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.forceRestart = mapped
    }

    /**
     * @param value The high availability mode. Valid values:
     * * **RPO**: Data persistence is preferred. The instance preferentially ensures data reliability to minimize data loss. Use this mode if you have higher requirements on data consistency.
     * * **RTO**: Instance availability is preferred. The instance restores services as soon as possible to ensure availability. Use this mode if you have higher requirements on service availability.
     */
    @JvmName("jtvkuihrdkqowyqo")
    public suspend fun haMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.haMode = mapped
    }

    /**
     * @param value The network type of the instance. Valid values:
     * * **Classic**: Classic Network.
     * * **VPC**: VPC.
     */
    @JvmName("tieswjolixkqerxg")
    public suspend fun instanceNetworkType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceNetworkType = mapped
    }

    /**
     * @param value The maintainable time period of the instance. Format: <I> HH:mm</I> Z-<I> HH:mm</I> Z(UTC time).
     */
    @JvmName("upljitlhgdbjnvyi")
    public suspend fun maintainTime(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maintainTime = mapped
    }

    /**
     * @param value Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("mjhdklutbhcbfsbj")
    public suspend fun parameters(`value`: List<RdsUpgradeDbInstanceParameterArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.parameters = mapped
    }

    /**
     * @param argument Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("lrgmyklsrauxluht")
    public suspend fun parameters(argument: List<suspend RdsUpgradeDbInstanceParameterArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            RdsUpgradeDbInstanceParameterArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.parameters = mapped
    }

    /**
     * @param argument Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("kwnyamjsfddkubyg")
    public suspend fun parameters(vararg argument: suspend RdsUpgradeDbInstanceParameterArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            RdsUpgradeDbInstanceParameterArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.parameters = mapped
    }

    /**
     * @param argument Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("kvfnxlvyjwqqcrmc")
    public suspend fun parameters(argument: suspend RdsUpgradeDbInstanceParameterArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            RdsUpgradeDbInstanceParameterArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.parameters = mapped
    }

    /**
     * @param values Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("ecaepcjyamotmagd")
    public suspend fun parameters(vararg values: RdsUpgradeDbInstanceParameterArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.parameters = mapped
    }

    /**
     * @param value The password of the certificate.
     * > **NOTE:** This parameter is available only when the instance runs SQL Server 2019 SE or an Enterprise Edition of SQL Server.
     */
    @JvmName("qvlwypafevqhylgk")
    public suspend fun password(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.password = mapped
    }

    /**
     * @param value The billing method of the new instance. Valid values: `PayAsYouGo` and `Subscription`.
     */
    @JvmName("klaunfxjapimfmcy")
    public suspend fun paymentType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.paymentType = mapped
    }

    /**
     * @param value The configuration of [AD domain](https://www.alibabacloud.com/help/en/doc-detail/349288.htm) . See `pg_hba_conf` below.
     */
    @JvmName("diedjycgecrvndfj")
    public suspend fun pgHbaConfs(`value`: List<RdsUpgradeDbInstancePgHbaConfArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.pgHbaConfs = mapped
    }

    /**
     * @param argument The configuration of [AD domain](https://www.alibabacloud.com/help/en/doc-detail/349288.htm) . See `pg_hba_conf` below.
     */
    @JvmName("hinkqnavuydcnwki")
    public suspend fun pgHbaConfs(argument: List<suspend RdsUpgradeDbInstancePgHbaConfArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            RdsUpgradeDbInstancePgHbaConfArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.pgHbaConfs = mapped
    }

    /**
     * @param argument The configuration of [AD domain](https://www.alibabacloud.com/help/en/doc-detail/349288.htm) . See `pg_hba_conf` below.
     */
    @JvmName("jivajihpdeutdxrm")
    public suspend fun pgHbaConfs(vararg argument: suspend RdsUpgradeDbInstancePgHbaConfArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            RdsUpgradeDbInstancePgHbaConfArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.pgHbaConfs = mapped
    }

    /**
     * @param argument The configuration of [AD domain](https://www.alibabacloud.com/help/en/doc-detail/349288.htm) . See `pg_hba_conf` below.
     */
    @JvmName("knepcbtqacascdsa")
    public suspend fun pgHbaConfs(argument: suspend RdsUpgradeDbInstancePgHbaConfArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            RdsUpgradeDbInstancePgHbaConfArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.pgHbaConfs = mapped
    }

    /**
     * @param values The configuration of [AD domain](https://www.alibabacloud.com/help/en/doc-detail/349288.htm) . See `pg_hba_conf` below.
     */
    @JvmName("mayaswgsjceaymdi")
    public suspend fun pgHbaConfs(vararg values: RdsUpgradeDbInstancePgHbaConfArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.pgHbaConfs = mapped
    }

    /**
     * @param value The port.
     */
    @JvmName("asdxhvtbhtvhikve")
    public suspend fun port(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.port = mapped
    }

    /**
     * @param value The intranet IP address of the new instance must be within the specified vSwitch IP address range. By default, the system automatically allocates by using **VPCId** and **VSwitchId**.
     */
    @JvmName("wptjaoocduiguvxb")
    public suspend fun privateIpAddress(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.privateIpAddress = mapped
    }

    /**
     * @param value The file that contains the private key used for TDE.
     */
    @JvmName("tqoamdkynwxacdrt")
    public suspend fun privateKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.privateKey = mapped
    }

    /**
     * @param value The released keep policy.
     */
    @JvmName("bjnvxtunmuiaebii")
    public suspend fun releasedKeepPolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.releasedKeepPolicy = mapped
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version, indicating the authentication method of the replication permission. It is only allowed when the public key of the client certificate authority is enabled. Valid values: `cert` and `perfer` and `verify-ca` and `verify-full (supported by RDS PostgreSQL above 12)`.
     */
    @JvmName("vrcjgkyiwblpcnal")
    public suspend fun replicationAcl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicationAcl = mapped
    }

    /**
     * @param value The resource group id.
     */
    @JvmName("sogxljafouecwfto")
    public suspend fun resourceGroupId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resourceGroupId = mapped
    }

    /**
     * @param value The Alibaba Cloud Resource Name (ARN) of a RAM role. A RAM role is a virtual RAM identity that you can create within your Alibaba Cloud account. For more information, see [RAM role overview](https://www.alibabacloud.com/help/en/ram/user-guide/ram-role-overview).
     * > **NOTE:** This parameter is available only when the instance runs MySQL.
     */
    @JvmName("nehsbthxmpxrdrlu")
    public suspend fun roleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.roleArn = mapped
    }

    /**
     * @param value The IP address whitelist of the instance. Separate multiple IP addresses with commas (,) and cannot be repeated. The following two formats are supported:
     * * IP address form, for example: 10.23.12.24.
     * * CIDR format, for example, 10.23.12.0/24 (no Inter-Domain Routing, 24 indicates the length of the prefix in the address, ranging from 1 to 32).
     * > **NOTE:** each instance can add up to 1000 IP addresses or IP segments, that is, the total number of IP addresses or IP segments in all IP whitelist groups cannot exceed 1000. When there are more IP addresses, it is recommended to merge them into IP segments, for example, 10.23.12.0/24.
     */
    @JvmName("jvkesakmygfmxsjc")
    public suspend fun securityIps(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityIps = mapped
    }

    /**
     * @param values The IP address whitelist of the instance. Separate multiple IP addresses with commas (,) and cannot be repeated. The following two formats are supported:
     * * IP address form, for example: 10.23.12.24.
     * * CIDR format, for example, 10.23.12.0/24 (no Inter-Domain Routing, 24 indicates the length of the prefix in the address, ranging from 1 to 32).
     * > **NOTE:** each instance can add up to 1000 IP addresses or IP segments, that is, the total number of IP addresses or IP segments in all IP whitelist groups cannot exceed 1000. When there are more IP addresses, it is recommended to merge them into IP segments, for example, 10.23.12.0/24.
     */
    @JvmName("aaickejqegmmjusk")
    public suspend fun securityIps(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.securityIps = mapped
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the content of the server certificate. If the CAType value is custom, this parameter must be configured.
     */
    @JvmName("ubuklsqdutiquwep")
    public suspend fun serverCert(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverCert = mapped
    }

    /**
     * @param value This parameter is only supported by the RDS PostgreSQL cloud disk version. It indicates the private key of the server certificate. If the value of CAType is custom, this parameter must be configured.
     */
    @JvmName("jipvltqsfywaevvm")
    public suspend fun serverKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverKey = mapped
    }

    /**
     * @param value The source biz.
     */
    @JvmName("rlackkffkceahuqh")
    public suspend fun sourceBiz(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceBiz = mapped
    }

    /**
     * @param value The source db instance id.
     */
    @JvmName("vkiqfbqqvymikuos")
    public suspend fun sourceDbInstanceId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceDbInstanceId = mapped
    }

    /**
     * @param value Enable or disable SSL. Valid values: `0` and `1`.
     */
    @JvmName("dcleguiwecadxbej")
    public suspend fun sslEnabled(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sslEnabled = mapped
    }

    /**
     * @param value Specifies whether ApsaraDB RDS automatically switches your workloads over to the new instance after data is migrated to the new instance. Valid values:
     * * **true**: ApsaraDB RDS automatically switches workloads over to the new instance. If you set this parameter to true, you must take note of the following information:
     * * After the switchover is complete, you cannot roll your workloads back to the original instance. Proceed with caution.
     * * During the switchover, the original instance processes only read requests. You must perform the switchover during off-peak hours.
     * * If read-only instances are attached to the original instance, you can set this parameter only to false. In this case, the read-only instances that are attached to the original instance cannot be cloned. After the upgrade is complete, you must create read-only instances for the new instance.
     * * **false**: ApsaraDB RDS does not automatically switch your workloads over to the new instance. Before you perform an upgrade, we recommend that you set this parameter to false to test whether the new major engine version is compatible with your workloads. If you set this parameter to false, you must take note of the following information:
     * * The data migration does not interrupt your workloads on the original instance.
     * * After data is migrated to the new instance, you must update the endpoint configuration on your application. This update requires you to replace the endpoint of the original instance with the endpoint of the new instance. For more information about how to view the endpoint of an instance, see [View and change the internal and public endpoints and port numbers of an ApsaraDB RDS for PostgreSQL instance](https://www.alibabacloud.com/help/doc-detail/96788.htm).
     */
    @JvmName("ulldssvniixjikuj")
    public suspend fun switchOver(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.switchOver = mapped
    }

    /**
     * @param value The time at which you want to apply the specification changes. The time follows the ISO 8601 standard in the yyyy-MM-ddTHH:mm:ssZ format. The time must be in UTC.
     */
    @JvmName("jnndpnoetqnvujwj")
    public suspend fun switchTime(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.switchTime = mapped
    }

    /**
     * @param value The time at which ApsaraDB RDS switches your workloads over to the new instance. This parameter is used together with the SwitchOver parameter and takes effect only when you set the SwitchOver parameter to true. Valid values:
     * * **Immediate**: After data is migrated to the new instance, ApsaraDB RDS immediately switches your workloads over to the new instance.
     * * **MaintainTime**: After data is migrated to the new instance, ApsaraDB RDS switches your workloads over to the new instance during the maintenance window that you specify. You can call the [ModifyDBInstanceMaintainTime](https://www.alibabacloud.com/help/doc-detail/26249.htm) operation to change the maintenance window of an instance.
     */
    @JvmName("rgnsljfvypsgiymu")
    public suspend fun switchTimeMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.switchTimeMode = mapped
    }

    /**
     * @param value [The data replication mode](https://www.alibabacloud.com/help/doc-detail/96055.htm). Valid values:
     * * **Sync**: strong synchronization.
     * * **Semi-sync**: Semi-synchronous.
     * * **Async**: asynchronous.
     * > **NOTE:** SQL Server 2017 cluster version is currently not supported.
     */
    @JvmName("gqfnkvaidwwvmihs")
    public suspend fun syncMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.syncMode = mapped
    }

    /**
     * @param value The major engine version of the new instance. The value of this parameter must be the major engine version on which an upgrade check is performed.
     * > **NOTE** You can call the [UpgradeDBInstanceMajorVersionPrecheck](https://www.alibabacloud.com/help/doc-detail/330050.htm) operation to perform an upgrade check on a major engine version.
     */
    @JvmName("uvctriqiaognmjis")
    public suspend fun targetMajorVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.targetMajorVersion = mapped
    }

    /**
     * @param value The availability check method of the instance. Valid values:
     * - **SHORT**: Alibaba Cloud uses short-lived connections to check the availability of the instance.
     * - **LONG**: Alibaba Cloud uses persistent connections to check the availability of the instance.
     */
    @JvmName("onqdyjhbpdgtgbps")
    public suspend fun tcpConnectionType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tcpConnectionType = mapped
    }

    /**
     * @param value Specifies whether to enable TDE. Valid values: `Enabled` and `Disabled`.
     */
    @JvmName("lcllwylmdbupbqkc")
    public suspend fun tdeStatus(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tdeStatus = mapped
    }

    /**
     * @param value The ID of the VPC to which the new instance belongs.
     * > **NOTE:** Make sure that the VPC resides in the specified region.
     */
    @JvmName("viwfvuughwcvbosk")
    public suspend fun vpcId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcId = mapped
    }

    /**
     * @param value The ID of the vSwitch associated with the specified VPC.
     * > **NOTE:** Make sure that the vSwitch belongs to the specified VPC and region.
     */
    @JvmName("dssnkcnarcceyidx")
    public suspend fun vswitchId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vswitchId = mapped
    }

    /**
     * @param value The ID of the zone to which the new instance belongs. You can call the [DescribeRegions](https://www.alibabacloud.com/help/en/rds/developer-reference/api-rds-2014-08-15-describeregions) operation to query the most recent region list.
     * > **NOTE:** The default value of this parameter is the ID of the zone to which the original instance belongs.
     */
    @JvmName("kcghiwvthhahalgh")
    public suspend fun zoneId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.zoneId = mapped
    }

    /**
     * @param value The ID of the zone to which the secondary instance of the new instance belongs. You can specify this parameter only when the original instance runs RDS High-availability Edition. You can select a zone that belongs to the region where the original instance resides. You can call the [DescribeRegions](https://www.alibabacloud.com/help/en/rds/developer-reference/api-rds-2014-08-15-describeregions) operation to query zone IDs.
     */
    @JvmName("saqrefjcvfwgmbhn")
    public suspend fun zoneIdSlave1(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.zoneIdSlave1 = mapped
    }

    internal fun build(): RdsUpgradeDbInstanceArgs = RdsUpgradeDbInstanceArgs(
        acl = acl,
        autoUpgradeMinorVersion = autoUpgradeMinorVersion,
        caType = caType,
        certificate = certificate,
        clientCaCert = clientCaCert,
        clientCaEnabled = clientCaEnabled,
        clientCertRevocationList = clientCertRevocationList,
        clientCrlEnabled = clientCrlEnabled,
        collectStatMode = collectStatMode,
        connectionStringPrefix = connectionStringPrefix,
        dbInstanceClass = dbInstanceClass,
        dbInstanceDescription = dbInstanceDescription,
        dbInstanceStorage = dbInstanceStorage,
        dbInstanceStorageType = dbInstanceStorageType,
        dbName = dbName,
        dedicatedHostGroupId = dedicatedHostGroupId,
        deletionProtection = deletionProtection,
        direction = direction,
        effectiveTime = effectiveTime,
        encryptionKey = encryptionKey,
        engine = engine,
        engineVersion = engineVersion,
        forceRestart = forceRestart,
        haMode = haMode,
        instanceNetworkType = instanceNetworkType,
        maintainTime = maintainTime,
        parameters = parameters,
        password = password,
        paymentType = paymentType,
        pgHbaConfs = pgHbaConfs,
        port = port,
        privateIpAddress = privateIpAddress,
        privateKey = privateKey,
        releasedKeepPolicy = releasedKeepPolicy,
        replicationAcl = replicationAcl,
        resourceGroupId = resourceGroupId,
        roleArn = roleArn,
        securityIps = securityIps,
        serverCert = serverCert,
        serverKey = serverKey,
        sourceBiz = sourceBiz,
        sourceDbInstanceId = sourceDbInstanceId,
        sslEnabled = sslEnabled,
        switchOver = switchOver,
        switchTime = switchTime,
        switchTimeMode = switchTimeMode,
        syncMode = syncMode,
        targetMajorVersion = targetMajorVersion,
        tcpConnectionType = tcpConnectionType,
        tdeStatus = tdeStatus,
        vpcId = vpcId,
        vswitchId = vswitchId,
        zoneId = zoneId,
        zoneIdSlave1 = zoneIdSlave1,
    )
}
