@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.alicloud.rds.kotlin

import com.pulumi.alicloud.rds.ReadOnlyInstanceArgs.builder
import com.pulumi.alicloud.rds.kotlin.inputs.ReadOnlyInstanceParameterArgs
import com.pulumi.alicloud.rds.kotlin.inputs.ReadOnlyInstanceParameterArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an RDS readonly instance resource, see [What is DB Readonly Instance](https://www.alibabacloud.com/help/en/apsaradb-for-rds/latest/api-rds-2014-08-15-createreadonlydbinstance).
 * > **NOTE:** Available since v1.52.1.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as alicloud from "@pulumi/alicloud";
 * const config = new pulumi.Config();
 * const name = config.get("name") || "tf-example";
 * const example = alicloud.rds.getZones({
 *     engine: "MySQL",
 *     engineVersion: "5.6",
 * });
 * const exampleNetwork = new alicloud.vpc.Network("example", {
 *     vpcName: name,
 *     cidrBlock: "172.16.0.0/16",
 * });
 * const exampleSwitch = new alicloud.vpc.Switch("example", {
 *     vpcId: exampleNetwork.id,
 *     cidrBlock: "172.16.0.0/24",
 *     zoneId: example.then(example => example.zones?.[0]?.id),
 *     vswitchName: name,
 * });
 * const exampleSecurityGroup = new alicloud.ecs.SecurityGroup("example", {
 *     name: name,
 *     vpcId: exampleNetwork.id,
 * });
 * const exampleInstance = new alicloud.rds.Instance("example", {
 *     engine: "MySQL",
 *     engineVersion: "5.6",
 *     instanceType: "rds.mysql.t1.small",
 *     instanceStorage: 20,
 *     instanceChargeType: "Postpaid",
 *     instanceName: name,
 *     vswitchId: exampleSwitch.id,
 *     securityIps: [
 *         "10.168.1.12",
 *         "100.69.7.112",
 *     ],
 * });
 * const exampleReadOnlyInstance = new alicloud.rds.ReadOnlyInstance("example", {
 *     zoneId: exampleInstance.zoneId,
 *     masterDbInstanceId: exampleInstance.id,
 *     engineVersion: exampleInstance.engineVersion,
 *     instanceStorage: exampleInstance.instanceStorage,
 *     instanceType: exampleInstance.instanceType,
 *     instanceName: `${name}readonly`,
 *     vswitchId: exampleSwitch.id,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_alicloud as alicloud
 * config = pulumi.Config()
 * name = config.get("name")
 * if name is None:
 *     name = "tf-example"
 * example = alicloud.rds.get_zones(engine="MySQL",
 *     engine_version="5.6")
 * example_network = alicloud.vpc.Network("example",
 *     vpc_name=name,
 *     cidr_block="172.16.0.0/16")
 * example_switch = alicloud.vpc.Switch("example",
 *     vpc_id=example_network.id,
 *     cidr_block="172.16.0.0/24",
 *     zone_id=example.zones[0].id,
 *     vswitch_name=name)
 * example_security_group = alicloud.ecs.SecurityGroup("example",
 *     name=name,
 *     vpc_id=example_network.id)
 * example_instance = alicloud.rds.Instance("example",
 *     engine="MySQL",
 *     engine_version="5.6",
 *     instance_type="rds.mysql.t1.small",
 *     instance_storage=20,
 *     instance_charge_type="Postpaid",
 *     instance_name=name,
 *     vswitch_id=example_switch.id,
 *     security_ips=[
 *         "10.168.1.12",
 *         "100.69.7.112",
 *     ])
 * example_read_only_instance = alicloud.rds.ReadOnlyInstance("example",
 *     zone_id=example_instance.zone_id,
 *     master_db_instance_id=example_instance.id,
 *     engine_version=example_instance.engine_version,
 *     instance_storage=example_instance.instance_storage,
 *     instance_type=example_instance.instance_type,
 *     instance_name=f"{name}readonly",
 *     vswitch_id=example_switch.id)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AliCloud = Pulumi.AliCloud;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var name = config.Get("name") ?? "tf-example";
 *     var example = AliCloud.Rds.GetZones.Invoke(new()
 *     {
 *         Engine = "MySQL",
 *         EngineVersion = "5.6",
 *     });
 *     var exampleNetwork = new AliCloud.Vpc.Network("example", new()
 *     {
 *         VpcName = name,
 *         CidrBlock = "172.16.0.0/16",
 *     });
 *     var exampleSwitch = new AliCloud.Vpc.Switch("example", new()
 *     {
 *         VpcId = exampleNetwork.Id,
 *         CidrBlock = "172.16.0.0/24",
 *         ZoneId = example.Apply(getZonesResult => getZonesResult.Zones[0]?.Id),
 *         VswitchName = name,
 *     });
 *     var exampleSecurityGroup = new AliCloud.Ecs.SecurityGroup("example", new()
 *     {
 *         Name = name,
 *         VpcId = exampleNetwork.Id,
 *     });
 *     var exampleInstance = new AliCloud.Rds.Instance("example", new()
 *     {
 *         Engine = "MySQL",
 *         EngineVersion = "5.6",
 *         InstanceType = "rds.mysql.t1.small",
 *         InstanceStorage = 20,
 *         InstanceChargeType = "Postpaid",
 *         InstanceName = name,
 *         VswitchId = exampleSwitch.Id,
 *         SecurityIps = new[]
 *         {
 *             "10.168.1.12",
 *             "100.69.7.112",
 *         },
 *     });
 *     var exampleReadOnlyInstance = new AliCloud.Rds.ReadOnlyInstance("example", new()
 *     {
 *         ZoneId = exampleInstance.ZoneId,
 *         MasterDbInstanceId = exampleInstance.Id,
 *         EngineVersion = exampleInstance.EngineVersion,
 *         InstanceStorage = exampleInstance.InstanceStorage,
 *         InstanceType = exampleInstance.InstanceType,
 *         InstanceName = $"{name}readonly",
 *         VswitchId = exampleSwitch.Id,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/ecs"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/rds"
 * 	"github.com/pulumi/pulumi-alicloud/sdk/v3/go/alicloud/vpc"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		cfg := config.New(ctx, "")
 * 		name := "tf-example"
 * 		if param := cfg.Get("name"); param != "" {
 * 			name = param
 * 		}
 * 		example, err := rds.GetZones(ctx, &rds.GetZonesArgs{
 * 			Engine:        pulumi.StringRef("MySQL"),
 * 			EngineVersion: pulumi.StringRef("5.6"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleNetwork, err := vpc.NewNetwork(ctx, "example", &vpc.NetworkArgs{
 * 			VpcName:   pulumi.String(name),
 * 			CidrBlock: pulumi.String("172.16.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleSwitch, err := vpc.NewSwitch(ctx, "example", &vpc.SwitchArgs{
 * 			VpcId:       exampleNetwork.ID(),
 * 			CidrBlock:   pulumi.String("172.16.0.0/24"),
 * 			ZoneId:      pulumi.String(example.Zones[0].Id),
 * 			VswitchName: pulumi.String(name),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ecs.NewSecurityGroup(ctx, "example", &ecs.SecurityGroupArgs{
 * 			Name:  pulumi.String(name),
 * 			VpcId: exampleNetwork.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleInstance, err := rds.NewInstance(ctx, "example", &rds.InstanceArgs{
 * 			Engine:             pulumi.String("MySQL"),
 * 			EngineVersion:      pulumi.String("5.6"),
 * 			InstanceType:       pulumi.String("rds.mysql.t1.small"),
 * 			InstanceStorage:    pulumi.Int(20),
 * 			InstanceChargeType: pulumi.String("Postpaid"),
 * 			InstanceName:       pulumi.String(name),
 * 			VswitchId:          exampleSwitch.ID(),
 * 			SecurityIps: pulumi.StringArray{
 * 				pulumi.String("10.168.1.12"),
 * 				pulumi.String("100.69.7.112"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewReadOnlyInstance(ctx, "example", &rds.ReadOnlyInstanceArgs{
 * 			ZoneId:             exampleInstance.ZoneId,
 * 			MasterDbInstanceId: exampleInstance.ID(),
 * 			EngineVersion:      exampleInstance.EngineVersion,
 * 			InstanceStorage:    exampleInstance.InstanceStorage,
 * 			InstanceType:       exampleInstance.InstanceType,
 * 			InstanceName:       pulumi.Sprintf("%vreadonly", name),
 * 			VswitchId:          exampleSwitch.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.alicloud.rds.RdsFunctions;
 * import com.pulumi.alicloud.rds.inputs.GetZonesArgs;
 * import com.pulumi.alicloud.vpc.Network;
 * import com.pulumi.alicloud.vpc.NetworkArgs;
 * import com.pulumi.alicloud.vpc.Switch;
 * import com.pulumi.alicloud.vpc.SwitchArgs;
 * import com.pulumi.alicloud.ecs.SecurityGroup;
 * import com.pulumi.alicloud.ecs.SecurityGroupArgs;
 * import com.pulumi.alicloud.rds.Instance;
 * import com.pulumi.alicloud.rds.InstanceArgs;
 * import com.pulumi.alicloud.rds.ReadOnlyInstance;
 * import com.pulumi.alicloud.rds.ReadOnlyInstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var name = config.get("name").orElse("tf-example");
 *         final var example = RdsFunctions.getZones(GetZonesArgs.builder()
 *             .engine("MySQL")
 *             .engineVersion("5.6")
 *             .build());
 *         var exampleNetwork = new Network("exampleNetwork", NetworkArgs.builder()
 *             .vpcName(name)
 *             .cidrBlock("172.16.0.0/16")
 *             .build());
 *         var exampleSwitch = new Switch("exampleSwitch", SwitchArgs.builder()
 *             .vpcId(exampleNetwork.id())
 *             .cidrBlock("172.16.0.0/24")
 *             .zoneId(example.applyValue(getZonesResult -> getZonesResult.zones()[0].id()))
 *             .vswitchName(name)
 *             .build());
 *         var exampleSecurityGroup = new SecurityGroup("exampleSecurityGroup", SecurityGroupArgs.builder()
 *             .name(name)
 *             .vpcId(exampleNetwork.id())
 *             .build());
 *         var exampleInstance = new Instance("exampleInstance", InstanceArgs.builder()
 *             .engine("MySQL")
 *             .engineVersion("5.6")
 *             .instanceType("rds.mysql.t1.small")
 *             .instanceStorage("20")
 *             .instanceChargeType("Postpaid")
 *             .instanceName(name)
 *             .vswitchId(exampleSwitch.id())
 *             .securityIps(
 *                 "10.168.1.12",
 *                 "100.69.7.112")
 *             .build());
 *         var exampleReadOnlyInstance = new ReadOnlyInstance("exampleReadOnlyInstance", ReadOnlyInstanceArgs.builder()
 *             .zoneId(exampleInstance.zoneId())
 *             .masterDbInstanceId(exampleInstance.id())
 *             .engineVersion(exampleInstance.engineVersion())
 *             .instanceStorage(exampleInstance.instanceStorage())
 *             .instanceType(exampleInstance.instanceType())
 *             .instanceName(String.format("%sreadonly", name))
 *             .vswitchId(exampleSwitch.id())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * configuration:
 *   name:
 *     type: string
 *     default: tf-example
 * resources:
 *   exampleNetwork:
 *     type: alicloud:vpc:Network
 *     name: example
 *     properties:
 *       vpcName: ${name}
 *       cidrBlock: 172.16.0.0/16
 *   exampleSwitch:
 *     type: alicloud:vpc:Switch
 *     name: example
 *     properties:
 *       vpcId: ${exampleNetwork.id}
 *       cidrBlock: 172.16.0.0/24
 *       zoneId: ${example.zones[0].id}
 *       vswitchName: ${name}
 *   exampleSecurityGroup:
 *     type: alicloud:ecs:SecurityGroup
 *     name: example
 *     properties:
 *       name: ${name}
 *       vpcId: ${exampleNetwork.id}
 *   exampleInstance:
 *     type: alicloud:rds:Instance
 *     name: example
 *     properties:
 *       engine: MySQL
 *       engineVersion: '5.6'
 *       instanceType: rds.mysql.t1.small
 *       instanceStorage: '20'
 *       instanceChargeType: Postpaid
 *       instanceName: ${name}
 *       vswitchId: ${exampleSwitch.id}
 *       securityIps:
 *         - 10.168.1.12
 *         - 100.69.7.112
 *   exampleReadOnlyInstance:
 *     type: alicloud:rds:ReadOnlyInstance
 *     name: example
 *     properties:
 *       zoneId: ${exampleInstance.zoneId}
 *       masterDbInstanceId: ${exampleInstance.id}
 *       engineVersion: ${exampleInstance.engineVersion}
 *       instanceStorage: ${exampleInstance.instanceStorage}
 *       instanceType: ${exampleInstance.instanceType}
 *       instanceName: ${name}readonly
 *       vswitchId: ${exampleSwitch.id}
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: alicloud:rds:getZones
 *       Arguments:
 *         engine: MySQL
 *         engineVersion: '5.6'
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * RDS readonly instance can be imported using the id, e.g.
 * ```sh
 * $ pulumi import alicloud:rds/readOnlyInstance:ReadOnlyInstance example rm-abc12345678
 * ```
 * @property acl The method that is used to verify the identities of clients. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. In addition, this parameter is available only when the public key of the CA that issues client certificates is enabled. It is valid only when `ssl_enabled  = 1`. Valid values:
 * - cert
 * - perfer
 * - verify-ca
 * - verify-full (supported only when the instance runs PostgreSQL 12 or later)
 * @property autoRenew Whether to renewal a DB instance automatically or not. It is valid when instance_charge_type is `PrePaid`. Default to `false`.
 * @property autoRenewPeriod Auto-renewal period of an instance, in the unit of the month. It is valid when instance_charge_type is `PrePaid`. Valid value:[1~12], Default to 1.
 * @property caType The type of the server certificate. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the SSLEnabled parameter to 1, the default value of this parameter is aliyun. It is valid only when `ssl_enabled  = 1`. Value range:
 * - aliyun: a cloud certificate
 * - custom: a custom certificate
 * @property clientCaCert The public key of the CA that issues client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the ClientCAEbabled parameter to 1, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
 * @property clientCaEnabled Specifies whether to enable the public key of the CA that issues client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. It is valid only when `ssl_enabled  = 1`. Valid values:
 * - 1: enables the public key
 * - 0: disables the public key
 * @property clientCertRevocationList The CRL that contains revoked client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the ClientCrlEnabled parameter to 1, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
 * @property clientCrlEnabled Specifies whether to enable a certificate revocation list (CRL) that contains revoked client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. In addition, this parameter is available only when the public key of the CA that issues client certificates is enabled. It is valid only when `ssl_enabled  = 1`. Valid values:
 * - 1: enables the CRL
 * - 0: disables the CRL
 * @property dbInstanceIpArrayAttribute The attribute of the IP address whitelist. By default, this parameter is empty.
 * > **NOTE:** The IP address whitelists that have the hidden attribute are not displayed in the ApsaraDB RDS console. These IP address whitelists are used to access Alibaba Cloud services, such as Data Transmission Service (DTS).
 * @property dbInstanceIpArrayName The name of the IP address whitelist. Default value: Default.
 * > **NOTE:** A maximum of 200 IP address whitelists can be configured for each instance.
 * @property dbInstanceStorageType The storage type of the instance. Valid values:
 * - local_ssd: specifies to use local SSDs. This value is recommended.
 * - cloud_ssd: specifies to use standard SSDs.
 * - cloud_essd: specifies to use enhanced SSDs (ESSDs).
 * - cloud_essd2: specifies to use enhanced SSDs (ESSDs).
 * - cloud_essd3: specifies to use enhanced SSDs (ESSDs).
 * @property deletionProtection The switch of delete protection. Valid values:
 * - true: delete protect.
 * - false: no delete protect.
 * @property direction The instance configuration type. Valid values:
 * - Up
 * - Down
 * - TempUpgrade
 * - Serverless
 * @property effectiveTime The method to change.  Default value: Immediate. Valid values:
 * - Immediate: The change immediately takes effect.
 * - MaintainTime: The change takes effect during the specified maintenance window. For more information, see ModifyDBInstanceMaintainTime.
 * @property engineVersion Database version. Value options can refer to the latest docs [CreateDBInstance](https://www.alibabacloud.com/help/doc-detail/26228.htm) `EngineVersion`.
 * @property forceRestart Set it to true to make some parameter efficient when modifying them. Default to false.
 * @property instanceChargeType Valid values are `Prepaid`, `Postpaid`, Default to `Postpaid`. The interval between the two conversion operations must be greater than 15 minutes. Only when this parameter is `Postpaid`, the instance can be released.
 * @property instanceName The name of DB instance. It a string of 2 to 256 characters.
 * @property instanceStorage User-defined DB instance storage space. Value range: [5, 2000] for MySQL/SQL Server HA dual node edition&#46; Increase progressively at a rate of 5 GB&#46; For details, see [Instance type table](https://www.alibabacloud.com/help/doc-detail/26312.htm).
 * @property instanceType DB Instance type. For details, see [Instance type table](https://www.alibabacloud.com/help/doc-detail/26312.htm).
 * @property masterDbInstanceId ID of the master instance.
 * @property modifyMode The method that is used to modify the IP address whitelist. Default value: Cover. Valid values:
 * - Cover: Use the value of the SecurityIps parameter to overwrite the existing entries in the IP address whitelist.
 * - Append: Add the IP addresses and CIDR blocks that are specified in the SecurityIps parameter to the IP address whitelist.
 * - Delete: Delete IP addresses and CIDR blocks that are specified in the SecurityIps parameter from the IP address whitelist. You must retain at least one IP address or CIDR block.
 * @property parameters Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
 * @property period The duration that you will buy DB instance (in month). It is valid when instance_charge_type is `PrePaid`. Valid values: [1~9], 12, 24, 36.
 * @property replicationAcl The method that is used to verify the replication permission. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. In addition, this parameter is available only when the public key of the CA that issues client certificates is enabled. It is valid only when `ssl_enabled  = 1`. Valid values:
 * - cert
 * - perfer
 * - verify-ca
 * - verify-full (supported only when the instance runs PostgreSQL 12 or later)
 * > **NOTE:** Because of data backup and migration, change DB instance type and storage would cost 15~20 minutes. Please make full preparation before changing them.
 * @property resourceGroupId The ID of resource group which the DB read-only instance belongs.
 * @property securityIpType The type of IP address in the IP address whitelist.
 * @property securityIps List of IP addresses allowed to access all databases of an instance. The list contains up to 1,000 IP addresses, separated by commas. Supported formats include 0.0.0.0/0, 10.23.12.24 (IP), and 10.23.12.24/24 (Classless Inter-Domain Routing (CIDR) mode. /24 represents the length of the prefix in an IP address. The range of the prefix length is [1,32]).
 * @property serverCert The content of the server certificate. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the CAType parameter to custom, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
 * @property serverKey The private key of the server certificate. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the CAType parameter to custom, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
 * @property sslEnabled Specifies whether to enable or disable SSL encryption. Valid values:
 * - 1: enables SSL encryption
 * - 0: disables SSL encryption
 * @property switchTime The specific point in time when you want to perform the update. Specify the time in the ISO 8601 standard in the yyyy-MM-ddTHH:mm:ssZ format. It is valid only when `upgrade_db_instance_kernel_version = true`. The time must be in UTC.
 * > **NOTE:** This parameter takes effect only when you set the UpgradeTime parameter to SpecifyTime.
 * @property tags A mapping of tags to assign to the resource.
 * - Key: It can be up to 64 characters in length. It cannot begin with "aliyun", "acs:", "http://", or "https://". It cannot be a null string.
 * - Value: It can be up to 128 characters in length. It cannot begin with "aliyun", "acs:", "http://", or "https://". It can be a null string.
 * @property targetMinorVersion The minor engine version to which you want to update the instance. If you do not specify this parameter, the instance is updated to the latest minor engine version. It is valid only when `upgrade_db_instance_kernel_version = true`. You must specify the minor engine version in one of the following formats:
 * - PostgreSQL: rds_postgres_<Major engine version>00_<Minor engine version>. Example: rds_postgres_1200_20200830.
 * - MySQL: <RDS edition>_<Minor engine version>. Examples: rds_20200229, xcluster_20200229, and xcluster80_20200229. The following RDS editions are supported:
 * - rds: The instance runs RDS Basic or High-availability Edition.
 * - xcluster: The instance runs MySQL 5.7 on RDS Enterprise Edition.
 * - xcluster80: The instance runs MySQL 8.0 on RDS Enterprise Edition.
 * - SQLServer: <Minor engine version>. Example: 15.0.4073.23.
 * > **NOTE:** For more information about minor engine versions, see Release notes of minor AliPG versions, Release notes of minor AliSQL versions, and Release notes of minor engine versions of ApsaraDB RDS for SQL Server.
 * @property upgradeDbInstanceKernelVersion Whether to upgrade a minor version of the kernel. Valid values:
 * - true: upgrade
 * - false: not to upgrade
 * @property upgradeTime The method to update the minor engine version. Default value: Immediate. It is valid only when `upgrade_db_instance_kernel_version = true`. Valid values:
 * - Immediate: The minor engine version is immediately updated.
 * - MaintainTime: The minor engine version is updated during the maintenance window. For more information about how to change the maintenance window, see ModifyDBInstanceMaintainTime.
 * - SpecifyTime: The minor engine version is updated at the point in time you specify.
 * @property vswitchId The virtual switch ID to launch DB instances in one VPC.
 * @property whitelistNetworkType The network type of the IP address whitelist. Default value: MIX. Valid values:
 * - Classic: classic network in enhanced whitelist mode
 * - VPC: virtual private cloud (VPC) in enhanced whitelist mode
 * - MIX: standard whitelist mode
 * > **NOTE:** In standard whitelist mode, IP addresses and CIDR blocks can be added only to the default IP address whitelist. In enhanced whitelist mode, IP addresses and CIDR blocks can be added to both IP address whitelists of the classic network type and those of the VPC network type.
 * @property zoneId The Zone to launch the DB instance.
 */
public data class ReadOnlyInstanceArgs(
    public val acl: Output<String>? = null,
    public val autoRenew: Output<Boolean>? = null,
    public val autoRenewPeriod: Output<Int>? = null,
    public val caType: Output<String>? = null,
    public val clientCaCert: Output<String>? = null,
    public val clientCaEnabled: Output<Int>? = null,
    public val clientCertRevocationList: Output<String>? = null,
    public val clientCrlEnabled: Output<Int>? = null,
    public val dbInstanceIpArrayAttribute: Output<String>? = null,
    public val dbInstanceIpArrayName: Output<String>? = null,
    public val dbInstanceStorageType: Output<String>? = null,
    public val deletionProtection: Output<Boolean>? = null,
    public val direction: Output<String>? = null,
    public val effectiveTime: Output<String>? = null,
    public val engineVersion: Output<String>? = null,
    public val forceRestart: Output<Boolean>? = null,
    public val instanceChargeType: Output<String>? = null,
    public val instanceName: Output<String>? = null,
    public val instanceStorage: Output<Int>? = null,
    public val instanceType: Output<String>? = null,
    public val masterDbInstanceId: Output<String>? = null,
    public val modifyMode: Output<String>? = null,
    public val parameters: Output<List<ReadOnlyInstanceParameterArgs>>? = null,
    public val period: Output<Int>? = null,
    public val replicationAcl: Output<String>? = null,
    public val resourceGroupId: Output<String>? = null,
    public val securityIpType: Output<String>? = null,
    public val securityIps: Output<List<String>>? = null,
    public val serverCert: Output<String>? = null,
    public val serverKey: Output<String>? = null,
    public val sslEnabled: Output<Int>? = null,
    public val switchTime: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val targetMinorVersion: Output<String>? = null,
    public val upgradeDbInstanceKernelVersion: Output<Boolean>? = null,
    public val upgradeTime: Output<String>? = null,
    public val vswitchId: Output<String>? = null,
    public val whitelistNetworkType: Output<String>? = null,
    public val zoneId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.alicloud.rds.ReadOnlyInstanceArgs> {
    override fun toJava(): com.pulumi.alicloud.rds.ReadOnlyInstanceArgs =
        com.pulumi.alicloud.rds.ReadOnlyInstanceArgs.builder()
            .acl(acl?.applyValue({ args0 -> args0 }))
            .autoRenew(autoRenew?.applyValue({ args0 -> args0 }))
            .autoRenewPeriod(autoRenewPeriod?.applyValue({ args0 -> args0 }))
            .caType(caType?.applyValue({ args0 -> args0 }))
            .clientCaCert(clientCaCert?.applyValue({ args0 -> args0 }))
            .clientCaEnabled(clientCaEnabled?.applyValue({ args0 -> args0 }))
            .clientCertRevocationList(clientCertRevocationList?.applyValue({ args0 -> args0 }))
            .clientCrlEnabled(clientCrlEnabled?.applyValue({ args0 -> args0 }))
            .dbInstanceIpArrayAttribute(dbInstanceIpArrayAttribute?.applyValue({ args0 -> args0 }))
            .dbInstanceIpArrayName(dbInstanceIpArrayName?.applyValue({ args0 -> args0 }))
            .dbInstanceStorageType(dbInstanceStorageType?.applyValue({ args0 -> args0 }))
            .deletionProtection(deletionProtection?.applyValue({ args0 -> args0 }))
            .direction(direction?.applyValue({ args0 -> args0 }))
            .effectiveTime(effectiveTime?.applyValue({ args0 -> args0 }))
            .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
            .forceRestart(forceRestart?.applyValue({ args0 -> args0 }))
            .instanceChargeType(instanceChargeType?.applyValue({ args0 -> args0 }))
            .instanceName(instanceName?.applyValue({ args0 -> args0 }))
            .instanceStorage(instanceStorage?.applyValue({ args0 -> args0 }))
            .instanceType(instanceType?.applyValue({ args0 -> args0 }))
            .masterDbInstanceId(masterDbInstanceId?.applyValue({ args0 -> args0 }))
            .modifyMode(modifyMode?.applyValue({ args0 -> args0 }))
            .parameters(
                parameters?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .period(period?.applyValue({ args0 -> args0 }))
            .replicationAcl(replicationAcl?.applyValue({ args0 -> args0 }))
            .resourceGroupId(resourceGroupId?.applyValue({ args0 -> args0 }))
            .securityIpType(securityIpType?.applyValue({ args0 -> args0 }))
            .securityIps(securityIps?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .serverCert(serverCert?.applyValue({ args0 -> args0 }))
            .serverKey(serverKey?.applyValue({ args0 -> args0 }))
            .sslEnabled(sslEnabled?.applyValue({ args0 -> args0 }))
            .switchTime(switchTime?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .targetMinorVersion(targetMinorVersion?.applyValue({ args0 -> args0 }))
            .upgradeDbInstanceKernelVersion(upgradeDbInstanceKernelVersion?.applyValue({ args0 -> args0 }))
            .upgradeTime(upgradeTime?.applyValue({ args0 -> args0 }))
            .vswitchId(vswitchId?.applyValue({ args0 -> args0 }))
            .whitelistNetworkType(whitelistNetworkType?.applyValue({ args0 -> args0 }))
            .zoneId(zoneId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [ReadOnlyInstanceArgs].
 */
@PulumiTagMarker
public class ReadOnlyInstanceArgsBuilder internal constructor() {
    private var acl: Output<String>? = null

    private var autoRenew: Output<Boolean>? = null

    private var autoRenewPeriod: Output<Int>? = null

    private var caType: Output<String>? = null

    private var clientCaCert: Output<String>? = null

    private var clientCaEnabled: Output<Int>? = null

    private var clientCertRevocationList: Output<String>? = null

    private var clientCrlEnabled: Output<Int>? = null

    private var dbInstanceIpArrayAttribute: Output<String>? = null

    private var dbInstanceIpArrayName: Output<String>? = null

    private var dbInstanceStorageType: Output<String>? = null

    private var deletionProtection: Output<Boolean>? = null

    private var direction: Output<String>? = null

    private var effectiveTime: Output<String>? = null

    private var engineVersion: Output<String>? = null

    private var forceRestart: Output<Boolean>? = null

    private var instanceChargeType: Output<String>? = null

    private var instanceName: Output<String>? = null

    private var instanceStorage: Output<Int>? = null

    private var instanceType: Output<String>? = null

    private var masterDbInstanceId: Output<String>? = null

    private var modifyMode: Output<String>? = null

    private var parameters: Output<List<ReadOnlyInstanceParameterArgs>>? = null

    private var period: Output<Int>? = null

    private var replicationAcl: Output<String>? = null

    private var resourceGroupId: Output<String>? = null

    private var securityIpType: Output<String>? = null

    private var securityIps: Output<List<String>>? = null

    private var serverCert: Output<String>? = null

    private var serverKey: Output<String>? = null

    private var sslEnabled: Output<Int>? = null

    private var switchTime: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var targetMinorVersion: Output<String>? = null

    private var upgradeDbInstanceKernelVersion: Output<Boolean>? = null

    private var upgradeTime: Output<String>? = null

    private var vswitchId: Output<String>? = null

    private var whitelistNetworkType: Output<String>? = null

    private var zoneId: Output<String>? = null

    /**
     * @param value The method that is used to verify the identities of clients. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. In addition, this parameter is available only when the public key of the CA that issues client certificates is enabled. It is valid only when `ssl_enabled  = 1`. Valid values:
     * - cert
     * - perfer
     * - verify-ca
     * - verify-full (supported only when the instance runs PostgreSQL 12 or later)
     */
    @JvmName("otfjoxkactjlnnru")
    public suspend fun acl(`value`: Output<String>) {
        this.acl = value
    }

    /**
     * @param value Whether to renewal a DB instance automatically or not. It is valid when instance_charge_type is `PrePaid`. Default to `false`.
     */
    @JvmName("brwfahedwogkwuld")
    public suspend fun autoRenew(`value`: Output<Boolean>) {
        this.autoRenew = value
    }

    /**
     * @param value Auto-renewal period of an instance, in the unit of the month. It is valid when instance_charge_type is `PrePaid`. Valid value:[1~12], Default to 1.
     */
    @JvmName("cdayrlkohobuwqgo")
    public suspend fun autoRenewPeriod(`value`: Output<Int>) {
        this.autoRenewPeriod = value
    }

    /**
     * @param value The type of the server certificate. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the SSLEnabled parameter to 1, the default value of this parameter is aliyun. It is valid only when `ssl_enabled  = 1`. Value range:
     * - aliyun: a cloud certificate
     * - custom: a custom certificate
     */
    @JvmName("wqderjyleqmpgppe")
    public suspend fun caType(`value`: Output<String>) {
        this.caType = value
    }

    /**
     * @param value The public key of the CA that issues client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the ClientCAEbabled parameter to 1, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
     */
    @JvmName("tnxlkycmnpuicqqy")
    public suspend fun clientCaCert(`value`: Output<String>) {
        this.clientCaCert = value
    }

    /**
     * @param value Specifies whether to enable the public key of the CA that issues client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. It is valid only when `ssl_enabled  = 1`. Valid values:
     * - 1: enables the public key
     * - 0: disables the public key
     */
    @JvmName("dgbebahvmobispyh")
    public suspend fun clientCaEnabled(`value`: Output<Int>) {
        this.clientCaEnabled = value
    }

    /**
     * @param value The CRL that contains revoked client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the ClientCrlEnabled parameter to 1, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
     */
    @JvmName("htkbekinxcqqccvn")
    public suspend fun clientCertRevocationList(`value`: Output<String>) {
        this.clientCertRevocationList = value
    }

    /**
     * @param value Specifies whether to enable a certificate revocation list (CRL) that contains revoked client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. In addition, this parameter is available only when the public key of the CA that issues client certificates is enabled. It is valid only when `ssl_enabled  = 1`. Valid values:
     * - 1: enables the CRL
     * - 0: disables the CRL
     */
    @JvmName("dqccynvctjbdcfab")
    public suspend fun clientCrlEnabled(`value`: Output<Int>) {
        this.clientCrlEnabled = value
    }

    /**
     * @param value The attribute of the IP address whitelist. By default, this parameter is empty.
     * > **NOTE:** The IP address whitelists that have the hidden attribute are not displayed in the ApsaraDB RDS console. These IP address whitelists are used to access Alibaba Cloud services, such as Data Transmission Service (DTS).
     */
    @JvmName("fhiblinhydfnjbfk")
    public suspend fun dbInstanceIpArrayAttribute(`value`: Output<String>) {
        this.dbInstanceIpArrayAttribute = value
    }

    /**
     * @param value The name of the IP address whitelist. Default value: Default.
     * > **NOTE:** A maximum of 200 IP address whitelists can be configured for each instance.
     */
    @JvmName("bwoqvcuprkuxjoow")
    public suspend fun dbInstanceIpArrayName(`value`: Output<String>) {
        this.dbInstanceIpArrayName = value
    }

    /**
     * @param value The storage type of the instance. Valid values:
     * - local_ssd: specifies to use local SSDs. This value is recommended.
     * - cloud_ssd: specifies to use standard SSDs.
     * - cloud_essd: specifies to use enhanced SSDs (ESSDs).
     * - cloud_essd2: specifies to use enhanced SSDs (ESSDs).
     * - cloud_essd3: specifies to use enhanced SSDs (ESSDs).
     */
    @JvmName("scmrooijlxhecwrn")
    public suspend fun dbInstanceStorageType(`value`: Output<String>) {
        this.dbInstanceStorageType = value
    }

    /**
     * @param value The switch of delete protection. Valid values:
     * - true: delete protect.
     * - false: no delete protect.
     */
    @JvmName("yxqsoajihxmeuqkt")
    public suspend fun deletionProtection(`value`: Output<Boolean>) {
        this.deletionProtection = value
    }

    /**
     * @param value The instance configuration type. Valid values:
     * - Up
     * - Down
     * - TempUpgrade
     * - Serverless
     */
    @JvmName("iaikintrukvnruav")
    public suspend fun direction(`value`: Output<String>) {
        this.direction = value
    }

    /**
     * @param value The method to change.  Default value: Immediate. Valid values:
     * - Immediate: The change immediately takes effect.
     * - MaintainTime: The change takes effect during the specified maintenance window. For more information, see ModifyDBInstanceMaintainTime.
     */
    @JvmName("ecvfcyyuxsbsonsh")
    public suspend fun effectiveTime(`value`: Output<String>) {
        this.effectiveTime = value
    }

    /**
     * @param value Database version. Value options can refer to the latest docs [CreateDBInstance](https://www.alibabacloud.com/help/doc-detail/26228.htm) `EngineVersion`.
     */
    @JvmName("wafloblkuvrhgflg")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value Set it to true to make some parameter efficient when modifying them. Default to false.
     */
    @JvmName("yipveuavaxjweces")
    public suspend fun forceRestart(`value`: Output<Boolean>) {
        this.forceRestart = value
    }

    /**
     * @param value Valid values are `Prepaid`, `Postpaid`, Default to `Postpaid`. The interval between the two conversion operations must be greater than 15 minutes. Only when this parameter is `Postpaid`, the instance can be released.
     */
    @JvmName("fygyyeubqpblkxya")
    public suspend fun instanceChargeType(`value`: Output<String>) {
        this.instanceChargeType = value
    }

    /**
     * @param value The name of DB instance. It a string of 2 to 256 characters.
     */
    @JvmName("kkbtlprxlruxoiwh")
    public suspend fun instanceName(`value`: Output<String>) {
        this.instanceName = value
    }

    /**
     * @param value User-defined DB instance storage space. Value range: [5, 2000] for MySQL/SQL Server HA dual node edition&#46; Increase progressively at a rate of 5 GB&#46; For details, see [Instance type table](https://www.alibabacloud.com/help/doc-detail/26312.htm).
     */
    @JvmName("wlgeserjnfajjqxi")
    public suspend fun instanceStorage(`value`: Output<Int>) {
        this.instanceStorage = value
    }

    /**
     * @param value DB Instance type. For details, see [Instance type table](https://www.alibabacloud.com/help/doc-detail/26312.htm).
     */
    @JvmName("asbtgjlotkvrktgr")
    public suspend fun instanceType(`value`: Output<String>) {
        this.instanceType = value
    }

    /**
     * @param value ID of the master instance.
     */
    @JvmName("sygxuriubxybnoby")
    public suspend fun masterDbInstanceId(`value`: Output<String>) {
        this.masterDbInstanceId = value
    }

    /**
     * @param value The method that is used to modify the IP address whitelist. Default value: Cover. Valid values:
     * - Cover: Use the value of the SecurityIps parameter to overwrite the existing entries in the IP address whitelist.
     * - Append: Add the IP addresses and CIDR blocks that are specified in the SecurityIps parameter to the IP address whitelist.
     * - Delete: Delete IP addresses and CIDR blocks that are specified in the SecurityIps parameter from the IP address whitelist. You must retain at least one IP address or CIDR block.
     */
    @JvmName("yvxuqjqocqaesskr")
    public suspend fun modifyMode(`value`: Output<String>) {
        this.modifyMode = value
    }

    /**
     * @param value Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("yjekkhrvstjieuge")
    public suspend fun parameters(`value`: Output<List<ReadOnlyInstanceParameterArgs>>) {
        this.parameters = value
    }

    @JvmName("bxkapifketdmrjeb")
    public suspend fun parameters(vararg values: Output<ReadOnlyInstanceParameterArgs>) {
        this.parameters = Output.all(values.asList())
    }

    /**
     * @param values Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("hltqfygubqpcpwgd")
    public suspend fun parameters(values: List<Output<ReadOnlyInstanceParameterArgs>>) {
        this.parameters = Output.all(values)
    }

    /**
     * @param value The duration that you will buy DB instance (in month). It is valid when instance_charge_type is `PrePaid`. Valid values: [1~9], 12, 24, 36.
     */
    @JvmName("ooibledfwgsoqbfd")
    public suspend fun period(`value`: Output<Int>) {
        this.period = value
    }

    /**
     * @param value The method that is used to verify the replication permission. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. In addition, this parameter is available only when the public key of the CA that issues client certificates is enabled. It is valid only when `ssl_enabled  = 1`. Valid values:
     * - cert
     * - perfer
     * - verify-ca
     * - verify-full (supported only when the instance runs PostgreSQL 12 or later)
     * > **NOTE:** Because of data backup and migration, change DB instance type and storage would cost 15~20 minutes. Please make full preparation before changing them.
     */
    @JvmName("lxqylxaxwdtrlsja")
    public suspend fun replicationAcl(`value`: Output<String>) {
        this.replicationAcl = value
    }

    /**
     * @param value The ID of resource group which the DB read-only instance belongs.
     */
    @JvmName("gmeaelirbusgbxui")
    public suspend fun resourceGroupId(`value`: Output<String>) {
        this.resourceGroupId = value
    }

    /**
     * @param value The type of IP address in the IP address whitelist.
     */
    @JvmName("ypyjyyeqgjnubllk")
    public suspend fun securityIpType(`value`: Output<String>) {
        this.securityIpType = value
    }

    /**
     * @param value List of IP addresses allowed to access all databases of an instance. The list contains up to 1,000 IP addresses, separated by commas. Supported formats include 0.0.0.0/0, 10.23.12.24 (IP), and 10.23.12.24/24 (Classless Inter-Domain Routing (CIDR) mode. /24 represents the length of the prefix in an IP address. The range of the prefix length is [1,32]).
     */
    @JvmName("xhgpjpjghaalihkp")
    public suspend fun securityIps(`value`: Output<List<String>>) {
        this.securityIps = value
    }

    @JvmName("tfkpfimrrasbufhi")
    public suspend fun securityIps(vararg values: Output<String>) {
        this.securityIps = Output.all(values.asList())
    }

    /**
     * @param values List of IP addresses allowed to access all databases of an instance. The list contains up to 1,000 IP addresses, separated by commas. Supported formats include 0.0.0.0/0, 10.23.12.24 (IP), and 10.23.12.24/24 (Classless Inter-Domain Routing (CIDR) mode. /24 represents the length of the prefix in an IP address. The range of the prefix length is [1,32]).
     */
    @JvmName("vtuwjcyfbbwmheed")
    public suspend fun securityIps(values: List<Output<String>>) {
        this.securityIps = Output.all(values)
    }

    /**
     * @param value The content of the server certificate. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the CAType parameter to custom, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
     */
    @JvmName("dvbnodqerafftptb")
    public suspend fun serverCert(`value`: Output<String>) {
        this.serverCert = value
    }

    /**
     * @param value The private key of the server certificate. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the CAType parameter to custom, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
     */
    @JvmName("wcywlolciaaixxkm")
    public suspend fun serverKey(`value`: Output<String>) {
        this.serverKey = value
    }

    /**
     * @param value Specifies whether to enable or disable SSL encryption. Valid values:
     * - 1: enables SSL encryption
     * - 0: disables SSL encryption
     */
    @JvmName("sgejblgmmfowjlhj")
    public suspend fun sslEnabled(`value`: Output<Int>) {
        this.sslEnabled = value
    }

    /**
     * @param value The specific point in time when you want to perform the update. Specify the time in the ISO 8601 standard in the yyyy-MM-ddTHH:mm:ssZ format. It is valid only when `upgrade_db_instance_kernel_version = true`. The time must be in UTC.
     * > **NOTE:** This parameter takes effect only when you set the UpgradeTime parameter to SpecifyTime.
     */
    @JvmName("mgvwqqyfpdhrmrei")
    public suspend fun switchTime(`value`: Output<String>) {
        this.switchTime = value
    }

    /**
     * @param value A mapping of tags to assign to the resource.
     * - Key: It can be up to 64 characters in length. It cannot begin with "aliyun", "acs:", "http://", or "https://". It cannot be a null string.
     * - Value: It can be up to 128 characters in length. It cannot begin with "aliyun", "acs:", "http://", or "https://". It can be a null string.
     */
    @JvmName("ygsehwbpolqflrdx")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The minor engine version to which you want to update the instance. If you do not specify this parameter, the instance is updated to the latest minor engine version. It is valid only when `upgrade_db_instance_kernel_version = true`. You must specify the minor engine version in one of the following formats:
     * - PostgreSQL: rds_postgres_<Major engine version>00_<Minor engine version>. Example: rds_postgres_1200_20200830.
     * - MySQL: <RDS edition>_<Minor engine version>. Examples: rds_20200229, xcluster_20200229, and xcluster80_20200229. The following RDS editions are supported:
     * - rds: The instance runs RDS Basic or High-availability Edition.
     * - xcluster: The instance runs MySQL 5.7 on RDS Enterprise Edition.
     * - xcluster80: The instance runs MySQL 8.0 on RDS Enterprise Edition.
     * - SQLServer: <Minor engine version>. Example: 15.0.4073.23.
     * > **NOTE:** For more information about minor engine versions, see Release notes of minor AliPG versions, Release notes of minor AliSQL versions, and Release notes of minor engine versions of ApsaraDB RDS for SQL Server.
     */
    @JvmName("mnsdabjhmrtlyaqm")
    public suspend fun targetMinorVersion(`value`: Output<String>) {
        this.targetMinorVersion = value
    }

    /**
     * @param value Whether to upgrade a minor version of the kernel. Valid values:
     * - true: upgrade
     * - false: not to upgrade
     */
    @JvmName("oilpfmcvtpqbbrui")
    public suspend fun upgradeDbInstanceKernelVersion(`value`: Output<Boolean>) {
        this.upgradeDbInstanceKernelVersion = value
    }

    /**
     * @param value The method to update the minor engine version. Default value: Immediate. It is valid only when `upgrade_db_instance_kernel_version = true`. Valid values:
     * - Immediate: The minor engine version is immediately updated.
     * - MaintainTime: The minor engine version is updated during the maintenance window. For more information about how to change the maintenance window, see ModifyDBInstanceMaintainTime.
     * - SpecifyTime: The minor engine version is updated at the point in time you specify.
     */
    @JvmName("evrjqcimsjdvwmoc")
    public suspend fun upgradeTime(`value`: Output<String>) {
        this.upgradeTime = value
    }

    /**
     * @param value The virtual switch ID to launch DB instances in one VPC.
     */
    @JvmName("yhfgaasbpogtyxhp")
    public suspend fun vswitchId(`value`: Output<String>) {
        this.vswitchId = value
    }

    /**
     * @param value The network type of the IP address whitelist. Default value: MIX. Valid values:
     * - Classic: classic network in enhanced whitelist mode
     * - VPC: virtual private cloud (VPC) in enhanced whitelist mode
     * - MIX: standard whitelist mode
     * > **NOTE:** In standard whitelist mode, IP addresses and CIDR blocks can be added only to the default IP address whitelist. In enhanced whitelist mode, IP addresses and CIDR blocks can be added to both IP address whitelists of the classic network type and those of the VPC network type.
     */
    @JvmName("ahlofbiylxidfjvd")
    public suspend fun whitelistNetworkType(`value`: Output<String>) {
        this.whitelistNetworkType = value
    }

    /**
     * @param value The Zone to launch the DB instance.
     */
    @JvmName("pjwgfxjbifvffldx")
    public suspend fun zoneId(`value`: Output<String>) {
        this.zoneId = value
    }

    /**
     * @param value The method that is used to verify the identities of clients. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. In addition, this parameter is available only when the public key of the CA that issues client certificates is enabled. It is valid only when `ssl_enabled  = 1`. Valid values:
     * - cert
     * - perfer
     * - verify-ca
     * - verify-full (supported only when the instance runs PostgreSQL 12 or later)
     */
    @JvmName("hncegdcotslknthe")
    public suspend fun acl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.acl = mapped
    }

    /**
     * @param value Whether to renewal a DB instance automatically or not. It is valid when instance_charge_type is `PrePaid`. Default to `false`.
     */
    @JvmName("kqexuyxttrvhkivs")
    public suspend fun autoRenew(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoRenew = mapped
    }

    /**
     * @param value Auto-renewal period of an instance, in the unit of the month. It is valid when instance_charge_type is `PrePaid`. Valid value:[1~12], Default to 1.
     */
    @JvmName("cnmegsoeladjsfde")
    public suspend fun autoRenewPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoRenewPeriod = mapped
    }

    /**
     * @param value The type of the server certificate. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the SSLEnabled parameter to 1, the default value of this parameter is aliyun. It is valid only when `ssl_enabled  = 1`. Value range:
     * - aliyun: a cloud certificate
     * - custom: a custom certificate
     */
    @JvmName("hlkheyxhexokvvhd")
    public suspend fun caType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.caType = mapped
    }

    /**
     * @param value The public key of the CA that issues client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the ClientCAEbabled parameter to 1, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
     */
    @JvmName("alrrqgdjtlttsgwe")
    public suspend fun clientCaCert(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clientCaCert = mapped
    }

    /**
     * @param value Specifies whether to enable the public key of the CA that issues client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. It is valid only when `ssl_enabled  = 1`. Valid values:
     * - 1: enables the public key
     * - 0: disables the public key
     */
    @JvmName("klmktbprosyrrwcq")
    public suspend fun clientCaEnabled(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clientCaEnabled = mapped
    }

    /**
     * @param value The CRL that contains revoked client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the ClientCrlEnabled parameter to 1, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
     */
    @JvmName("pivibxseehtnplwu")
    public suspend fun clientCertRevocationList(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clientCertRevocationList = mapped
    }

    /**
     * @param value Specifies whether to enable a certificate revocation list (CRL) that contains revoked client certificates. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. In addition, this parameter is available only when the public key of the CA that issues client certificates is enabled. It is valid only when `ssl_enabled  = 1`. Valid values:
     * - 1: enables the CRL
     * - 0: disables the CRL
     */
    @JvmName("weuwdyvpstarfoto")
    public suspend fun clientCrlEnabled(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clientCrlEnabled = mapped
    }

    /**
     * @param value The attribute of the IP address whitelist. By default, this parameter is empty.
     * > **NOTE:** The IP address whitelists that have the hidden attribute are not displayed in the ApsaraDB RDS console. These IP address whitelists are used to access Alibaba Cloud services, such as Data Transmission Service (DTS).
     */
    @JvmName("fjhfkcobgnoptvon")
    public suspend fun dbInstanceIpArrayAttribute(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbInstanceIpArrayAttribute = mapped
    }

    /**
     * @param value The name of the IP address whitelist. Default value: Default.
     * > **NOTE:** A maximum of 200 IP address whitelists can be configured for each instance.
     */
    @JvmName("bspovqfgusunlhww")
    public suspend fun dbInstanceIpArrayName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbInstanceIpArrayName = mapped
    }

    /**
     * @param value The storage type of the instance. Valid values:
     * - local_ssd: specifies to use local SSDs. This value is recommended.
     * - cloud_ssd: specifies to use standard SSDs.
     * - cloud_essd: specifies to use enhanced SSDs (ESSDs).
     * - cloud_essd2: specifies to use enhanced SSDs (ESSDs).
     * - cloud_essd3: specifies to use enhanced SSDs (ESSDs).
     */
    @JvmName("moksokgdhussrnnd")
    public suspend fun dbInstanceStorageType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbInstanceStorageType = mapped
    }

    /**
     * @param value The switch of delete protection. Valid values:
     * - true: delete protect.
     * - false: no delete protect.
     */
    @JvmName("eabnltygruoxqxjm")
    public suspend fun deletionProtection(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deletionProtection = mapped
    }

    /**
     * @param value The instance configuration type. Valid values:
     * - Up
     * - Down
     * - TempUpgrade
     * - Serverless
     */
    @JvmName("sjeaatbgdjwbdmwp")
    public suspend fun direction(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.direction = mapped
    }

    /**
     * @param value The method to change.  Default value: Immediate. Valid values:
     * - Immediate: The change immediately takes effect.
     * - MaintainTime: The change takes effect during the specified maintenance window. For more information, see ModifyDBInstanceMaintainTime.
     */
    @JvmName("gccppvlymsusxpdr")
    public suspend fun effectiveTime(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.effectiveTime = mapped
    }

    /**
     * @param value Database version. Value options can refer to the latest docs [CreateDBInstance](https://www.alibabacloud.com/help/doc-detail/26228.htm) `EngineVersion`.
     */
    @JvmName("xfjdukwwktmtbopk")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value Set it to true to make some parameter efficient when modifying them. Default to false.
     */
    @JvmName("nmtscofucsgkxmgr")
    public suspend fun forceRestart(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.forceRestart = mapped
    }

    /**
     * @param value Valid values are `Prepaid`, `Postpaid`, Default to `Postpaid`. The interval between the two conversion operations must be greater than 15 minutes. Only when this parameter is `Postpaid`, the instance can be released.
     */
    @JvmName("jadervddmpblndnn")
    public suspend fun instanceChargeType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceChargeType = mapped
    }

    /**
     * @param value The name of DB instance. It a string of 2 to 256 characters.
     */
    @JvmName("flijttebnmlxnbjm")
    public suspend fun instanceName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceName = mapped
    }

    /**
     * @param value User-defined DB instance storage space. Value range: [5, 2000] for MySQL/SQL Server HA dual node edition&#46; Increase progressively at a rate of 5 GB&#46; For details, see [Instance type table](https://www.alibabacloud.com/help/doc-detail/26312.htm).
     */
    @JvmName("lmaswkqcasycgdhp")
    public suspend fun instanceStorage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceStorage = mapped
    }

    /**
     * @param value DB Instance type. For details, see [Instance type table](https://www.alibabacloud.com/help/doc-detail/26312.htm).
     */
    @JvmName("lhkfhmocokkrwgdt")
    public suspend fun instanceType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceType = mapped
    }

    /**
     * @param value ID of the master instance.
     */
    @JvmName("vvnquelhskrcufqx")
    public suspend fun masterDbInstanceId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.masterDbInstanceId = mapped
    }

    /**
     * @param value The method that is used to modify the IP address whitelist. Default value: Cover. Valid values:
     * - Cover: Use the value of the SecurityIps parameter to overwrite the existing entries in the IP address whitelist.
     * - Append: Add the IP addresses and CIDR blocks that are specified in the SecurityIps parameter to the IP address whitelist.
     * - Delete: Delete IP addresses and CIDR blocks that are specified in the SecurityIps parameter from the IP address whitelist. You must retain at least one IP address or CIDR block.
     */
    @JvmName("ocuuiighftcnneyv")
    public suspend fun modifyMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.modifyMode = mapped
    }

    /**
     * @param value Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("cpvnitwunwxeshae")
    public suspend fun parameters(`value`: List<ReadOnlyInstanceParameterArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.parameters = mapped
    }

    /**
     * @param argument Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("dgqlmmvgccxuwoix")
    public suspend fun parameters(argument: List<suspend ReadOnlyInstanceParameterArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            ReadOnlyInstanceParameterArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.parameters = mapped
    }

    /**
     * @param argument Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("jxlphkuxitthsbxo")
    public suspend fun parameters(vararg argument: suspend ReadOnlyInstanceParameterArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            ReadOnlyInstanceParameterArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.parameters = mapped
    }

    /**
     * @param argument Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("gplsgtctelibtdmd")
    public suspend fun parameters(argument: suspend ReadOnlyInstanceParameterArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            ReadOnlyInstanceParameterArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.parameters = mapped
    }

    /**
     * @param values Set of parameters needs to be set after DB instance was launched. Available parameters can refer to the latest docs [View database parameter templates](https://www.alibabacloud.com/help/doc-detail/26284.htm). See `parameters` below.
     */
    @JvmName("pojhmlfxybpphbiu")
    public suspend fun parameters(vararg values: ReadOnlyInstanceParameterArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.parameters = mapped
    }

    /**
     * @param value The duration that you will buy DB instance (in month). It is valid when instance_charge_type is `PrePaid`. Valid values: [1~9], 12, 24, 36.
     */
    @JvmName("fxehsybqsklgsyol")
    public suspend fun period(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.period = mapped
    }

    /**
     * @param value The method that is used to verify the replication permission. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. In addition, this parameter is available only when the public key of the CA that issues client certificates is enabled. It is valid only when `ssl_enabled  = 1`. Valid values:
     * - cert
     * - perfer
     * - verify-ca
     * - verify-full (supported only when the instance runs PostgreSQL 12 or later)
     * > **NOTE:** Because of data backup and migration, change DB instance type and storage would cost 15~20 minutes. Please make full preparation before changing them.
     */
    @JvmName("nfjfahwpylkkoflp")
    public suspend fun replicationAcl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicationAcl = mapped
    }

    /**
     * @param value The ID of resource group which the DB read-only instance belongs.
     */
    @JvmName("wupvkofgwpkqfqbl")
    public suspend fun resourceGroupId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resourceGroupId = mapped
    }

    /**
     * @param value The type of IP address in the IP address whitelist.
     */
    @JvmName("jesddyfmjthdujlr")
    public suspend fun securityIpType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityIpType = mapped
    }

    /**
     * @param value List of IP addresses allowed to access all databases of an instance. The list contains up to 1,000 IP addresses, separated by commas. Supported formats include 0.0.0.0/0, 10.23.12.24 (IP), and 10.23.12.24/24 (Classless Inter-Domain Routing (CIDR) mode. /24 represents the length of the prefix in an IP address. The range of the prefix length is [1,32]).
     */
    @JvmName("xxbjmtgljcffuknn")
    public suspend fun securityIps(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityIps = mapped
    }

    /**
     * @param values List of IP addresses allowed to access all databases of an instance. The list contains up to 1,000 IP addresses, separated by commas. Supported formats include 0.0.0.0/0, 10.23.12.24 (IP), and 10.23.12.24/24 (Classless Inter-Domain Routing (CIDR) mode. /24 represents the length of the prefix in an IP address. The range of the prefix length is [1,32]).
     */
    @JvmName("qhuxvlrridlngxit")
    public suspend fun securityIps(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.securityIps = mapped
    }

    /**
     * @param value The content of the server certificate. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the CAType parameter to custom, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
     */
    @JvmName("yjflvbyaocayytht")
    public suspend fun serverCert(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverCert = mapped
    }

    /**
     * @param value The private key of the server certificate. This parameter is supported only when the instance runs PostgreSQL with standard or enhanced SSDs. If you set the CAType parameter to custom, you must also specify this parameter. It is valid only when `ssl_enabled  = 1`.
     */
    @JvmName("nshmrwxhaahdqftd")
    public suspend fun serverKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverKey = mapped
    }

    /**
     * @param value Specifies whether to enable or disable SSL encryption. Valid values:
     * - 1: enables SSL encryption
     * - 0: disables SSL encryption
     */
    @JvmName("wrlbodulumwfsoec")
    public suspend fun sslEnabled(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sslEnabled = mapped
    }

    /**
     * @param value The specific point in time when you want to perform the update. Specify the time in the ISO 8601 standard in the yyyy-MM-ddTHH:mm:ssZ format. It is valid only when `upgrade_db_instance_kernel_version = true`. The time must be in UTC.
     * > **NOTE:** This parameter takes effect only when you set the UpgradeTime parameter to SpecifyTime.
     */
    @JvmName("xndswxqipsjryvjf")
    public suspend fun switchTime(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.switchTime = mapped
    }

    /**
     * @param value A mapping of tags to assign to the resource.
     * - Key: It can be up to 64 characters in length. It cannot begin with "aliyun", "acs:", "http://", or "https://". It cannot be a null string.
     * - Value: It can be up to 128 characters in length. It cannot begin with "aliyun", "acs:", "http://", or "https://". It can be a null string.
     */
    @JvmName("sjkxmwgallvctrly")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A mapping of tags to assign to the resource.
     * - Key: It can be up to 64 characters in length. It cannot begin with "aliyun", "acs:", "http://", or "https://". It cannot be a null string.
     * - Value: It can be up to 128 characters in length. It cannot begin with "aliyun", "acs:", "http://", or "https://". It can be a null string.
     */
    @JvmName("rujhkmadjnolhbqu")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The minor engine version to which you want to update the instance. If you do not specify this parameter, the instance is updated to the latest minor engine version. It is valid only when `upgrade_db_instance_kernel_version = true`. You must specify the minor engine version in one of the following formats:
     * - PostgreSQL: rds_postgres_<Major engine version>00_<Minor engine version>. Example: rds_postgres_1200_20200830.
     * - MySQL: <RDS edition>_<Minor engine version>. Examples: rds_20200229, xcluster_20200229, and xcluster80_20200229. The following RDS editions are supported:
     * - rds: The instance runs RDS Basic or High-availability Edition.
     * - xcluster: The instance runs MySQL 5.7 on RDS Enterprise Edition.
     * - xcluster80: The instance runs MySQL 8.0 on RDS Enterprise Edition.
     * - SQLServer: <Minor engine version>. Example: 15.0.4073.23.
     * > **NOTE:** For more information about minor engine versions, see Release notes of minor AliPG versions, Release notes of minor AliSQL versions, and Release notes of minor engine versions of ApsaraDB RDS for SQL Server.
     */
    @JvmName("shqusivaxiwwhiya")
    public suspend fun targetMinorVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.targetMinorVersion = mapped
    }

    /**
     * @param value Whether to upgrade a minor version of the kernel. Valid values:
     * - true: upgrade
     * - false: not to upgrade
     */
    @JvmName("hsheoguqokjwpswk")
    public suspend fun upgradeDbInstanceKernelVersion(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.upgradeDbInstanceKernelVersion = mapped
    }

    /**
     * @param value The method to update the minor engine version. Default value: Immediate. It is valid only when `upgrade_db_instance_kernel_version = true`. Valid values:
     * - Immediate: The minor engine version is immediately updated.
     * - MaintainTime: The minor engine version is updated during the maintenance window. For more information about how to change the maintenance window, see ModifyDBInstanceMaintainTime.
     * - SpecifyTime: The minor engine version is updated at the point in time you specify.
     */
    @JvmName("atsoymmhwsigtfqa")
    public suspend fun upgradeTime(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.upgradeTime = mapped
    }

    /**
     * @param value The virtual switch ID to launch DB instances in one VPC.
     */
    @JvmName("mxcudfoaeyogxlmg")
    public suspend fun vswitchId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vswitchId = mapped
    }

    /**
     * @param value The network type of the IP address whitelist. Default value: MIX. Valid values:
     * - Classic: classic network in enhanced whitelist mode
     * - VPC: virtual private cloud (VPC) in enhanced whitelist mode
     * - MIX: standard whitelist mode
     * > **NOTE:** In standard whitelist mode, IP addresses and CIDR blocks can be added only to the default IP address whitelist. In enhanced whitelist mode, IP addresses and CIDR blocks can be added to both IP address whitelists of the classic network type and those of the VPC network type.
     */
    @JvmName("xgdxslbebdtidlsi")
    public suspend fun whitelistNetworkType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.whitelistNetworkType = mapped
    }

    /**
     * @param value The Zone to launch the DB instance.
     */
    @JvmName("hsawaxpsfkbjibim")
    public suspend fun zoneId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.zoneId = mapped
    }

    internal fun build(): ReadOnlyInstanceArgs = ReadOnlyInstanceArgs(
        acl = acl,
        autoRenew = autoRenew,
        autoRenewPeriod = autoRenewPeriod,
        caType = caType,
        clientCaCert = clientCaCert,
        clientCaEnabled = clientCaEnabled,
        clientCertRevocationList = clientCertRevocationList,
        clientCrlEnabled = clientCrlEnabled,
        dbInstanceIpArrayAttribute = dbInstanceIpArrayAttribute,
        dbInstanceIpArrayName = dbInstanceIpArrayName,
        dbInstanceStorageType = dbInstanceStorageType,
        deletionProtection = deletionProtection,
        direction = direction,
        effectiveTime = effectiveTime,
        engineVersion = engineVersion,
        forceRestart = forceRestart,
        instanceChargeType = instanceChargeType,
        instanceName = instanceName,
        instanceStorage = instanceStorage,
        instanceType = instanceType,
        masterDbInstanceId = masterDbInstanceId,
        modifyMode = modifyMode,
        parameters = parameters,
        period = period,
        replicationAcl = replicationAcl,
        resourceGroupId = resourceGroupId,
        securityIpType = securityIpType,
        securityIps = securityIps,
        serverCert = serverCert,
        serverKey = serverKey,
        sslEnabled = sslEnabled,
        switchTime = switchTime,
        tags = tags,
        targetMinorVersion = targetMinorVersion,
        upgradeDbInstanceKernelVersion = upgradeDbInstanceKernelVersion,
        upgradeTime = upgradeTime,
        vswitchId = vswitchId,
        whitelistNetworkType = whitelistNetworkType,
        zoneId = zoneId,
    )
}
