package org.vitrivr.cottontail.execution.operators.predicates

import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.flow
import org.vitrivr.cottontail.database.queries.components.KnnPredicate
import org.vitrivr.cottontail.execution.ExecutionEngine
import org.vitrivr.cottontail.execution.operators.basics.Operator
import org.vitrivr.cottontail.execution.operators.basics.OperatorStatus
import org.vitrivr.cottontail.execution.operators.basics.PipelineBreaker
import org.vitrivr.cottontail.math.knn.selection.ComparablePair
import org.vitrivr.cottontail.math.knn.selection.MinHeapSelection
import org.vitrivr.cottontail.math.knn.selection.MinSingleSelection
import org.vitrivr.cottontail.math.knn.selection.Selection
import org.vitrivr.cottontail.model.basics.ColumnDef
import org.vitrivr.cottontail.model.basics.Name
import org.vitrivr.cottontail.model.basics.Record
import org.vitrivr.cottontail.model.recordset.StandaloneRecord
import org.vitrivr.cottontail.model.values.DoubleValue
import org.vitrivr.cottontail.model.values.types.VectorValue
import org.vitrivr.cottontail.utilities.math.KnnUtilities

/**
 * Performs a kNN lookup on the input generated by the parent [Operator] using the given [KnnPredicate].
 *
 * This is a [PipelineBreaker]
 *
 * @author Ralph Gasser
 * @version 1.1.1
 */
class KnnOperator<T : VectorValue<*>>(parent: Operator, context: ExecutionEngine.ExecutionContext, val knn: KnnPredicate<T>) : PipelineBreaker(parent, context) {

    /** The columns produced by this [KnnOperator]. */
    override val columns: Array<ColumnDef<*>> = arrayOf(
        *this.parent.columns,
        KnnUtilities.columnDef(this.knn.column.name.entity())
    )

    override fun prepareOpen() { /* NoOp */ }

    override fun prepareClose() { /* NoOp */ }

    /**
     * Converts this [KnnOperator] to a [Flow] and returns it.
     *
     * @param scope The [CoroutineScope] used for execution
     * @return [Flow] representing this [KnnOperator]
     *
     * @throws IllegalStateException If this [Operator.status] is not [OperatorStatus.OPEN]
     */
    override fun toFlow(scope: CoroutineScope): Flow<Record> {
        check(this.status == OperatorStatus.OPEN) { "Cannot convert operator $this to flow because it is in state ${this.status}." }

        /* Obtain parent flow. */
        val parentFlow = this.parent.toFlow(scope)

        /* Prepare data structures and logic for kNN. */
        val knnSet: List<Selection<ComparablePair<Record, DoubleValue>>> = if (this.knn.k == 1) {
            knn.query.map { MinSingleSelection() }
        } else {
            knn.query.map { MinHeapSelection(this.knn.k) }
        }
        val action: (Record) -> Unit = if (this.knn.weights != null) {
            {
                val value = it[this.knn.column]
                if (value != null) {
                    this.knn.query.forEachIndexed { i, query ->
                        knnSet[i].offer(ComparablePair(it, this.knn.distance(query, value, this.knn.weights[i])))
                    }
                }
            }
        } else {
            {
                val value = it[this.knn.column]
                if (value != null) {
                    this.knn.query.forEachIndexed { i, query ->
                        knnSet[i].offer(ComparablePair(it, this.knn.distance(query, value)))
                    }
                }
            }
        }

        /* Generate new flow. */
        return flow {
            parentFlow.collect { record ->
                action(record)
            }
            for (knn in knnSet) {
                for (i in 0 until knn.size) {
                    emit(StandaloneRecord(knn[i].first.tupleId, this@KnnOperator.columns, arrayOf(*knn[i].first.values, DoubleValue(knn[i].second))))
                }
            }
        }
    }
}