package org.vitrivr.cottontail.execution.operators.predicates

import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import org.vitrivr.cottontail.database.queries.components.KnnPredicate
import org.vitrivr.cottontail.execution.ExecutionEngine
import org.vitrivr.cottontail.execution.operators.basics.MergingPipelineBreaker
import org.vitrivr.cottontail.execution.operators.basics.Operator
import org.vitrivr.cottontail.execution.operators.basics.OperatorStatus
import org.vitrivr.cottontail.math.knn.selection.ComparablePair
import org.vitrivr.cottontail.math.knn.selection.MinHeapSelection
import org.vitrivr.cottontail.math.knn.selection.MinSingleSelection
import org.vitrivr.cottontail.math.knn.selection.Selection
import org.vitrivr.cottontail.model.basics.ColumnDef
import org.vitrivr.cottontail.model.basics.Name
import org.vitrivr.cottontail.model.basics.Record
import org.vitrivr.cottontail.model.recordset.StandaloneRecord
import org.vitrivr.cottontail.model.values.DoubleValue
import org.vitrivr.cottontail.utilities.math.KnnUtilities

/**
 * Performs a kNN lookup on the input generated by the parent [Operator]s using the given [KnnPredicate]. The incoming
 * branches are executed in parallel (if possible).
 *
 * This operator is a [MergingPipelineBreaker]
 *
 * @author Ralph Gasser
 * @version 1.1.0
 */
class ParallelKnnOperator(parents: List<Operator>, context: ExecutionEngine.ExecutionContext, val knn: KnnPredicate<*>) : MergingPipelineBreaker(parents, context) {

    /** The columns produced by this [ParallelKnnOperator]. */
    override val columns: Array<ColumnDef<*>> = arrayOf(
            *this.parents.first().columns,
            ColumnDef(this.knn.column.name.entity()?.column(KnnUtilities.DISTANCE_COLUMN_NAME)
                    ?: Name.ColumnName(KnnUtilities.DISTANCE_COLUMN_NAME), KnnUtilities.DISTANCE_COLUMN_TYPE)
    )

    override fun prepareOpen() { /* NoOp. */
    }

    override fun prepareClose() { /* NoOp. */
    }

    /**
     * Converts this [ParallelKnnOperator] to a [Flow] and returns it.
     *
     * @param scope The [CoroutineScope] used for execution
     * @return [Flow] representing this [ParallelKnnOperator]
     *
     * @throws IllegalStateException If this [Operator.status] is not [OperatorStatus.OPEN]
     */
    override fun toFlow(scope: CoroutineScope): Flow<Record> {
        check(this.status == OperatorStatus.OPEN) { "Cannot convert operator $this to flow because it is in state ${this.status}." }

        /* Obtain parent flows. */
        val parentFlows = this.parents.map { it.toFlow(scope) }

        /* Prepare data structures and logic for kNN. */
        val knnSet: List<Selection<ComparablePair<Record, DoubleValue>>> = if (this.knn.k == 1) {
            knn.query.map { MinSingleSelection() }
        } else {
            knn.query.map { MinHeapSelection(this.knn.k) }
        }
        val action: (Record) -> Unit = if (this.knn.weights != null) {
            {
                val value = it[this.knn.column]
                if (value != null) {
                    this.knn.query.forEachIndexed { i, query ->
                        knnSet[i].offer(ComparablePair(it, this.knn.distance(query, value, this.knn.weights[i])))
                    }
                }
            }
        } else {
            {
                val value = it[this.knn.column]
                if (value != null) {
                    this.knn.query.forEachIndexed { i, query ->
                        knnSet[i].offer(ComparablePair(it, this.knn.distance(query, value)))
                    }
                }
            }
        }

        /* Compose new flow. */
        return flow {
            /* Execute incoming flows and wait for completion. */
            parentFlows.map { flow ->
                flow.onEach { record ->
                    action(record)
                }.launchIn(scope)
            }.forEach {
                it.join()
            }

            /* Emit kNN values. */
            for (knn in knnSet) {
                for (i in 0 until knn.size) {
                    emit(StandaloneRecord(knn[i].first.tupleId, this@ParallelKnnOperator.columns, arrayOf(*knn[i].first.values, DoubleValue(knn[i].second))))
                }
            }
        }
    }
}