package org.vitrivr.cottontail.database.queries.planning.nodes.logical.projection

import org.vitrivr.cottontail.database.column.ColumnDef
import org.vitrivr.cottontail.database.queries.planning.nodes.logical.UnaryLogicalOperatorNode
import org.vitrivr.cottontail.database.queries.planning.nodes.physical.projection.AggregatingProjectionPhysicalOperatorNode
import org.vitrivr.cottontail.database.queries.projection.Projection
import org.vitrivr.cottontail.model.basics.Name
import org.vitrivr.cottontail.model.exceptions.QueryException

/**
 * A [UnaryLogicalOperatorNode] that represents a projection operation involving aggregate functions such as [Projection.MAX], [Projection.MIN] or [Projection.SUM].
 *
 * @author Ralph Gasser
 * @version 2.1.0
 */
class AggregatingProjectionLogicalOperatorNode(input: Logical? = null, type: Projection, fields: List<Pair<Name.ColumnName, Name.ColumnName?>>) : AbstractProjectionLogicalOperatorOperator(input, type, fields) {

    /** The [ColumnDef] generated by this [AggregatingProjectionLogicalOperatorNode]. */
    override val columns: Array<ColumnDef<*>>
        get() = this.fields.map {
            val inColumn = this.input?.columns?.find { c -> c.name == it.first } ?: throw QueryException.QueryBindException("Column with name $it could not be found on input.")
            val outColumn = inColumn.copy(name = it.second ?: inColumn.name)
            if (!inColumn.type.numeric) throw QueryException.QueryBindException("Projection of type ${this.type} can only be applied to numeric column, which $inColumn isn't.")
            outColumn
        }.toTypedArray()

    /** The [ColumnDef] required by this [AggregatingProjectionLogicalOperatorNode]. */
    override val requires: Array<ColumnDef<*>>
        get() = this.fields.map {
            this.input?.columns?.find { c -> c.name == it.first } ?: throw QueryException.QueryBindException("Column with name $it could not be found on input.")
        }.toTypedArray()

    init {
        /* Sanity check. */
        require(this.type in arrayOf(Projection.MIN, Projection.MAX, Projection.MAX, Projection.SUM)) {
            "Projection of type ${this.type} cannot be used with instances of AggregatingProjectionLogicalNodeExpression."
        }
    }

    /**
     * Creates and returns a copy of this [AggregatingProjectionLogicalOperatorNode] without any children or parents.
     *
     * @return Copy of this [AggregatingProjectionLogicalOperatorNode].
     */
    override fun copy() = AggregatingProjectionLogicalOperatorNode(type = this.type, fields = this.fields)


    /**
     * Returns a [AggregatingProjectionPhysicalOperatorNode] representation of this [AggregatingProjectionLogicalOperatorNode]
     *
     * @return [AggregatingProjectionPhysicalOperatorNode]
     */
    override fun implement(): Physical = AggregatingProjectionPhysicalOperatorNode(this.input?.implement(), this.type, this.fields)
}