package org.vitrivr.cottontail.functions.basics

import org.vitrivr.cottontail.model.values.types.Value

/**
 * A [FunctionGenerator] can be used by Cottontail DB to generate an arbitrary [Function].
 *
 * Some [Function]s are parametrized by the types arguments they produce and depending on a concrete type,
 * a separate instance is required. Furthermore, some [Function]s may have local state information that
 * requires a new instance to be returned for each use.
 *
 * @author Ralph Gasser
 * @version 1.0.0
 */
sealed interface FunctionGenerator<out R: Value> {
    /** [Signature.Open] of the [Function]s generated by this [FunctionGenerator]. */
    val signature: Signature.Open<out R>

    /**
     * Generates a [Function] for the given arguments.
     *
     * @param arguments The argument [Value]s to generate a [Function] for.
     * @return The generated [Function]
     */
    fun generate(vararg arguments: Value): Function<R> = generate(*arguments.map { Argument.Typed(it.type) }.toTypedArray())

    /**
     * Generates a [Function] for the given arguments.
     *
     * @param arguments The [Argument.Typed]s to generate a [Function] for.
     * @return The generated [Function]
     */
    fun generate(vararg arguments: Argument.Typed<*>): Function<R>
}