package org.vitrivr.cottontail.dbms.queries.operators.physical.projection

import org.vitrivr.cottontail.core.database.ColumnDef
import org.vitrivr.cottontail.core.database.Name
import org.vitrivr.cottontail.core.queries.planning.cost.Cost
import org.vitrivr.cottontail.core.values.types.Types
import org.vitrivr.cottontail.dbms.exceptions.QueryException
import org.vitrivr.cottontail.dbms.execution.operators.basics.Operator
import org.vitrivr.cottontail.dbms.execution.operators.projection.*
import org.vitrivr.cottontail.dbms.queries.QueryContext
import org.vitrivr.cottontail.dbms.queries.operators.physical.UnaryPhysicalOperatorNode
import org.vitrivr.cottontail.dbms.queries.projection.Projection

/**
 * A [UnaryPhysicalOperatorNode] that represents a projection operation involving aggregate functions such as [Projection.MAX], [Projection.MIN] or [Projection.SUM].
 *
 * @author Ralph Gasser
 * @version 2.3.0
 */
class AggregatingProjectionPhysicalOperatorNode(input: Physical? = null, type: Projection, val fields: List<Name.ColumnName>) : AbstractProjectionPhysicalOperatorNode(input, type) {

    /** The [ColumnDef] generated by this [AggregatingProjectionPhysicalOperatorNode]. */
    override val columns: List<ColumnDef<*>>
        get() = this.fields.map {
            val col = this.input?.columns?.find { c -> c.name == it } ?: throw QueryException.QueryBindException("Column with name $it could not be found on input.")
            if (col.type !is Types.Numeric<*>) throw QueryException.QueryBindException("Projection of type ${this.type} can only be applied to numeric column, which $col isn't.")
            col
        }

    /** The [ColumnDef] required by this [AggregatingProjectionPhysicalOperatorNode]. */
    override val requires: List<ColumnDef<*>>
        get() = this.fields.map {
            this.input?.columns?.find { c -> c.name == it } ?: throw QueryException.QueryBindException("Column with name $it could not be found on input.")
        }

    /** The output size of this [AggregatingProjectionPhysicalOperatorNode] is always one. */
    override val outputSize: Long = 1

    /** The [Cost] of a [AggregatingProjectionPhysicalOperatorNode]. */
    override val cost: Cost
        get() = (Cost.MEMORY_ACCESS + Cost.FLOP) * (3.0f * (this.input?.outputSize ?: 0))

    init {
        /* Sanity check. */
        require(this.type in arrayOf(Projection.MIN, Projection.MAX, Projection.MAX, Projection.SUM)) {
            "Projection of type ${this.type} cannot be used with instances of AggregatingProjectionPhysicalOperatorNode."
        }
    }

    /**
     * Creates and returns a copy of this [AggregatingProjectionPhysicalOperatorNode] without any children or parents.
     *
     * @return Copy of this [AggregatingProjectionPhysicalOperatorNode].
     */
    override fun copy() = AggregatingProjectionPhysicalOperatorNode(type = this.type, fields = this.fields)

    /**
     * Converts this [CountProjectionPhysicalOperatorNode] to a [CountProjectionOperator].
     *
     * @param ctx The [QueryContext] used for the conversion (e.g. late binding).
     */
    override fun toOperator(ctx: QueryContext): Operator {
        val `in` = this.input?.toOperator(ctx) ?: throw IllegalStateException("Cannot convert disconnected OperatorNode to Operator (node = $this)")
        return when (this.type) {
            Projection.SUM -> SumProjectionOperator(`in`, this.fields)
            Projection.MAX -> MaxProjectionOperator(`in`, this.fields)
            Projection.MIN -> MinProjectionOperator(`in`, this.fields)
            Projection.MEAN -> MeanProjectionOperator(`in`, this.fields)
            else -> throw IllegalStateException("An AggregatingProjectionPhysicalOperatorNode requires a project of type SUM, MAX, MIN or MEAN but encountered ${this.type}.")
        }
    }

    /**
     * Compares this [AggregatingProjectionPhysicalOperatorNode] to another object.
     *
     * @param other The other [Any] to compare this [AggregatingProjectionPhysicalOperatorNode] to.
     * @return True if other equals this, false otherwise.
     */
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is AggregatingProjectionPhysicalOperatorNode) return false
        if (this.type != other.type) return false
        if (this.fields != other.fields) return false
        return true
    }

    /**
     * Generates and returns a hash code for this [AggregatingProjectionPhysicalOperatorNode].
     */
    override fun hashCode(): Int {
        var result = this.type.hashCode()
        result = 31 * result + this.fields.hashCode()
        return result
    }
}