package org.vitrivr.cottontail.dbms.queries.operators.physical.projection

import org.vitrivr.cottontail.core.database.ColumnDef
import org.vitrivr.cottontail.core.queries.binding.Binding
import org.vitrivr.cottontail.core.queries.planning.cost.Cost
import org.vitrivr.cottontail.dbms.execution.operators.projection.CountProjectionOperator
import org.vitrivr.cottontail.dbms.queries.QueryContext
import org.vitrivr.cottontail.dbms.queries.operators.logical.projection.CountProjectionLogicalOperatorNode
import org.vitrivr.cottontail.dbms.queries.operators.physical.UnaryPhysicalOperatorNode
import org.vitrivr.cottontail.dbms.queries.projection.Projection

/**
 * A [UnaryPhysicalOperatorNode] that represents a projection operation involving aggregate functions such as [Projection.COUNT].
 *
 * @author Ralph Gasser
 * @version 2.3.0
 */
class CountProjectionPhysicalOperatorNode(input: Physical? = null, val out: Binding.Column) : AbstractProjectionPhysicalOperatorNode(input, Projection.COUNT) {

    /** The [ColumnDef] generated by this [CountProjectionLogicalOperatorNode]. */
    override val columns: List<ColumnDef<*>>
        = listOf(this.out.column)


    /** The [ColumnDef] required by this [CountProjectionLogicalOperatorNode]. */
    override val requires: List<ColumnDef<*>>
        get() = emptyList()

    /** The output size of this [CountProjectionPhysicalOperatorNode] is always one. */
    override val outputSize: Long = 1

    /** The [Cost] of a [CountProjectionPhysicalOperatorNode]. */
    override val cost: Cost
        get() = Cost.MEMORY_ACCESS * (this.input?.outputSize ?: 0)

    /**The [ExistsProjectionPhysicalOperatorNode] cannot be partitioned. */
    override val canBePartitioned: Boolean = false

    /**
     * Creates and returns a copy of this [CountProjectionPhysicalOperatorNode] without any children or parents.
     *
     * @return Copy of this [CountProjectionPhysicalOperatorNode].
     */
    override fun copy() = CountProjectionPhysicalOperatorNode(out = this.out)

    /**
     * Converts this [CountProjectionPhysicalOperatorNode] to a [CountProjectionOperator].
     *
     * @param ctx The [QueryContext] used for the conversion (e.g. late binding).
     */
    override fun toOperator(ctx: QueryContext) = CountProjectionOperator(this.input?.toOperator(ctx) ?: throw IllegalStateException("Cannot convert disconnected OperatorNode to Operator (node = $this)"))

    /**
     * Compares this [CountProjectionPhysicalOperatorNode] to another object.
     *
     * @param other The other [Any] to compare this [CountProjectionPhysicalOperatorNode] to.
     * @return True if other equals this, false otherwise.
     */
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is CountProjectionPhysicalOperatorNode) return false
        if (this.type != other.type) return false
        if (this.out != other.out) return false
        return true
    }

    /**
     * Generates and returns a hash code for this [CountProjectionPhysicalOperatorNode].
     */
    override fun hashCode(): Int {
        var result = type.hashCode()
        result = 31 * result + this.out.hashCode()
        return result
    }
}