package org.vitrivr.cottontail.dbms.queries.operators.logical.projection

import org.vitrivr.cottontail.core.database.ColumnDef
import org.vitrivr.cottontail.core.database.Name
import org.vitrivr.cottontail.dbms.exceptions.QueryException
import org.vitrivr.cottontail.dbms.queries.operators.physical.projection.SelectProjectionPhysicalOperatorNode
import org.vitrivr.cottontail.dbms.queries.projection.Projection

/**
 * A [AbstractProjectionLogicalOperatorOperator] that formalizes a [Projection.SELECT] operation in a Cottontail DB query plan.
 *
 * @author Ralph Gasser
 * @version 2.4.0
 */
class SelectProjectionLogicalOperatorNode(input: Logical? = null, val fields: List<Name.ColumnName>): AbstractProjectionLogicalOperatorOperator(input, Projection.SELECT) {

    /** The [ColumnDef] generated by this [SelectProjectionLogicalOperatorNode]. */
    override val columns: List<ColumnDef<*>>
        get() = this.input?.columns?.filter { c -> this.fields.any { f -> f == c.name }} ?: emptyList()

    /** The [ColumnDef] required by this [SelectProjectionLogicalOperatorNode]. */
    override val requires: List<ColumnDef<*>>
        get() = this.columns

    init {
        /* Sanity check. */
        if (this.fields.isEmpty()) {
            throw QueryException.QuerySyntaxException("Projection of type $type must specify at least one column.")
        }
    }

    /**
     * Creates and returns a copy of this [SelectProjectionLogicalOperatorNode] without any children or parents.
     *
     * @return Copy of this [SelectProjectionLogicalOperatorNode].
     */
    override fun copy() = SelectProjectionLogicalOperatorNode(fields = this.fields)

    /**
     * Returns a [SelectProjectionPhysicalOperatorNode] representation of this [SelectProjectionLogicalOperatorNode]
     *
     * @return [SelectProjectionPhysicalOperatorNode]
     */
    override fun implement(): Physical = SelectProjectionPhysicalOperatorNode(this.input?.implement(), this.fields)

    /**
     * Compares this [SelectProjectionLogicalOperatorNode] to another object.
     *
     * @param other The other [Any] to compare this [SelectProjectionLogicalOperatorNode] to.
     * @return True if other equals this, false otherwise.
     */
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is SelectProjectionLogicalOperatorNode) return false
        if (this.type != other.type) return false
        if (this.fields != other.fields) return false
        return true
    }

    /**
     * Generates and returns a hash code for this [SelectProjectionLogicalOperatorNode].
     */
    override fun hashCode(): Int {
        var result = this.type.hashCode()
        result = 31 * result + this.fields.hashCode()
        return result
    }
}