package org.vitrivr.cottontail.dbms.queries.operators.physical.projection

import org.vitrivr.cottontail.core.database.ColumnDef
import org.vitrivr.cottontail.core.queries.binding.Binding
import org.vitrivr.cottontail.core.queries.nodes.traits.NotPartitionableTrait
import org.vitrivr.cottontail.core.queries.nodes.traits.Trait
import org.vitrivr.cottontail.core.queries.nodes.traits.TraitType
import org.vitrivr.cottontail.core.queries.planning.cost.Cost
import org.vitrivr.cottontail.dbms.execution.operators.projection.ExistsProjectionOperator
import org.vitrivr.cottontail.dbms.execution.transactions.TransactionContext
import org.vitrivr.cottontail.dbms.queries.context.QueryContext
import org.vitrivr.cottontail.dbms.queries.operators.physical.UnaryPhysicalOperatorNode
import org.vitrivr.cottontail.dbms.queries.projection.Projection

/**
 * A [UnaryPhysicalOperatorNode] that represents a projection operation involving aggregate functions such as [Projection.EXISTS].
 *
 * @author Ralph Gasser
 * @version 2.3.0
 */
class ExistsProjectionPhysicalOperatorNode(input: Physical? = null, val out: Binding.Column) : AbstractProjectionPhysicalOperatorNode(input, Projection.EXISTS) {

    /** The [ColumnDef] generated by this [ExistsProjectionPhysicalOperatorNode]. */
    override val columns: List<ColumnDef<*>>
        get() = listOf(this.out.column)

    /** The [ColumnDef] required by this [ExistsProjectionPhysicalOperatorNode]. */
    override val requires: List<ColumnDef<*>>
        get() = emptyList()

    /** The output size of this [ExistsProjectionPhysicalOperatorNode] is always one. */
    override val outputSize: Long = 1

    /** The [Cost] of a [ExistsProjectionPhysicalOperatorNode]. */
    override val cost: Cost
        get() = Cost.MEMORY_ACCESS

    /** The [CountProjectionPhysicalOperatorNode] cannot be partitioned. */
    override val traits: Map<TraitType<*>, Trait>
        get() = (this.input?.traits ?: emptyMap()) + (NotPartitionableTrait to NotPartitionableTrait)

    /**
     * Creates and returns a copy of this [ExistsProjectionPhysicalOperatorNode] without any children or parents.
     *
     * @return Copy of this [ExistsProjectionPhysicalOperatorNode].
     */
    override fun copy() = ExistsProjectionPhysicalOperatorNode(out = this.out)

    /**
     * Converts this [ExistsProjectionPhysicalOperatorNode] to a [ExistsProjectionOperator].
     *
     * @param ctx The [TransactionContext] used for the conversion (e.g. late binding).
     */
    override fun toOperator(ctx: QueryContext) = ExistsProjectionOperator(
        this.input?.toOperator(ctx) ?: throw IllegalStateException("Cannot convert disconnected OperatorNode to Operator (node = $this)"),
        this.out
    )

    /**
     * Compares this [ExistsProjectionPhysicalOperatorNode] to another object.
     *
     * @param other The other [Any] to compare this [ExistsProjectionPhysicalOperatorNode] to.
     * @return True if other equals this, false otherwise.
     */
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is ExistsProjectionPhysicalOperatorNode) return false
        if (this.type != other.type) return false
        if (this.out != other.out) return false
        return true
    }

    /**
     * Generates and returns a hash code for this [ExistsProjectionPhysicalOperatorNode].
     */
    override fun hashCode(): Int {
        var result = this.type.hashCode()
        result = 31 * result + this.out.hashCode()
        return result
    }
}