package org.vitrivr.cottontail.dbms.queries.operators.physical.projection

import org.vitrivr.cottontail.config.Config
import org.vitrivr.cottontail.core.database.ColumnDef
import org.vitrivr.cottontail.core.database.Name
import org.vitrivr.cottontail.core.queries.planning.cost.Cost
import org.vitrivr.cottontail.dbms.exceptions.QueryException
import org.vitrivr.cottontail.dbms.execution.operators.basics.Operator
import org.vitrivr.cottontail.dbms.execution.operators.projection.SelectDistinctProjectionOperator
import org.vitrivr.cottontail.dbms.queries.context.QueryContext
import org.vitrivr.cottontail.dbms.queries.projection.Projection

/**
 * Formalizes a [Projection.SELECT_DISTINCT] operation in a Cottontail DB query plan.
 *
 * @author Ralph Gasser
 * @version 1.0.0
 */
class SelectDistinctProjectionPhysicalOperatorNode(input: Physical? = null, val fields: List<Pair<Name.ColumnName,Boolean>>, val config: Config): AbstractProjectionPhysicalOperatorNode(input, Projection.SELECT_DISTINCT) {
    /** The name of this [SelectDistinctProjectionPhysicalOperatorNode]. */
    override val name: String = "SelectDistinct"

    /** The [ColumnDef] generated by this [SelectDistinctProjectionPhysicalOperatorNode]. */
    override val columns: List<ColumnDef<*>>
        get() = this.input?.columns?.filter { c -> fields.any { f -> f.first == c.name }} ?: emptyList()

    /** The [ColumnDef] required by this [SelectProjectionPhysicalOperatorNode]. */
    override val requires: List<ColumnDef<*>>
        get() = this.columns

    /** The [Cost] of a [SelectDistinctProjectionPhysicalOperatorNode]. */
    override val cost: Cost
        get() = Cost.MEMORY_ACCESS * (this.outputSize * this.fields.size) + Cost.DISK_ACCESS_WRITE * (16 * this.fields.size * this.outputSize)

    init {
        /* Sanity check. */
        if (this.fields.isEmpty()) {
            throw QueryException.QuerySyntaxException("Projection of type $type must specify at least one column.")
        }
    }

    /**
     * Creates and returns a copy of this [SelectDistinctProjectionPhysicalOperatorNode] without any children or parents.
     *
     * @return Copy of this [SelectDistinctProjectionPhysicalOperatorNode].
     */
    override fun copy() = SelectDistinctProjectionPhysicalOperatorNode(fields = this.fields, config = this.config)

    /**
     * Converts this [SelectDistinctProjectionPhysicalOperatorNode] to a [SelectDistinctProjectionPhysicalOperatorNode].
     *
     * @param ctx The [QueryContext] used for the conversion (e.g. late binding).
     */
    override fun toOperator(ctx: QueryContext): Operator {
        val input = this.input ?: throw IllegalStateException("Cannot convert disconnected OperatorNode to Operator (node = $this)")
        return SelectDistinctProjectionOperator(input.toOperator(ctx), this.fields, this.config)
    }

    /**
     *
     */
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is SelectDistinctProjectionPhysicalOperatorNode) return false
        if (this.type != other.type) return false
        if (this.fields != other.fields) return false
        return true
    }

    /**
     * Generates and returns a hash code for this [SelectDistinctProjectionPhysicalOperatorNode].
     */
    override fun hashCode(): Int {
        var result = this.type.hashCode()
        result = 33 * result + this.fields.hashCode()
        return result
    }
}