package org.vitrivr.cottontail.dbms.queries.operators.physical.projection

import org.vitrivr.cottontail.core.database.ColumnDef
import org.vitrivr.cottontail.core.queries.Digest
import org.vitrivr.cottontail.core.queries.binding.Binding
import org.vitrivr.cottontail.core.queries.nodes.traits.NotPartitionableTrait
import org.vitrivr.cottontail.core.queries.nodes.traits.Trait
import org.vitrivr.cottontail.core.queries.nodes.traits.TraitType
import org.vitrivr.cottontail.core.queries.planning.cost.Cost
import org.vitrivr.cottontail.dbms.execution.operators.projection.ExistsProjectionOperator
import org.vitrivr.cottontail.dbms.execution.transactions.Transaction
import org.vitrivr.cottontail.dbms.queries.context.QueryContext
import org.vitrivr.cottontail.dbms.queries.operators.basics.OperatorNode
import org.vitrivr.cottontail.dbms.queries.operators.basics.UnaryPhysicalOperatorNode
import org.vitrivr.cottontail.dbms.queries.projection.Projection

/**
 * A [UnaryPhysicalOperatorNode] that represents a projection operation involving aggregate functions such as [Projection.EXISTS].
 *
 * @author Ralph Gasser
 * @version 2.9.0
 */
class ExistsProjectionPhysicalOperatorNode(input: Physical, val out: Binding.Column) : UnaryPhysicalOperatorNode(input) {
    /** The name of this [CountProjectionPhysicalOperatorNode]. */
    override val name: String
        get() = Projection.EXISTS.label()

    /** The [ColumnDef] generated by this [ExistsProjectionPhysicalOperatorNode]. */
    override val columns: List<Binding.Column> = listOf(this.out)

    /** The output size of this [ExistsProjectionPhysicalOperatorNode] is always one. */
    override val outputSize: Long = 1

    /** The [Cost] of a [ExistsProjectionPhysicalOperatorNode]. */
    override val cost: Cost
        get() = Cost.MEMORY_ACCESS

    /** The [ExistsProjectionPhysicalOperatorNode] cannot be partitioned. */
    override val traits: Map<TraitType<*>, Trait> by lazy {
        this.input.traits + (NotPartitionableTrait to NotPartitionableTrait)
    }

    /**
     * Creates and returns a copy of this [AggregatingProjectionPhysicalOperatorNode] using the given parents as input.
     *
     * @param input The [OperatorNode.Physical]s that act as input.
     * @return Copy of this [AggregatingProjectionPhysicalOperatorNode].
     */
    override fun copyWithNewInput(vararg input: Physical): ExistsProjectionPhysicalOperatorNode {
        require(input.size == 1) { "The input arity for ExistsProjectionPhysicalOperatorNode.copyWithNewInput() must be 1 but is ${input.size}. This is a programmer's error!"}
        return ExistsProjectionPhysicalOperatorNode(input = input[0], out = this.out)
    }

    /**
     * Converts this [ExistsProjectionPhysicalOperatorNode] to a [ExistsProjectionOperator].
     *
     * @param ctx The [Transaction] used for the conversion (e.g. late binding).
     */
    override fun toOperator(ctx: QueryContext) = ExistsProjectionOperator(this.input.toOperator(ctx), this.out, ctx)

    /** Generates and returns a [String] representation of this [ExistsProjectionOperator]. */
    override fun toString() = "${super.toString()}[${this.columns.joinToString(",") { it.column.name.toString() }}]"

    /**
     * Generates and returns a [Digest] for this [ExistsProjectionPhysicalOperatorNode].
     *
     * @return [Digest]
     */
    override fun digest(): Digest {
        var result = Projection.EXISTS.hashCode().toLong()
        result += 31L * result + this.out.hashCode()
        return result
    }
}