package org.vitrivr.cottontail.dbms.execution.operators.predicates

import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.filter
import org.vitrivr.cottontail.core.database.ColumnDef
import org.vitrivr.cottontail.core.queries.predicates.BooleanPredicate
import org.vitrivr.cottontail.core.tuple.Tuple
import org.vitrivr.cottontail.dbms.execution.operators.basics.Operator
import org.vitrivr.cottontail.dbms.queries.context.QueryContext

/**
 * An [Operator.PipelineOperator] used during query execution. Filters the input generated by the
 * parent [Operator] using the given [BooleanPredicate].
 *
 * @author Ralph Gasser
 * @version 2.0.0
 */
class FilterOperator(parent: Operator, private val predicate: BooleanPredicate, override val context: QueryContext) : Operator.PipelineOperator(parent) {

    /** Columns returned by [FilterOperator] depend on the parent [Operator]. */
    override val columns: List<ColumnDef<*>>
        get() = this.parent.columns

    /** [FilterOperator] does not act as a pipeline breaker. */
    override val breaker: Boolean = false

    /**
     * Converts this [FilterOperator] to a [Flow] and returns it.
     *
     * @return [Flow] representing this [FilterOperator]
     */
    override fun toFlow(): Flow<Tuple> {
        with(this@FilterOperator.context.bindings) {
            this@FilterOperator.predicate.prepare()
            return this@FilterOperator.parent.toFlow().filter { record ->
                with(record) {
                    this@FilterOperator.predicate.isMatch()
                }
            }
        }
    }
}