package org.vitrivr.cottontail.dbms.queries.operators.logical.projection

import org.vitrivr.cottontail.core.database.ColumnDef
import org.vitrivr.cottontail.core.queries.Digest
import org.vitrivr.cottontail.core.queries.binding.Binding
import org.vitrivr.cottontail.dbms.queries.operators.basics.OperatorNode
import org.vitrivr.cottontail.dbms.queries.operators.basics.UnaryLogicalOperatorNode
import org.vitrivr.cottontail.dbms.queries.operators.physical.projection.CountProjectionPhysicalOperatorNode
import org.vitrivr.cottontail.dbms.queries.projection.Projection

/**
 * A [UnaryLogicalOperatorNode] that represents a projection operation involving aggregate functions such as [Projection.COUNT].
 *
 * @author Ralph Gasser
 * @version 3.0.0
 */
class CountProjectionLogicalOperatorNode(input: Logical, val out: Binding.Column) : AbstractProjectionLogicalOperatorOperator(input, Projection.COUNT) {

    /** The [ColumnDef] generated by this [CountProjectionLogicalOperatorNode]. */
    override val columns: List<Binding.Column>
        get() = listOf(this.out)

    /**
     * Creates a copy of this [CountProjectionLogicalOperatorNode].
     *
     * @param input The new input [OperatorNode.Logical]
     * @return Copy of this [CountProjectionLogicalOperatorNode]
     */
    override fun copyWithNewInput(vararg input: Logical): CountProjectionLogicalOperatorNode {
        require(input.size == 1) { "The input arity for CountProjectionLogicalOperatorNode.copyWithNewInput() must be 1 but is ${input.size}. This is a programmer's error!"}
        return CountProjectionLogicalOperatorNode(input = input[0], out = this.out)
    }

    /**
     * Returns a [CountProjectionPhysicalOperatorNode] representation of this [CountProjectionLogicalOperatorNode]
     *
     * @return [CountProjectionPhysicalOperatorNode]
     */
    override fun implement(): Physical = CountProjectionPhysicalOperatorNode(this.input.implement(), this.out)

    /**
     * Generates and returns a [Digest] for this [CountProjectionPhysicalOperatorNode].
     *
     * @return [Digest]
     */
    override fun digest(): Digest {
        var result = Projection.COUNT.hashCode().toLong()
        result += 33L * result + this.out.hashCode()
        return result
    }
}