package org.vitrivr.cottontail.dbms.queries.operators.physical.projection

import org.vitrivr.cottontail.core.database.ColumnDef
import org.vitrivr.cottontail.core.queries.Digest
import org.vitrivr.cottontail.core.queries.binding.Binding
import org.vitrivr.cottontail.core.queries.binding.BindingContext
import org.vitrivr.cottontail.core.queries.nodes.traits.NotPartitionableTrait
import org.vitrivr.cottontail.core.queries.nodes.traits.Trait
import org.vitrivr.cottontail.core.queries.nodes.traits.TraitType
import org.vitrivr.cottontail.core.queries.planning.cost.Cost
import org.vitrivr.cottontail.core.tuple.Tuple
import org.vitrivr.cottontail.dbms.execution.operators.projection.CountProjectionOperator
import org.vitrivr.cottontail.dbms.queries.context.QueryContext
import org.vitrivr.cottontail.dbms.queries.operators.basics.OperatorNode
import org.vitrivr.cottontail.dbms.queries.operators.basics.UnaryPhysicalOperatorNode
import org.vitrivr.cottontail.dbms.queries.operators.logical.projection.CountProjectionLogicalOperatorNode
import org.vitrivr.cottontail.dbms.queries.projection.Projection

/**
 * A [UnaryPhysicalOperatorNode] that represents a projection operation involving aggregate functions such as [Projection.COUNT].
 *
 * @author Ralph Gasser
 * @version 2.6.0
 */
class CountProjectionPhysicalOperatorNode(input: Physical, val out: Binding.Column) : UnaryPhysicalOperatorNode(input) {
    /** The name of this [CountProjectionPhysicalOperatorNode]. */
    override val name: String
        get() = Projection.COUNT.label()

    /** The [ColumnDef] generated by this [CountProjectionLogicalOperatorNode]. */
    override val columns: List<Binding.Column> = listOf(this.out)

    /** The output size of this [CountProjectionPhysicalOperatorNode] is always one. */
    context(BindingContext, Tuple)
    override val outputSize: Long
        get() = 1L

    /** The [Cost] of a [CountProjectionPhysicalOperatorNode]. */
    context(BindingContext, Tuple)
    override val cost: Cost
        get() = Cost.MEMORY_ACCESS * this.input.outputSize

    /** The [CountProjectionPhysicalOperatorNode] cannot be partitioned. */
    override val traits: Map<TraitType<*>, Trait> by lazy {
        this.input.traits + (NotPartitionableTrait to NotPartitionableTrait)
    }

    /**
     * Creates and returns a copy of this [CountProjectionPhysicalOperatorNode] using the given parents as input.
     *
     * @param input The [OperatorNode.Physical]s that act as input.
     * @return Copy of this [CountProjectionPhysicalOperatorNode].
     */
    override fun copyWithNewInput(vararg input: Physical): CountProjectionPhysicalOperatorNode {
        require(input.size == 1) { "The input arity for SelectProjectionPhysicalOperatorNode.copyWithNewInput() must be 1 but is ${input.size}. This is a programmer's error!"}
        return CountProjectionPhysicalOperatorNode(input = input[0], out = this.out)
    }

    /**
     * Converts this [CountProjectionPhysicalOperatorNode] to a [CountProjectionOperator].
     *
     * @param ctx The [QueryContext] used for the conversion (e.g. late binding).
     */
    override fun toOperator(ctx: QueryContext) = CountProjectionOperator(this.input.toOperator(ctx), ctx)

    /** Generates and returns a [String] representation of this [CountProjectionPhysicalOperatorNode]. */
    override fun toString() = "${super.toString()}[${this.columns.joinToString(",") { it.column.name.toString() }}]"

    /**
     * Generates and returns a [Digest] for this [CountProjectionPhysicalOperatorNode].
     *
     * @return [Digest]
     */
    override fun digest(): Digest {
        var result = Projection.COUNT.hashCode().toLong()
        result += 31L * result + this.out.hashCode()
        return result
    }
}