/*
 * Decompiled with CFR 0.152.
 */
package org.vrspace.server.core;

import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.neo4j.core.Neo4jClient;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.vrspace.server.core.NotFoundException;
import org.vrspace.server.core.VRObjectRepository;
import org.vrspace.server.core.WorldManager;
import org.vrspace.server.dto.GroupEvent;
import org.vrspace.server.obj.Client;
import org.vrspace.server.obj.GroupMember;
import org.vrspace.server.obj.GroupMessage;
import org.vrspace.server.obj.Ownership;
import org.vrspace.server.obj.UserGroup;

@Component
@Lazy
public class GroupManager {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(GroupManager.class);
    @Autowired
    private VRObjectRepository db;
    @Autowired
    private WorldManager worldManager;
    @Autowired
    private Neo4jClient neo4jClient;

    @Transactional
    public List<UserGroup> listGroups(Client client) {
        return this.db.listUserGroups(client.getId());
    }

    @Transactional
    public List<UserGroup> listOwnedGroups(Client client) {
        return this.db.listOwnedGroups(client.getId());
    }

    @Transactional
    public UserGroup createGroup(Client client, UserGroup group) {
        if (this.db.findGroup((long)client.getId(), group.getName()).isPresent()) {
            throw new IllegalArgumentException("Client already belongs to group " + group.getName());
        }
        group = (UserGroup)this.db.save(group);
        this.addOwner(group, client);
        this.addMember(group, client);
        return group;
    }

    @Transactional
    public UserGroup updateGroup(Client client, UserGroup group) {
        if (this.db.getOwnership(client.getId(), group.getId()) == null) {
            throw new SecurityException("Not an owner");
        }
        group = (UserGroup)this.db.save(group);
        return group;
    }

    @Transactional
    public void deleteGroup(Client client, UserGroup group) {
        if (this.db.getOwnership(client.getId(), group.getId()) == null) {
            throw new SecurityException("Not an owner");
        }
        this.db.listGroupMembers(group.getId()).forEach(groupMember -> this.db.delete(groupMember));
        this.db.getOwnersOf(group.getId()).forEach(ownership -> this.db.delete(ownership));
    }

    @Transactional
    public List<Client> show(UserGroup group) {
        return this.db.listGroupClients(group.getId());
    }

    @Transactional
    public void invite(UserGroup group, Long memberId, Client owner) {
        this.invite(group, this.getClient(memberId), owner);
    }

    @Transactional
    public void invite(UserGroup group, Client member, Client owner) {
        if (this.db.findGroupMember(group.getId(), member.getId()).isPresent()) {
            throw new IllegalArgumentException("Client " + member.getId() + " is already member of group " + group.getId());
        }
        if (group.isPrivate() && (owner == null || this.db.findOwnership(owner.getId(), group.getId()).isEmpty())) {
            throw new IllegalArgumentException("Only group owners can invite members");
        }
        GroupMember gm = new GroupMember(group, member).invite(owner);
        this.save(gm);
        Client cachedClient = this.getCachedClient(member);
        if (cachedClient == null) {
            log.debug("Invite for offline client:" + member);
        } else {
            cachedClient.sendMessage(GroupEvent.invite(gm));
        }
    }

    @Transactional
    public void accept(UserGroup group, Client member) {
        Optional<GroupMember> existingMember = this.db.findGroupMember(group.getId(), member.getId());
        if (existingMember.isEmpty() || existingMember.get().getPendingInvite() == null) {
            throw new IllegalArgumentException("Not invited client: " + member.getId());
        }
        GroupMember updatedMember = ((GroupMember)this.db.get(existingMember)).accept();
        this.save(updatedMember);
    }

    @Transactional
    public void ask(UserGroup group, Client member) {
        if (this.db.findGroupMember(group.getId(), member.getId()).isPresent()) {
            throw new IllegalArgumentException("Client " + member.getId() + " is already joining group " + group.getId());
        }
        GroupMember gm = new GroupMember(group, member).request();
        this.save(gm);
    }

    @Transactional
    public void allow(UserGroup group, Long memberId, Client owner) {
        this.allow(group, owner, this.getClient(memberId));
    }

    @Transactional
    public void allow(UserGroup group, Client member, Client owner) {
        GroupMember gm;
        if (this.db.findOwnership(owner.getId(), group.getId()).isEmpty()) {
            throw new IllegalArgumentException("Only group owners can allow members");
        }
        Optional<GroupMember> invited = this.db.findGroupMember(group.getId(), member.getId());
        if (invited.isPresent()) {
            gm = (GroupMember)this.db.get(invited);
            if (gm.getPendingRequest() == null) {
                throw new IllegalArgumentException("No pending request for client: " + member.getId());
            }
        } else {
            throw new IllegalArgumentException("Not invited client: " + member.getId());
        }
        this.save(gm.allow(owner));
    }

    @Transactional
    public void join(UserGroup group, Client member) {
        if (group.isPrivate()) {
            throw new IllegalArgumentException("Ask to join a private group");
        }
        this.addMember(group, member);
    }

    @Transactional
    public void leave(UserGroup group, Client member) {
        if (this.db.findOwnership(member.getId(), group.getId()).isPresent()) {
            throw new IllegalArgumentException("Group owners can not leave their groups");
        }
        this.removeMember(group, member);
    }

    @Transactional
    public void kick(UserGroup group, long memberId, Client owner) {
        Optional<GroupMember> member = this.db.findGroupMember(group.getId(), memberId);
        if (member.isEmpty()) {
            throw new IllegalArgumentException("Group does not contain client: " + memberId);
        }
        this.kick(group, member.get().getClient(), owner);
    }

    @Transactional
    public void kick(UserGroup group, Client member, Client owner) {
        if (!group.isPrivate()) {
            throw new IllegalArgumentException("Can't kick members from public groups");
        }
        if (this.db.findOwnership(owner.getId(), group.getId()).isEmpty()) {
            throw new SecurityException("Only group owners can kick members");
        }
        this.removeMember(group, member);
    }

    @Transactional
    public List<GroupMember> pendingRequests(UserGroup group, Client member) {
        if (this.db.findOwnership(member.getId(), group.getId()).isEmpty()) {
            throw new SecurityException("Only group owners can list pending requets");
        }
        return this.db.listPendingRequests(group.getId());
    }

    @Transactional
    public List<GroupMember> pendingInvitations(Client member) {
        return this.db.listPendingInvitations(member.getId());
    }

    private void addMember(UserGroup group, Client member) {
        GroupMember gm = new GroupMember(group, member);
        this.save(gm);
    }

    private void removeMember(UserGroup group, Client member) {
        this.db.findGroupMember(group.getId(), member.getId()).ifPresent(groupMember -> this.db.delete(groupMember));
    }

    @Transactional
    public void addOwner(UserGroup group, Client owner) {
        Ownership ownership = new Ownership(owner, group);
        this.db.save(ownership);
    }

    @Transactional
    public void removeOwner(UserGroup group, Client owner) {
        this.db.findOwnership(owner.getId(), group.getId()).ifPresent(ownership -> this.db.delete(ownership));
    }

    @Transactional
    public void write(Client sender, UserGroup group, String text) {
        if (this.db.findGroupMember(group.getId(), sender.getId()).isEmpty()) {
            throw new SecurityException("Only members can post to groups");
        }
        GroupMessage message = (GroupMessage)this.db.save(new GroupMessage(sender, group, text, Instant.now()));
        this.db.listGroupClients(group.getId()).forEach(client -> {
            Client cachedClient = this.getCachedClient((Client)client);
            if (cachedClient == null) {
                log.debug("Message for offline client:" + client);
            } else {
                cachedClient.sendMessage(GroupEvent.message(message));
            }
        });
    }

    @Transactional
    public UserGroup getGroup(Client client, String name) {
        Optional<UserGroup> group = this.db.findGroup((long)client.getId(), name);
        if (group.isEmpty()) {
            throw new NotFoundException("Non-existing group: " + name + " clientId:" + client.getId());
        }
        return group.get();
    }

    @Transactional
    public UserGroup getGroup(Client client, long groupId) {
        Optional<UserGroup> group = this.db.findGroup((long)client.getId(), groupId);
        if (group.isEmpty()) {
            throw new NotFoundException("Non-existing group: " + groupId + " clientId:" + client.getId());
        }
        return group.get();
    }

    @Transactional
    public UserGroup getGroup(long groupId) {
        UserGroup group = (UserGroup)this.db.get(UserGroup.class, groupId);
        if (group == null) {
            throw new NotFoundException("Non-existing group: " + groupId);
        }
        return group;
    }

    @Transactional
    public List<UserGroup> unreadGroups(Client client) {
        List<GroupMember> groups = this.db.listGroupMemberships(client.getId());
        groups.forEach(gm -> {
            UserGroup group = gm.getGroup();
            int unread = this.db.unreadMessageCount(group.getId(), gm.getLastRead());
            group.setUnread(unread);
        });
        return groups.stream().map(gm -> gm.getGroup()).filter(g -> g.getUnread() > 0).collect(Collectors.toList());
    }

    @Transactional
    public List<GroupMessage> unreadMessages(Client client, UserGroup group) {
        Instant now = Instant.now();
        Optional<GroupMember> gm = this.db.findGroupMember(group.getId(), client.getId());
        if (gm.isEmpty()) {
            throw new NotFoundException("Not a member group: " + group.getId() + " clientId:" + client.getId());
        }
        GroupMember member = gm.get();
        Instant lastRead = member.getLastRead();
        member.setLastRead(now);
        this.save(member);
        return this.db.messagesSince(group.getId(), lastRead);
    }

    @Transactional
    public List<Client> listOwners(UserGroup group) {
        return this.db.getOwners(group.getId()).stream().map(ownership -> ownership.getOwner()).toList();
    }

    private Client getClient(long clientId) {
        Client client = this.db.getClient(clientId);
        if (client == null) {
            throw new NotFoundException("Non-existing client: " + clientId);
        }
        return client;
    }

    private Client getCachedClient(Client c) {
        Client cachedClient = (Client)this.worldManager.get(c.getObjectId());
        if (cachedClient != null && !cachedClient.isActive()) {
            log.error("Client is not active " + c);
        }
        return cachedClient;
    }

    private void save(GroupMember gm) {
        this.db.save(gm);
    }
}

