package org.vrspace.server.obj;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.data.annotation.Transient;
import org.springframework.data.neo4j.core.schema.Node;
import org.vrspace.server.dto.VREvent;
import org.vrspace.server.types.ID;
import org.vrspace.server.types.Owned;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.extern.slf4j.Slf4j;

/**
 * A game in a world. Typically does not have mesh, but does have a script, that
 * implements client-side game logic. It's an active object - all observers
 * receive events generated by the game.
 * 
 * Game object is Owned - only the Client that created it can change properties.
 * 
 * Games can be persistent, or added by user(s). Depending on the game and world
 * type, users can opt to join, or may join as soon as they enter the world.
 * 
 * Server-side Game object maintains the list of currently active players, and
 * provides basic functions to Clients, like joining and leaving the game. Note
 * that a game members are Clients rather than Users - robots can play games,
 * why not. The game listens to all events generated by players.
 */
@Data
@EqualsAndHashCode(callSuper = true, onlyExplicitlyIncluded = true)
@Node
@Owned
@Slf4j
public class Game extends VRObject {
  /** Name of the game, supposed to be unique within the world */
  private String name;
  /** Current number of players */
  private int numberOfPlayers = 0;
  private String status;
  @Transient
  private transient Set<ID> players = ConcurrentHashMap.newKeySet();

  /**
   * A client wants to join the game.
   * 
   * @param client
   */
  public void join(Client client) {
    client.addListener(this);
    players.add(client.getObjectId());
    event("joined", client);
  }

  /**
   * A client left the game
   * 
   * @param client
   */
  public void quit(Client client) {
    players.remove(client.getObjectId());
    client.removeListener(this);
    event("quit", client);
  }

  public void start(Client client) {
    // TODO check ownership
    event("start", client);
  }

  public void end(Client client) {
    // TODO check ownership
    event("end", client);
  }

  private void event(String type, Client client) {
    this.numberOfPlayers = players.size();
    VREvent event = new VREvent(this);
    Map<String, Object> changes = new HashMap<>();
    changes.put(type, client.getObjectId());
    event.setChanges(changes);
    this.notifyListeners(event);
    log.debug("Client " + client + " " + type + " " + this);
  }

  /**
   * Process a client (player) event. This implementation does nothing, only
   * removes the client from the player list on disconnect. CHECKME should we do
   * that, what if they reconnect?
   */
  @Override
  public void processEvent(VREvent event) {
    if (!event.getSource().isActive() && event.getSource() instanceof Client) {
      // stop listening to inactive objects (disconnected clients)
      this.quit((Client) event.getSource());
    }
  }

}
