/*
 * Copyright 2008-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 
package org.wamblee.glassfish.auth.cache.api;

import java.util.List;
import java.util.Properties;

/**
 * <p>
 * Interface for a cached of the authentication information.
 * </p>
 * 
 * <p>
 * Implementations should provide a public constructor with 
 * one argument of type {@link Properties}. The property object
 * passed at construction contains the same properties that were
 * passed to flexible JDBC realm. This allows users of a cache to
 * configure a cache through the standard Glassfish security realm
 * configuration.
 * </p>
 * 
 * <p>
 * Implementations should implemnet {@link #toString()} to print a meaningful
 * description of the cache for logging. 
 * </p>
 * 
 * <p>
 * This interface is very similar to the <code>AuthenticationRepository</code>
 * but is slightly different because it gets cached data and does not 'find' it.
 * Also the contract differs in that it may return null and it has 
 * method for putting things into it.
 * </p> 
 * 
 * @author Erik Brakkee
 */
public interface AuthenticationCache {

    /**
     * Gets the cached password for a given user. 
     * @param aUsername Username. 
     * @return Password or null if no cached value is available.  
     */
    String getPassword(String aUsername);
    
    /**
     * Sets the cached password for a user. 
     * @param aUserName Username. 
     * @param aPassword Password. 
     */
    void setPassword(String aUserName, String aPassword);
    
    /**
     * Gets the groups a user belongs to from the cache.  
     * @param aUsername User. 
     * @return Groups. Returns a (possibly) empty list of cached groups or null of no 
     * cached groups are found.    
     */
    List<String> getGroups(String aUsername);
    
    /**
     * Sets the groups for a given user. 
     * @param aUsername User name. 
     * @param aGroups Groups (must be non-null).
     */
    void setGroups(String aUsername, List<String> aGroups); 
    
    /**
     * Gets a cached seed for a given username based on the configured sql seed query.
     * 
     * @param aUsername
     *            Username.
     * @return The seed found or the empty string if no seeds or multiple seeds
     *         were found. Returns null in case no cached seed is available. 
     */
    String getSeed(String aUsername);
    
    /**
     * Sets the seed. 
     * @param aUsername Username. 
     * @param aSeed Seed (may not be null). 
     */
    void setSeed(String aUsername, String aSeed); 
}
