/*
 * Copyright 2008-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.glassfish.auth.cache.impl;

import java.lang.management.ManagementFactory;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.ObjectName;

import org.wamblee.glassfish.auth.cache.api.AuthenticationCache;

/**
 * <p>
 * Simple cache that has entries that expire after a certain amount of time.
 * Also, the cache has a JMX interface {@link SimpleExpiryCacheManagementMBean} 
 * by which the entries for one or all users
 * can be expired. The cache does not do an automatic cleanup of expired items
 * although it is possible to trigger cleanup through JMX.
 * </p>
 *
 * <p>
 * The cache exposes an MBean for remote management and for managing it from an
 * application. This access is unsecured so anyone can in principle clean the 
 * cache. The main intention of the JMX MBean is to use it for invalidating the cache
 * for a specific user in case his authentication information has changed.
 * The MBean is exposed at the domain specified by {@link #JMX_DOMAIN} with
 * a property specified by {@link #JMX_REALM_PROPERTY} with the value of the realm.
 * </p>
 * 
 * <p>To clear a specific user, proceed as follows: </p>
 * <pre>
 *    ObjectName objectName = new ObjectName("org.wamblee.glassfish.auth.FlexibleJdbcRealm", "realm", REALM_NAME);
 *    ManagementFactory.getPlatformMBeanServer().invoke(objectName,
 *           "clearUser", new Object[] { "username" },
 *           new String[] { String.class.getName() });
 * </pre>
 * 
 * @author Erik Brakkee
 * 
 */
public class SimpleExpiryCache implements AuthenticationCache {
        
    private static final String JMX_REALM_PROPERTY = "realm";

    /**
     * Class representing cached information about a user. In practic
     * concurrency will not be an issue for this class. Nevertheless we guard it
     * with synchronized just in case.
     * 
     * @author Erik Brakkee
     * 
     */
    public static class UserEntry {
        private long expiryTime;
        private String password;
        private List<String> groups;
        private String seed;

        public UserEntry(long aExpiryTime) {
            expiryTime = aExpiryTime;
        }

        public synchronized List<String> getGroups() {
            return groups;
        }

        public synchronized String getPassword() {
            return password;
        }

        public synchronized String getSeed() {
            return seed;
        }

        public synchronized void setGroups(List<String> aGroups) {
            groups = aGroups;
        }

        public synchronized void setPassword(String aPassword) {
            password = aPassword;
        }

        public synchronized void setSeed(String aSeed) {
            seed = aSeed;
        }

        public long getExpiryTime() {
            return expiryTime;
        }
    }

    public static interface Clock {
        long currentTimeMillis();
    }

    public static class SystemClock implements Clock {
        @Override
        public long currentTimeMillis() {
            return System.currentTimeMillis();
        }
    }

    private static final Logger LOGGER = Logger
        .getLogger(SimpleExpiryCache.class.getName());

    /**
     * Name of the property that defines the timeout in seconds for
     * caching. In case the value is &lt; 0, then entries in the cache will 
     * never expire. 
     */
    public static final String PROP_EXPIRY_TIME_SECONDS = "cache.timeoutSeconds";

    public static final String JMX_DOMAIN = "org.wamblee.glassfish.auth.FlexibleJdbcRealm";

    public static final String PROP_REALM_NAME = "jaas.context";

    /**
     * Default expiry time in seconds.
     */
    public static final long DEFAULT_EXPIRY_TIME_SECONDS = 60;

    private Map<String, UserEntry> entries;

    private Clock clock = new SystemClock();
    private long expiryTimeMillis = DEFAULT_EXPIRY_TIME_SECONDS*1000;

    /**
     * Constructs the cache.
     * 
     * @param aProperties
     *            Properties to configure the cache with.
     */
    public SimpleExpiryCache(Properties aProperties) {
        entries = new ConcurrentHashMap<String, UserEntry>();
        String timeoutString = aProperties.getProperty(PROP_EXPIRY_TIME_SECONDS);
        if (timeoutString != null) {
            try {
                // parsing as int because we want to make sure the config param fits in the integer range.
                expiryTimeMillis = ((long)Integer.parseInt(timeoutString))*1000L;
            } catch (NumberFormatException e) {
                throw new IllegalArgumentException(
                    "Error parsing timeout value from property " +
                        PROP_EXPIRY_TIME_SECONDS + " with value '" +
                        timeoutString, e);
            }
        }
        
        LOGGER.info("Cache expiry time is " + (expiryTimeMillis/1000) + " seconds");

        String realm = aProperties.getProperty(PROP_REALM_NAME);

        SimpleExpiryCacheManagementMBean mbean = new SimpleExpiryCacheManagement(this);
        register(mbean, realm);
    }
    
    Map<String, UserEntry> getEntries() {
        return entries;
    }

    private void register(SimpleExpiryCacheManagementMBean aMbean, String aRealm) {
        MBeanServer server = ManagementFactory.getPlatformMBeanServer();
        try {
            ObjectName name = new ObjectName(JMX_DOMAIN, JMX_REALM_PROPERTY, aRealm);
            try {
                server.unregisterMBean(name);
            } catch (JMException e) {
                // ignore. We need to do this because realms can be deleted at
                // runtime but there
                // is no callback for this so we cannot unregister. Therefore we
                // try
                // to unregister
                // when the realm is started.
            }
            server.registerMBean(aMbean, name);
            LOGGER.info("Registered MBean at '" + name + "'");
        } catch (JMException e) {
            LOGGER
                .log(Level.WARNING, "Could not register MBean for SimpleExpiryCache for realm '" + aRealm + "'", e);
        }
    }

    protected void clearExpired() {
        Set<String> users = new HashSet<String>(entries.keySet());
        for (String user : users) {
            getUserEntry(user);
        }
    }

    /**
     * Constructor for unit test.
     * 
     * @param aProperties
     *            Properties.
     * @param aClock
     *            Clock.
     */
    public SimpleExpiryCache(Properties aProperties, Clock aClock) {
        this(aProperties);
        clock = aClock;
    }

    public int getExpiryTimeSeconds() {
        return (int)(expiryTimeMillis/1000);
    }

    public int size() {
        return entries.size();
    }

    @Override
    public String toString() {
        return getClass().getName() + "(expiryTimeMillis = " + expiryTimeMillis + " seconds)";
    }

    @Override
    public List<String> getGroups(String aUsername) {
        UserEntry entry = getUserEntry(aUsername);
        return entry == null ? null : entry.getGroups();
    }

    @Override
    public String getPassword(String aUsername) {
        UserEntry entry = getUserEntry(aUsername);
        return entry == null ? null : entry.getPassword();
    }

    @Override
    public String getSeed(String aUsername) {
        UserEntry entry = getUserEntry(aUsername);
        return entry == null ? null : entry.getSeed();
    }

    @Override
    public void setGroups(String aUsername, List<String> aGroups) {
        ensureUserEntry(aUsername).setGroups(aGroups);
    }

    @Override
    public void setPassword(String aUserName, String aPassword) {
        ensureUserEntry(aUserName).setPassword(aPassword);
    }

    @Override
    public void setSeed(String aUsername, String aSeed) {
        ensureUserEntry(aUsername).setSeed(aSeed);
    }

    /**
     * Returns cached user entry, expiring the user if needed.
     * 
     * @param aUser
     *            User.
     * @return Cached user entry of null if not or no longer cached.
     */
    private UserEntry getUserEntry(String aUser) {
        UserEntry user = entries.get(aUser);      
        if (user == null) {
            entries.remove(aUser);
            return null;
        }
        long t = clock.currentTimeMillis();
        if (user.getExpiryTime() < t) {
            entries.remove(aUser);
            return null;
        }
        return user;
    }

    private UserEntry ensureUserEntry(String aUser) {
        UserEntry entry = getUserEntry(aUser);
        if (entry == null) {
            long expiry = expiryTimeMillis < 0 ? Long.MAX_VALUE: clock.currentTimeMillis() + expiryTimeMillis;
            entry = new UserEntry(expiry);
            entries.put(aUser, entry);
        }
        return entry;
    }

}
