/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.safe.model.jpa;

import java.io.Serializable;
import java.util.List;

import javax.enterprise.context.ApplicationScoped;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;

import org.wamblee.inject.Injectable;
import org.wamblee.persistence.Detachable;
import org.wamblee.persistence.JpaDetachable;
import org.wamblee.safe.model.Account;
import org.wamblee.safe.model.Entry;
import org.wamblee.safe.model.Storage;

@ApplicationScoped
public class JpaStorage extends Injectable implements Storage, Serializable {

    private static final String DEFAULT_ACCOUNT = "default";
    private static final String ACCOUNT_NAME_PARAM = "name";

    private static final String FIND_ACCOUNT_QUERY = "select a from Account a where a.name = :" +
        ACCOUNT_NAME_PARAM;

    // Entry queries. Assuming there is only one account now so no need to join
    // with the account.

    private static final String FIND_ENTRIES_QUERY = "select e from Entry e order by e.name";
    private static final String COUNT_ENTRIES_QUERY = "select count(e) from Entry e";

    // This somehow got broken in glassfish v4, but using the entity manager
    // directly through @PersistenceContext also works.....
    // @ApplicationDatabase
    // @Inject
    @PersistenceContext
    private EntityManager em;

    public JpaStorage() {
        // Empty
    }

    public <T> Detachable<T> createDetachable(final T aObject) {
        return new JpaDetachable<T>(em, aObject);
    }

    @Override
    public void flush() {
        em.flush();
    }

    @Override
    public Detachable<Account> getDefaultAccount() {
        TypedQuery<Account> query = em.createQuery(FIND_ACCOUNT_QUERY,
            Account.class).setParameter(ACCOUNT_NAME_PARAM, DEFAULT_ACCOUNT);
        Account account = null;
        try {
            account = query.getSingleResult();
        } catch (NoResultException e) {
            account = new Account(DEFAULT_ACCOUNT);
            em.persist(account);
        }
        return createDetachable(account);
    }

    @Override
    public List<Entry> findEntries(int aStart, int aCount) {
        TypedQuery<Entry> query = em.createQuery(FIND_ENTRIES_QUERY,
            Entry.class);
        query.setFirstResult(aStart);
        query.setMaxResults(aCount);
        return query.getResultList();
    }

    @Override
    public int countEntries() {
        TypedQuery<Long> query = em
            .createQuery(COUNT_ENTRIES_QUERY, Long.class);
        return query.getSingleResult().intValue();
    }
}
