/*
 * Copyright 2005-2011 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.safe.security;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

public class AuthenticationFilter implements Filter {

    private static final String LOGINPAGE = "loginpage";
    private static final String REQUIRED_ROLE = "role";
    private static final String RESOURCES = "resources";

    private String loginPage;
    private String role;
    private String resources;

    public AuthenticationFilter() {
        // Empty.
    }

    @Override
    public void init(FilterConfig aFilterConfig) throws ServletException {
        loginPage = aFilterConfig.getInitParameter(LOGINPAGE);
        if (loginPage == null) {
            throw new ServletException("No login page defined! Must specify '" +
                LOGINPAGE + "' filter init parameter.");
        }
        role = aFilterConfig.getInitParameter(REQUIRED_ROLE);
        if (role == null) {
            throw new ServletException("No role name defined! Must specify '" +
                REQUIRED_ROLE + "' filter init parameter.");
        }
        resources = aFilterConfig.getInitParameter(RESOURCES);
    }

    @Override
    public void doFilter(ServletRequest aRequest, ServletResponse aResponse,
        FilterChain aChain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) aRequest;
        String fullPath = request.getRequestURI();
        String contextPath = request.getContextPath();
        String relpath = null;
        if ( fullPath.startsWith(contextPath)) { 
            relpath = fullPath.substring(contextPath.length());
        }
        
        if (request.isUserInRole(role) ||
            (resources != null && relpath!= null && relpath.startsWith(resources))) {
            aChain.doFilter(aRequest, aResponse);
        } else {
            request.getSession().invalidate();
            request.getRequestDispatcher(loginPage)
                .forward(aRequest, aResponse);
        }
    }

    @Override
    public void destroy() {
        // Empty.
    }

}
