/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 
package org.wamblee.safe.wicket;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.List;

import javax.inject.Inject;

import org.apache.wicket.PageParameters;
import org.apache.wicket.markup.html.WebResource;
import org.apache.wicket.markup.html.form.Form;
import org.apache.wicket.markup.html.form.upload.FileUpload;
import org.apache.wicket.markup.html.form.upload.FileUploadField;
import org.apache.wicket.markup.html.link.ResourceLink;
import org.apache.wicket.protocol.http.WebResponse;
import org.apache.wicket.util.lang.Bytes;
import org.apache.wicket.util.resource.IResourceStream;
import org.apache.wicket.util.resource.StringResourceStream;
import org.wamblee.safe.model.Entry;
import org.wamblee.safe.model.Storage;
import org.wamblee.safe.xml.XMLMapper;

/**
 * Admin page. 
 */
public class AdminPage extends BasePage {

    private static final long serialVersionUID = 1L;
    

    @Inject
    private Storage storage;

    @Inject
    private XMLMapper mapper;

    private FileUploadField fileUpload;

    /**
     * Constructor that is invoked when page is invoked without a session.
     * 
     * @param parameters
     *            Page parameters
     */
    public AdminPage(final PageParameters parameters) throws Exception {
        super();

        WebResource resource = new WebResource() {
            @Override
            public IResourceStream getResourceStream() {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                List<Entry> entries = storage.findEntries(0, 1000000);
                mapper.encode(bos, entries);
                return new StringResourceStream(bos.toString(), "text/xml");
            }

            @Override
            protected void setHeaders(WebResponse aResponse) {
                super.setHeaders(aResponse);
                aResponse.setAttachmentHeader("export.xml");
            }
        };
        resource.setCacheable(false);
        add(new ResourceLink("export", resource));

        Form form = new Form("uploadForm") {
            protected void onSubmit() {
                try {
                    FileUpload upload = fileUpload.getFileUpload();
                    if (upload != null) {
                        InputStream is = upload.getInputStream();
                        List<Entry> entries = mapper.parse(is);
                        storage.getDefaultAccount().get().getEntries().addAll(
                            entries);
                        getSession().info(
                            "Uploaded " + entries.size() + " entries");
                        setResponsePage(HomePage.class);
                    } else {
                        getSession().info("Nothing uploaded");
                    }
                } catch (Exception e) {
                    getSession().info(
                        "Exception occured: " + e.getClass() + ": " +
                            e.getMessage());
                }
            }
        };
        form.setMultiPart(true);
        form.setMaxSize(Bytes.megabytes(5.0));
        form.add(fileUpload = new FileUploadField("upload"));

        add(form);
        
        

    }
}
