/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 
package org.wamblee.safe.wicket;

import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import org.apache.wicket.markup.html.CSSPackageResource;
import org.apache.wicket.markup.html.WebPage;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.link.Link;
import org.apache.wicket.markup.html.panel.FeedbackPanel;
import org.apache.wicket.model.IModel;
import org.apache.wicket.protocol.http.WebRequestCycle;
import org.wamblee.safe.model.Storage;
import org.wamblee.wicket.behavior.TitleAttributeTooltipBehavior;
import org.wamblee.wicket.css.ResetCssBehavior;
import org.wamblee.wicket.page.ExpireBehavior;
import org.wamblee.wicket.page.FlushEntityManagerBehavior;
import org.wamblee.wicket.page.WebApplicationBasePage;

public class BasePage extends WebApplicationBasePage {
    
    @Inject
    private Storage storage;
    
    @PersistenceContext
    private EntityManager entityManager;
    
    private boolean isExpired = false; 

    public BasePage() {
        this(null);
    }

    public BasePage(IModel aModel) {
        super(aModel);
        add(new ResetCssBehavior());
        add(new TitleAttributeTooltipBehavior());
        add(CSSPackageResource.getHeaderContribution(BasePage.class, "safe.css"));
        addBehavior(new FlushEntityManagerBehavior() {            
            @Override
            public EntityManager getEntityManager() {
                return entityManager;
            }
        });
        disableCaching();
       
        add(new FeedbackPanel("feedback"));
        add(new Label("title", getTitle()));
        
        addBehavior(new ExpireBehavior() { 
            @Override
            protected boolean isExpired(WebPage aPage) {
                return isExpired;
            }
        }); 
        
        add(new Link("logout") { 
            @Override
            public void onClick() {
               getRequestCycle().getSession().invalidate();
            }
        });
     }
    
    
    public void setExpired(boolean aExpired) { 
        isExpired = aExpired;
    }
   
    
    private String getTitle() { 
        String name = getClass().getSimpleName();
        name = name.replaceAll("([A-Z]+)([A-Z][a-z])", "$1 $2");
        name = name.replaceAll("[A-Z]+", " $0");
        name = name.replaceAll("^ ", "");
        name = name.replaceAll("  ", " ");
        return name;
    }
}
