/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.safe.wicket;

import javax.inject.Inject;

import org.apache.wicket.markup.html.WebPage;
import org.apache.wicket.model.IModel;
import org.apache.wicket.model.Model;
import org.wamblee.safe.model.Entry;
import org.wamblee.safe.model.Storage;

/**
 * Homepage
 */
public class ViewEntryPage extends BasePage {

    private static final long serialVersionUID = 1L;

    @Inject
    private Storage storage;

    @Inject
    private DetachableEntityFactory factory;

    /**
     * Constructor that is invoked when page is invoked without a session.
     * 
     * @param parameters
     *            Page parameters
     */
    public ViewEntryPage(final WebPage aResponsePage, Entry aEntry) {
        super();
        IModel<Entry> entry;
        if (aEntry.getId() == 0) {
            entry = new Model(aEntry);
        } else { 
            // persistent object, refresh will show changes.
            entry = factory.detachable(aEntry);
        }
        add(new ViewEntryPanel("viewentryform", entry) {

            @Override
            protected void editPressed(Entry aViewedEntry) {
                setResponsePage(new EditEntryPage(aViewedEntry));
            }

            @Override
            protected void deletePressed(Entry aViewedEntry) {
                storage.getDefaultAccount().get().getEntries().remove(
                    aViewedEntry);
                storage.flush();
                getSession().info("Entry deleted");
                setExpired(true);
                setResponsePage(aResponsePage);
            }

            @Override
            protected void backPressed() {
                setResponsePage(aResponsePage);
            }
        });
    }

}
