/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.safe.model;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.OneToMany;
import javax.persistence.OrderColumn;
import javax.persistence.Table;

@Entity
@Table(name = "SAFE_ENTRY")
public class Entry implements Serializable {

    @Id
    @GeneratedValue(strategy = GenerationType.AUTO)
    private long id;

    @Column(nullable = false)
    private String name;

    @OneToMany(cascade = CascadeType.ALL, orphanRemoval = true, targetEntity = Field.class,
            fetch = FetchType.EAGER)
    @JoinTable(name = "SAFE_ENTRY_FIELD", joinColumns = {@JoinColumn(name = "ENTRY_ID")},
            inverseJoinColumns = {@JoinColumn(name = "FIELD_ID")})
    @OrderColumn(name = "FIELD_INDEX")
    private List<Field> fields;

    public Entry() {
        fields = new ArrayList<Field>();
    }

    public Entry(String aName) {
        this(aName, new ArrayList<Field>());
    }

    public Entry(String aName, List<Field> aFields) {
        name = aName;
        fields = aFields;
    }

    public void set(Entry aEntry) {
        name = aEntry.name;
        // Look for deleted fields.
        List<Field> removed = new ArrayList<Field>();
        for (Field field : fields) {
            if (aEntry.find(field.getName()) == null) {
                removed.add(field);
            }
        }
        // Remove deleted fields
        for (Field remove : removed) {
            fields.remove(remove);
        }
        // Modify the existing fields. 
        for (Field field : aEntry.getFields()) {
            Field myField = find(field.getName());
            if (myField == null) {
                myField = new Field(field.getName());
                fields.add(myField);
            }
            myField.set(field);
        }
    }

    public long getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public List<Field> getFields() {
        return fields;
    }

    public String getField(String aName) {
        return findAndCreate(aName).getValue();
    }

    public void setField(String aName, String aValue) {
        findAndCreate(aName).setValue(aValue);
    }

    private Field findAndCreate(String aName) {
        Field field = find(aName);
        if (field == null) {
            field = new Field(aName);
            fields.add(field);
        }
        return field;
    }

    private Field find(String aName) {
        for (Field field : fields) {
            if (field.getName().equals(aName)) {
                return field;
            }
        }
        return null;
    }

    @Override
    public boolean equals(Object aObj) {
        if (aObj == null) {
            return false;
        }
        if (!(aObj instanceof Entry)) {
            return false;
        }
        Entry entry = (Entry) aObj;
        return name.equals(entry.name) && id == entry.id;
    }

    @Override
    public int hashCode() {
        return name.hashCode();
    }

    @Override
    public String toString() {
        return "Entry(id = " + id + ", name = " + name + ")";
    }

    public boolean isDecryptedOk() {
        for (Field field : getFields()) {
            if (FieldEncryptor.CANNOT_DECRYPT_VALUE.equals(field.getValue())) {
                return false;
            }
        }
        return true;
    }
}
