/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.safe.model;

import java.io.Serializable;
import javax.inject.Inject;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.Table;

import org.wamblee.inject.InjectorBuilder;

@Entity
@Table(name = "SAFE_FIELD")
public class Field implements Serializable {

    @Id
    @GeneratedValue(strategy = GenerationType.AUTO)
    private long id;

    @Column(nullable = false)
    private String name;

    @Column(nullable = false, length = 30000)
    private String value;

    @Inject
    private transient FieldEncryptor fieldEncryptor;

    protected Field() {
        // Empty
    }

    public Field(String aName) {
        name = aName;
        value = null;
    }

    public Field(String aName, String aValue, FieldEncryptor aEncryptor) {
        name = aName;
        fieldEncryptor = aEncryptor;
        value = fieldEncryptor.encrypt(aValue);
    }

    public Field(String aName, String aValue) {
        name = aName;
        inject();
        value = fieldEncryptor.encrypt(aValue);
    }

    public void inject() {
        if (fieldEncryptor == null) {
            InjectorBuilder.getInjector().inject(this);
        }
    }

    public void set(Field aField) {
        name = aField.name;
        value = aField.value;
        fieldEncryptor = aField.fieldEncryptor;
    }

    public String getName() {
        return name;
    }

    public String getValue() {
        inject();
        return fieldEncryptor.decrypt(value);
    }

    public void setValue(String aValue) {
        inject();
        value = fieldEncryptor.encrypt(aValue);
    }

    @Override
    public boolean equals(Object aObj) {
        if (aObj == null) {
            return false;
        }
        if (!(aObj instanceof Field)) {
            return false;
        }
        Field field = (Field) aObj;
        return name.equals(field.name);
    }

    @Override
    public int hashCode() {
        return name.hashCode();
    }
}
