/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 
package org.wamblee.safe.xml;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import javax.enterprise.context.ApplicationScoped;

import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.DOMReader;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.XMLWriter;
import org.w3c.dom.Document;
import org.wamblee.safe.model.Entry;
import org.wamblee.safe.model.Field;
import org.wamblee.xml.DomUtils;

@ApplicationScoped
public class XMLMapperImpl implements XMLMapper {

    @Override
    public void encode(OutputStream aOs, List<Entry> aEntries) {
        org.dom4j.Document doc = DocumentHelper.createDocument();
        Element root = doc.addElement("entries");
        try {
            for (Entry entry: aEntries) { 
                Element entryElem = root.addElement("entry"); 
                Element nameElem = entryElem.addElement("name");
                nameElem.setText(entry.getName());
                for (Field field: entry.getFields()) { 
                    Element fieldElem = entryElem.addElement("field");
                    fieldElem.setText(field.getValue());
                    fieldElem.addAttribute("name", field.getName());
                }
            }
            OutputFormat format = OutputFormat.createPrettyPrint();
            format.setTrimText(false);
            XMLWriter writer = new XMLWriter(aOs, format);
            writer.write(doc);
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally { 
            try { 
                aOs.close();
            } catch (IOException e) { 
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public List<Entry> parse(InputStream aIs) {
        try {
            List<Entry> result = new ArrayList<Entry>();
            Document domDoc = DomUtils.read(aIs);
            org.dom4j.Document doc = new DOMReader().read(domDoc);
            Element root = doc.getRootElement();
            for (Element entryElem : (List<Element>) root.elements("entry")) {
                String name = entryElem.elementText("name");
                Entry entry = new Entry(name);
                for (Element fieldElem : (List<Element>) entryElem
                    .elements("field")) {
                    String fieldName = fieldElem.attributeValue("name");
                    String fieldValue = fieldElem.getText();
                    entry.getFields().add(new Field(fieldName, fieldValue));
                }
                result.add(entry);
            }
            return result;
        } catch (Exception e) {
            throw new RuntimeException("Count not parse xml document", e);
        } finally {
            try {
                aIs.close();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

}
