/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.gps.geometry;

import java.io.Serializable;

/**
 * Coordinates in some 3-dimensional coordinate system.  
 *
 * @author Erik Brakkee
 */
public class Coordinates implements Serializable {

    private double _x1; 
    private double _x2; 
    private double _x3; 
    
    /**
     * Constructs the coordinates. 
     * @param aX1 First coordinate. 
     * @param aX2 Second coordinate. 
     * @param aX3 Third coordinate. 
     */
    public Coordinates(double aX1, double aX2, double aX3) { 
        _x1 = aX1; 
        _x2 = aX2;
        _x3 = aX3; 
    }
    
    public double getX1() { 
        return _x1; 
    }
    
    public double getX2() { 
        return _x2; 
    }
    
    public double getX3() { 
        return _x3; 
    }
    
    public double getX(int i) { 
        switch (i) { 
        case 1: return _x1; 
        case 2: return _x2;
        case 3: return _x3;
        }
        throw new IllegalArgumentException("coordinate out of range " + i);
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "(" + getX1() + ", " + getX2() + ", " + getX3() + ")";
    }
    
    public Coordinates add(Coordinates aC) { 
        return new Coordinates(_x1 + aC._x1, _x2 + aC._x2, _x3 + aC._x3);
    }
    
    public Coordinates subtract(Coordinates aC) { 
        return new Coordinates(_x1 - aC._x1, _x2 - aC._x2, _x3 - aC._x3);
    }
  
    public double innerProduct(Coordinates aC) { 
        return _x1 * aC._x1 + _x2 * aC._x2 +  _x3 * aC._x3;
    }
    
    public Coordinates outerProduct(Coordinates aC) { 
        return new Coordinates(
           _x2*aC._x3 - _x3*aC._x2,
           -_x1*aC._x3 + _x3*aC._x1,
           _x1*aC._x2 - _x2*aC._x1
        );
    }
    
    public double norm() { 
        return Math.sqrt(innerProduct(this));
    }
    
    public Coordinates scale(double aMultiplier) { 
        return new Coordinates(_x1*aMultiplier, _x2*aMultiplier, _x3*aMultiplier);
    }
    
    public Coordinates normalize() {
        return scale(1.0/norm());
    }
}
