/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.gps.geometry;

import static java.lang.Math.PI;
import static java.lang.Math.cos;
import static java.lang.Math.sin;


/**
 * Represents the WGS 84 coordinate system for a GPS measurement identified by
 * <ul>
 *   <li> x1: latitude in degrees </li>
 *   <li> x2: longitude in degrees </li>
 *   <li> x3: elevation in meters </li>
 * </ul>
 * WGS84 models the earth as an ellipse.
 *
 * @author Erik Brakkee
 */
public class Wgs84CoordinateSystem implements CoordinateSystem {
    /*
     * Ellipsoide parameters, where the ellipsoide is defined by
     * 
     * (x^2 + y^2)/a^2 + z^2/b^2 = 1
     */
    
    /**
     * The radius of the ellipse at the equator
     */
    private static final double A = 6378137.000;
    
    /**
     * The distance of the North and South poles to the center of the ellipsoide. 
     */
    private static final double B = 6356752.314;  

 
    /* (non-Javadoc)
     * @see org.wamblee.gpx.CoordinateSystem#toReferenceSystem(org.wamblee.gpx.Coordinates)
     */
    public Coordinates toReferenceSystem(Coordinates aCoordinates) {
        double latrad = radians(aCoordinates.getX1()); 
        double lonrad = radians(aCoordinates.getX2());
        double coslat = cos(latrad); 
        double sinlat = sin(latrad);
        double coslon = cos(lonrad);
        double sinlon = sin(lonrad); 
        
        double r = A*B/Math.sqrt(B*B*coslat*coslat + A*A*sinlat*sinlat) + aCoordinates.getX3();
        
        return new Coordinates(r*coslat*coslon,
                               r*coslat*sinlon,
                               r*sinlat); 
        
    }
    
    private double radians(double aDegrees) { 
        return aDegrees/180.0*PI; 
    }
}
