/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.gps.track;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.wamblee.gps.geometry.Point;

/**
 * Represents a GPS track. 
 *
 * @author Erik Brakkee
 */
public class TrackSegmentImpl implements Serializable, TrackSegment {
    
    private String _description; 
    private List<TrackPoint> _points; 
    
    /**
     * Constructs an empty track. 
     *
     */
    public TrackSegmentImpl(String aDescription) { 
        _description = aDescription;  
        _points = new ArrayList<TrackPoint>();
    }
    
    
    @Override
    public String getDescription() {
        return _description;
    }
    
    /**
     * Adds a point to a track. 
     * @param aPoint Point. 
     */
    public void addPoint(TrackPoint aPoint) { 
        _points.add(aPoint);
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.gps.track.TrackSegment#size()
     */
    public int size()  { 
        return _points.size(); 
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.gps.track.TrackSegment#getMinCoordinate(int)
     */
    public double getMinCoordinate(int i) {
        if ( size() == 0 ) { 
            throw new IllegalArgumentException("empty track");
        }
        double min = getPoint(0).getCoordinates().getX(i);
        for (int j = 1; j < size(); j++) { 
            min = Math.min(min, getPoint(j).getCoordinates().getX(i));
        }
        return min;
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.gps.track.TrackSegment#getMaxCoordinate(int)
     */
    public double getMaxCoordinate(int i) {
        if ( size() == 0 ) { 
            throw new IllegalArgumentException("empty track");
        }
        double max = getPoint(0).getCoordinates().getX(i);
        for (int j = 1; j < size(); j++) { 
            max = Math.max(max, getPoint(j).getCoordinates().getX(i));
        }
        return max;
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.gps.track.TrackSegment#getPoint(int)
     */
    public Point getPoint(int aIndex) { 
        return _points.get(aIndex);
    }
}
