/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.gpx;

import java.awt.Color;
import java.awt.Image;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartFrame;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;
import org.wamblee.general.Pair;
import org.wamblee.gps.geometry.Plane;
import org.wamblee.gps.geometry.Point;
import org.wamblee.gps.geometry.ReferenceCoordinateSystem;
import org.wamblee.gps.track.TrackSegment;
import org.wamblee.utils.JpegUtils;

/**
 * Parses a GPX file and prints out a data file with each trackpoints distance from the start of the 
 * track and its elevation, separated 0by a space. 
 *
 * @author Erik Brakkee
 */
public class GpxPlotter {
    
    public static void main(String[] aArgs) throws Exception { 
        File file = new File(aArgs[0]);
        GpxParser parser = new GpxParser(); 
        TrackSegment track = parser.parse(file.getName(), new FileInputStream(file));
        
        TrackStatistics profile = new TrackStatistics(track);
        profile.writeHeightProfilePng(new FileOutputStream("x.png"), 600, 300);
    
        List<Pair<Double,Double>> elevationProfile = computeElevationProfile(track);
        printTrack(elevationProfile); 
        computeTotalClimb(elevationProfile);
        plotElevationProfile(elevationProfile);
        List<Pair<Double,Double>> trackXy = computeTrackXY(track);
        List<Pair<Double,Double>> trackLatLon = computeTrackLatLon(track);
        plotTrack(trackLatLon);
    }
    
    private static List<Pair<Double, Double>> computeElevationProfile(TrackSegment aTrack) {
        List<Pair<Double,Double>> results = new ArrayList<Pair<Double,Double>>();
        double distance = 0.0; 
        for (int i = 0; i < aTrack.size(); i++) { 
            Point point = aTrack.getPoint(i);
            results.add(new Pair<Double,Double>(distance, point.getCoordinates().getX3()));
            if ( i+1 < aTrack.size()) { 
                Point nextPoint = aTrack.getPoint(i+1); 
                distance += ReferenceCoordinateSystem.distance(point, nextPoint);
            }
        }
        return results; 
    }
    
    private static List<Pair<Double, Double>> computeTrackXY(TrackSegment aTrack) {
        Point reference = aTrack.getPoint(0);
        Plane plane = new Plane(reference, reference); // assume the earth is spherical.
        List<Pair<Double,Double>> results = new ArrayList<Pair<Double,Double>>();
        for (int i = 0; i < aTrack.size(); i++) { 
            Point point = aTrack.getPoint(i);
            Pair<Double,Double> projection = plane.normalizedProjection(point);
            results.add(projection);
            System.out.println(point);
        }
        return results; 
    }
    
    private static List<Pair<Double, Double>> computeTrackLatLon(TrackSegment aTrack) {
        List<Pair<Double,Double>> results = new ArrayList<Pair<Double,Double>>();
        for (int i = 0; i < aTrack.size(); i++) { 
            Point point = aTrack.getPoint(i);
            results.add(new Pair<Double,Double>(point.getCoordinates().getX1(), point.getCoordinates().getX2()));
        }
        return results; 
    }
    
    
    
    private static void printTrack(List<Pair<Double,Double>> aHeightProfile) { 
       for (Pair<Double,Double> point: aHeightProfile) { 
           System.out.println(point.getFirst() + " " + point.getSecond());
       }
    }
    
    private static void computeTotalClimb(List<Pair<Double,Double>> aHeightProfile) {
        double result = 0.0;
        
        double lastHeight = aHeightProfile.get(0).getSecond();
        for ( int i = 1; i < aHeightProfile.size(); i++) { 
            double height = aHeightProfile.get(i).getSecond();
            if ( height > lastHeight) { 
                result += (height-lastHeight); 
            }
            lastHeight = height; 
        }
        System.out.println("Total climb: " + result);
    }
    
    private static void plotElevationProfile(List<Pair<Double,Double>> aHeightProfile) throws IOException {
        XYSeriesCollection dataset = createDataset(aHeightProfile, "height");
        JFreeChart chart = ChartFactory.createXYLineChart(
                "Height Profile", 
                "Distance(m)",
                "Height(m)",
                dataset,
                PlotOrientation.VERTICAL,
                true,
                true,
                false);
        ChartUtilities.writeChartAsPNG(new FileOutputStream("height.png"), chart, 600, 300);
        ChartFrame frame = new ChartFrame("test", chart);
        frame.pack();
        frame.setVisible(true);
    }
    
    private static void plotTrack(List<Pair<Double,Double>> aPoints) throws IOException, InterruptedException {
        XYSeriesCollection dataset = createDataset(aPoints, "track");
        JFreeChart chart = createLineChart(dataset);
        
        Pair<Pair<Double,Double>,Pair<Double,Double>> bounds = getBounds(aPoints); 
        
        chart.getXYPlot().getDomainAxis().setLowerBound(bounds.getFirst().getFirst());
        chart.getXYPlot().getDomainAxis().setUpperBound(bounds.getFirst().getSecond());
        
        chart.getXYPlot().getRangeAxis().setLowerBound(bounds.getSecond().getFirst());
        chart.getXYPlot().getRangeAxis().setUpperBound(bounds.getSecond().getSecond());
        
        Image background = JpegUtils.loadJpegImage(new FileInputStream("/home/erik/vakantie.jpg"));
        chart.getPlot().setBackgroundImage(background);
       
        XYLineAndShapeRenderer renderer = (XYLineAndShapeRenderer)chart.getXYPlot().getRenderer();
        renderer.setShapesVisible(true);
        renderer.setShapesFilled(true);
        renderer.setPaint(Color.BLACK);
     
        ChartUtilities.writeChartAsPNG(new FileOutputStream("test.png"), chart, 1280, 800);
        ChartFrame frame = new ChartFrame("test", chart);
        frame.pack();
        frame.setVisible(true);
    }

    /**
     * @param dataset
     * @return
     */
    private static JFreeChart createLineChart(XYSeriesCollection dataset) {
        NumberAxis xAxis = new NumberAxis("S->N");
        xAxis.setAutoRangeIncludesZero(false);
        
        NumberAxis yAxis = new NumberAxis("W->E");
        yAxis.setAutoRangeIncludesZero(false);
       
        XYItemRenderer renderer = new XYLineAndShapeRenderer(true, false);
        XYPlot plot = new ZoomableBackgroundXYPlot(dataset, xAxis, yAxis, renderer);
        plot.setOrientation(PlotOrientation.HORIZONTAL);
      
        JFreeChart chart = new JFreeChart(
            "Track", JFreeChart.DEFAULT_TITLE_FONT, plot, true
        );

        return chart;
        /*
        JFreeChart chart = ChartFactory.createXYLineChart(
                "Track", 
                "S->N",
                "W->E",
                dataset,
                PlotOrientation.HORIZONTAL,
                true,
                true,
                false);
        return chart;
        */
    }

    /**
     * @param aHeightProfile
     * @return
     */
    private static XYSeriesCollection createDataset(List<Pair<Double, Double>> aHeightProfile, String aName) {
        XYSeries series = new XYSeries(aName, false);
        for (Pair<Double,Double> point: aHeightProfile) { 
            series.add(point.getFirst(), point.getSecond());
        }
        XYSeriesCollection dataset = new XYSeriesCollection(series);
        return dataset;
    }

    private static Pair<Pair<Double,Double>,Pair<Double,Double>> getBounds(List<Pair<Double,Double>> aList) { 
        Pair<Double,Double> first = aList.get(0); 
        double minx= first.getFirst();
        double maxx = minx;
        double miny = first.getSecond();
        double maxy = miny;
        
        for (int i = 0; i < aList.size(); i++) { 
            Pair<Double,Double> value = aList.get(i);
            minx = Math.min(minx, value.getFirst());
            maxx = Math.max(maxx, value.getFirst());
            miny = Math.min(miny, value.getSecond());
            maxy = Math.max(maxy, value.getSecond());
        }
        if ( maxx == minx ) { 
            maxx += 1.0; // to avoid problems. 
        }
        if ( maxy == miny ) { 
            maxy += 1.0; // to avoid problems.
        }
        final double paddingFactor = 0.3; // allow some space around min and max
        return new Pair<Pair<Double,Double>,Pair<Double,Double>>(
                new Pair<Double,Double>( minx - paddingFactor*(maxx-minx), 
                                         maxx + paddingFactor*(maxx-minx)),
                new Pair<Double,Double>( miny - paddingFactor*(maxy-miny), 
                                         maxy + paddingFactor*(maxy-miny))
                );
    }
}

