/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.gpx;

import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.geom.Rectangle2D;

import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.data.xy.XYDataset;

/**
 * Extension of XY plot that provides automatic zooming of the background 
 * image. 
 *
 * @author Erik Brakkee
 */
public class ZoomableBackgroundXYPlot extends XYPlot {
    
    /*
     * Initial domain axis. 
     */
    private double _x1 = 1.0;
    private double _x2 = -1.0; // _x2 < _x1 initially to make signify uninitialized values.
    
    /*
     * Initial range axis.
     */
    private double _y1 = 1.0;
    private double _y2 = -1.0; // _y2 < _y1 initially to make signify uninitialized values.
    
    public ZoomableBackgroundXYPlot(XYDataset aDataset, 
            ValueAxis aDomainAxis, ValueAxis aRangeAxis, XYItemRenderer aRenderer) { 
        super(aDataset, aDomainAxis, aRangeAxis, aRenderer);
    }
    
    /* (non-Javadoc)
     * @see org.jfree.chart.plot.Plot#drawBackgroundImage(java.awt.Graphics2D, java.awt.geom.Rectangle2D)
     */
    @Override
    protected void drawBackgroundImage(Graphics2D g2, Rectangle2D area) {
        //System.out.println("--------");
        //System.out.println("Area: " + area);
        //System.out.println("Graphics clip: " + g2.getClipBounds());
        //System.out.println("Domain axis: " + getDomainAxis().getLowerBound() + " " + 
        //        getDomainAxis().getUpperBound());
        
        // Get the current domain axis bounds
        double y1 = getDomainAxis().getLowerBound();
        double y2 = getDomainAxis().getUpperBound();
        double x1 = getRangeAxis().getLowerBound();
        double x2 = getRangeAxis().getUpperBound();
        
        if ( _x2 < _x1 ) { 
            // initial domain axis bounds
            _y1 = y1;
            _y2 = y2;
            _x1 = x1;
            _x2 = x2;
        }
        
        Image background = getBackgroundImage();
        int width = background.getWidth(null);
        int height = background.getHeight(null);
        
        // Determine the part of the image to be drawn on the screen based on the scaling
        // of the domain axes. 
        int imageX1 = (int)Math.round(1 + (x1 - _x1)*(width-1)/(_x2 - _x1));
        int imageX2 = (int)Math.round(1 + (x2 - _x1)*(width-1)/(_x2 - _x1));
        // Note: y-axis of image goes from bottom to top. 
        int imageY2 = (int)Math.round(height + (y1 - _y1)*(1-height)/(_y2 - _y1));
        int imageY1 = (int)Math.round(height + (y2 - _y1)*(1-height)/(_y2 - _y1));
        
        // Draw the correct part of the image on the screen. 
        g2.drawImage(background, (int)area.getMinX(), (int)area.getMinY(), (int)area.getMaxX(), (int)area.getMaxY(), 
                       imageX1, imageY1, imageX2, imageY2, null);
     
       // System.out.println("========");
    }

}
